package io.adbrix.sdk.component.migration_v1;

import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteException;
import android.database.sqlite.SQLiteOpenHelper;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Iterator;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.domain.model.EventModel;
import io.adbrix.sdk.domain.model.EventPackage;
import io.adbrix.sdk.utils.CommonUtils;

public class V1DatabaseOpenHelper extends SQLiteOpenHelper implements IV1DatabaseOpenHelper {
    protected static final String STRING_DATA_STORE_TABLE_NAME = "string_store";
    protected static final String EVENTS_TABLE_NAME = "events";
    private static final String DATABASE_NAME = "abrix.v2.sql";
    private static final String KEY_FIELD = "key";
    private static final String VALUE_FIELD = "value";
    private static final String EVENT_ID_FIELD = "id";
    private static final String EVENT_DATA_FIELD = "data";
    private SQLiteDatabase db;

    public V1DatabaseOpenHelper(Context context) {
        super(context, DATABASE_NAME, null, 1);

        try {
            this.db = getReadableDatabase();
        } catch (Exception e) {
            this.db = null;
            AbxLog.e(e, true);
        }
    }

    @Override
    public void onCreate(SQLiteDatabase db) {

    }

    @Override
    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {

    }

    @Override
    public void onDowngrade(SQLiteDatabase db, int oldVersion, int newVersion) {

    }

    @Override
    public String getValue(String key) {
        return (String) getValueFromTable(STRING_DATA_STORE_TABLE_NAME, key);
    }

    @Override
    public EventPackage getV1UnsentEventPackage() {
        Queue<EventModel> eventModels = new ConcurrentLinkedQueue<>();

        if (db == null)
            return new EventPackage(eventModels);
        Cursor cursor = null;
        try {
            cursor = queryDb(db, EVENTS_TABLE_NAME, new String[]{EVENT_ID_FIELD, EVENT_DATA_FIELD}, null, null, null, null, EVENT_ID_FIELD + " ASC", null);
            while (cursor.moveToNext()) {
                String event = cursor.getString(1);
                if (CommonUtils.isNullOrEmpty(event)) {
                    continue;
                }
                eventModels.offer(EventModel.fromJson(convertJSONObjectV1ToV2(new JSONObject(event))));
            }
        } catch (JSONException e) {
            AbxLog.e(e, true);
        } catch (SQLiteException e) {
            AbxLog.w(String.format("getValue from %s failed: %s", EVENTS_TABLE_NAME, e.getMessage()), true);
        } catch (StackOverflowError e) {
            AbxLog.w(String.format("getValue from %s failed: %s", EVENTS_TABLE_NAME, e.getMessage()), true);
        } catch (RuntimeException e) {
            AbxLog.w(String.format("getValue from %s failed: %s", EVENTS_TABLE_NAME, e.getMessage()), true);
        } finally {
            CommonUtils.close(cursor);
        }
        return new EventPackage(eventModels);
    }

    private JSONObject convertJSONObjectV1ToV2(JSONObject eventJson) {
        JSONObject result = new JSONObject();

        try {
            Iterator<String> keys = eventJson.keys();

            while (keys.hasNext()) {
                String key = keys.next();

                if (key.equals("param") || key.equals("user_properties") || key.equals("location")) {
                    if (eventJson.optJSONObject(key) != null)
                        result.put(key, convertJSONObjectV1ToV2(eventJson.optJSONObject(key)));
                    else
                        result.put(key, JSONObject.NULL);
                } else if (key.equals("abx:items")) {
                    JSONArray items = eventJson.getJSONArray(key);
                    JSONArray convertedItems = new JSONArray();

                    for (int i = 0; i < items.length(); i++) {
                        JSONObject item = items.getJSONObject(i);
                        if (item != null)
                            convertedItems.put(convertJSONObjectV1ToV2(item));
                    }

                    result.put(key, convertedItems);

                } else {
                    Object value = eventJson.get(key);

                    if (value instanceof String) {
                        value = convertSuffixToPrefix((String) value);
                    }

                    result.put(key, value);
                }
            }

        } catch (JSONException e) {
            AbxLog.e(e, true);
        }

        return result;
    }

    private String convertSuffixToPrefix(String value) {
        String substring = value.substring(0, value.length() - 2);

        if (value.endsWith("_s")) {
            value = "string:" + substring;
        } else if (value.endsWith("_l")) {
            value = "long:" + substring;
        } else if (value.endsWith("_d")) {
            value = "double:" + substring;
        } else if (value.endsWith("_b")) {
            value = "boolean:" + substring;
        }

        return value;
    }

    private Object getValueFromTable(String table, String key) {
        Object value = null;

        if (db == null)
            return value;
        Cursor cursor = null;
        try {
            cursor = queryDb(db, table, new String[]{KEY_FIELD, VALUE_FIELD}, KEY_FIELD + " = ?", new String[]{key}, null, null, null, null);
            if (cursor.moveToFirst()) {
                value = cursor.getString(1);
            }
        } catch (SQLiteException e) {
            AbxLog.w(String.format("getValue from %s failed: %s", table, e.getMessage()), true);
        } catch (StackOverflowError e) {
            AbxLog.w(String.format("getValue from %s failed: %s", table, e.getMessage()), true);
        } catch (RuntimeException e) {
            AbxLog.w(String.format("getValue from %s failed: %s", table, e.getMessage()), true);
        } finally {
            CommonUtils.close(cursor);
        }
        return value;
    }

    private Cursor queryDb(
            SQLiteDatabase db, String table, String[] columns, String selection,
            String[] selectionArgs, String groupBy, String having, String orderBy, String limit
    ) {
        return db.query(table, columns, selection, selectionArgs, groupBy, having, orderBy, limit);
    }

    @Override
    public boolean isTableDetected() {
        if (db == null)
            return false;

        String findStringStoreQuery = "select DISTINCT tbl_name from sqlite_master where tbl_name = '" + STRING_DATA_STORE_TABLE_NAME + "'";
        String findEventsQuery = "select DISTINCT tbl_name from sqlite_master where tbl_name = '" + EVENTS_TABLE_NAME + "'";

        boolean isDetected = false;
        Cursor findStringStoreCursor = null;
        Cursor findEventsCursor = null;
        try {
            findStringStoreCursor = db.rawQuery(findStringStoreQuery, null);
            findEventsCursor = db.rawQuery(findEventsQuery, null);
            if (findStringStoreCursor != null && findEventsCursor != null) {
                isDetected = findStringStoreCursor.getCount() > 0 || findEventsCursor.getCount() > 0;
            }
        }catch (Exception e){
            AbxLog.w(e, true);
        }finally {
            CommonUtils.close(findStringStoreCursor);
            CommonUtils.close(findEventsCursor);
        }
        return isDetected;
    }

    @Override
    public void deleteDB() {
        if (db == null)
            return;

        db.execSQL("DROP TABLE IF EXISTS " + STRING_DATA_STORE_TABLE_NAME);
        db.execSQL("DROP TABLE IF EXISTS " + EVENTS_TABLE_NAME);

        if (!isTableDetected()) {
            AbxLog.d("V1 Database is deleted!", true);
        } else {
            AbxLog.e("V1 Database is not deleted!", true);
        }
    }
}
