package io.adbrix.sdk.configuration;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;
import android.os.Bundle;

import com.google.firebase.messaging.RemoteMessage;
import com.igaworks.v2.core.AdBrixRm;
import com.igaworks.v2.core.push.notification.AbxPushReceiver;

import org.json.JSONObject;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import javax.annotation.Nullable;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.GooglePlayReferrerProperties;
import io.adbrix.sdk.component.IABXComponentsFactory;
import io.adbrix.sdk.component.ILogger;
import io.adbrix.sdk.component.UserPropertyManager;
import io.adbrix.sdk.data.SdkVersion;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CompatConstants;
import io.adbrix.sdk.domain.LogMessageFormat;
import io.adbrix.sdk.domain.Repository;
import io.adbrix.sdk.domain.exception.BeforeInitializeException;
import io.adbrix.sdk.domain.function.Completion;
import io.adbrix.sdk.domain.interactor.FlushAllNowUseCase;
import io.adbrix.sdk.domain.interactor.GdprForgetMeUseCase;
import io.adbrix.sdk.domain.model.ActionHistory;
import io.adbrix.sdk.domain.model.ActionHistoryIdType;
import io.adbrix.sdk.domain.model.DRState;
import io.adbrix.sdk.domain.model.DfnInAppMessage;
import io.adbrix.sdk.domain.model.DfnInAppMessageFetchMode;
import io.adbrix.sdk.domain.model.Empty;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.domain.model.Response;
import io.adbrix.sdk.domain.model.Result;
import io.adbrix.sdk.domain.model.ResultCallback;
import io.adbrix.sdk.domain.model.SubscriptionStatus;
import io.adbrix.sdk.domain.model.UserPropertyCommand;
import io.adbrix.sdk.domain.model.UserPropertyModel;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.CoreUtils;

public class DefaultABXContextController implements IABXContextController {
    private IABXComponentsFactory componentsFactory;
    private ILogger logger;
    private IABXContext abxContext;
    private DataRegistry dataRegistry;
    private ExecutorService executorService;
    private boolean isInitSubmitted = false;
    private boolean isOnResumeSubmitted = false;
    private MessageInvokeRunnable postponedOnResumeJob;
    private Queue<MessageInvokeRunnable> etcQueue;
    private Queue<Message> messageQueue;
    private Repository repository;
    private boolean isInitRestartProcessingNow = false;
    private boolean isDeleteProcessingNow = false;
//    private InAppMessageViewWrapper carriedInAppMessageViewWrapper;

    public DefaultABXContextController() {
    }

    public void changeABXContext(IABXContext newContext) {
        if (abxContext == null || componentsFactory == null) {
            AbxLog.d("controller.changeABXContext is called before controller.startcontroller", true);
            return;
        }

        if (abxContext.getClass() == newContext.getClass()) {
            AbxLog.d("Same ABXContext! : " + abxContext.getClass().getName() + ", Cannot change ABXContext!", true);
            return;
        }
        abxContext = newContext;
        componentsFactory.setABXContext(newContext);
    }

    @Override
    public IABXComponentsFactory getComponentsFactory() {
        return this.componentsFactory;
    }

    /*
    public static DefaultABXContextController getInstance() {
        return Singleton.INSTANCE;
    }
     */

    //TODO: 생성자에서 아래로직을 처리하도록 해야함. (두번호출되면 안되기 때문)
    @Override
    public void startController(IABXComponentsFactory componentsFactory) {
        this.componentsFactory = componentsFactory;

        this.logger = this.componentsFactory.createOrGetLogger();
        executorService = Executors.newSingleThreadExecutor();
        etcQueue = new ConcurrentLinkedQueue<>();
        messageQueue = new ConcurrentLinkedQueue<>();

        try {
            this.dataRegistry = this.componentsFactory.createOrGetDataRegistry();
            registerDataUnitEventListener();

            this.repository = componentsFactory.createOrGetRepository();
            this.abxContext = new InitializingABXContext(this.componentsFactory, repository);
            this.componentsFactory.setABXContext(abxContext);

        } catch (Exception e) { //이런일이 발생해서는 안된다.
            this.logger.error(e.toString());
            disableSDK("Cannot start controller.");
        }
    }

    @Override
    public void registerDataUnitEventListener() {
        this.dataRegistry.setListenOnChange(true);

        this.dataRegistry.registerDataUnitEventListener(DataRegistryKey.LONG_S3_CONFIG_ADBRIX_PAUSE, (oldValue, newValue) -> {
            try {
                if (newValue.getLong() == 1) {
                    new FlushAllNowUseCase(repository).execute();
                    disableSDK("Adbrix Pause");
                }
            } catch (DataUnit.DataUnitInvalidTypeCastingException e) {
                AbxLog.e(e, true);
            }
        });

        this.dataRegistry.registerDataUnitEventListener(DataRegistryKey.LONG_S3_CONFIG_ADBRIX_ALL_STOP, (oldValue, newValue) -> {
            try {
                if (newValue.getLong() == 1) {
                    disableSDK("Adbrix All Stop");
                }
            } catch (DataUnit.DataUnitInvalidTypeCastingException e) {
                AbxLog.e(e, true);
            }
        });

        List<DataRegistryKey> attributionIdKeys = Arrays.asList(
                DataRegistryKey.STRING_LAST_FIRSTOPEN_ID,
                DataRegistryKey.STRING_LAST_DEEPLINK_ID,
                DataRegistryKey.STRING_LAST_OPEN_ID,
                DataRegistryKey.STRING_DFN_ID
        );

        this.dataRegistry.registerDataUnitEventListener(attributionIdKeys, (oldValue, newValue) -> {
            dataRegistry.saveRegistry();
        });

        this.dataRegistry.registerDataUnitEventListener(DataRegistryKey.STRING_SDK_VERSION, (oldValue, newValue) -> {
            try {
                if (oldValue == null) {
                    return;
                }

                if (SdkVersion.compare(oldValue.getString(), "2.3.0.0") >= 0) {
                    return;
                }

                AbxLog.d("SDKVersion changed, oldValue : " + oldValue.getString() + ", newValue : " + newValue.getString(), true);

                dataRegistry.putDataRegistry(new DataUnit(
                        DataRegistryKey.INT_IN_APP_MESSAGE_MINUTES_TO_EXPIRY,
                        0,
                        5,
                        this.getClass().getName(),
                        true
                ));

                dataRegistry.putDataRegistry(new DataUnit(
                        DataRegistryKey.STRING_IN_APP_MESSAGE_CHECKSUM,
                        null,
                        5,
                        this.getClass().getName(),
                        true
                ));
            } catch (Exception e) {
                AbxLog.e(e, true);
            }
        });

        this.dataRegistry.registerDataUnitEventListener(DataRegistryKey.INT_IN_APP_MESSAGE_FETCH_MODE_VALUE, (oldValue, newValue) -> {
            try {
                if (oldValue == null)
                    return;

                DfnInAppMessageFetchMode oldMode = DfnInAppMessageFetchMode.fromInteger(oldValue.getInt());
                DfnInAppMessageFetchMode newMode = DfnInAppMessageFetchMode.fromInteger(newValue.getInt());

                if (oldMode != newMode) {
                    deleteAllInAppMessageDBContents();
                }
            } catch (DataUnit.DataUnitInvalidTypeCastingException e) {
                AbxLog.e(e, true);
            }
        });
    }

    @Override
    public void stopController() {
        if (executorService != null)
            executorService.shutdown();

        if (repository != null)
            repository.shutDownInAppMessageExecutor();
    }

    @Override
    public void initialize(Context context, String appkey, String secretkey) {
        submit(MessageType.INITIALIZE, context, appkey, secretkey);
    }

    @Override
    public void requestInstallReferrer(){
        submit(MessageType.REQUEST_INSTALL_REFERRER);
    }

    @Override
    public void saveUserProperty(UserPropertyCommand userPropertyCommand) {
        submit(MessageType.SAVE_USER_PROPERTY, userPropertyCommand);
    }

    @Override
    public void saveUserPropertyWithoutEvent(UserPropertyCommand userPropertyCommand) {
        submit(MessageType.SAVE_USER_PROPERTY_WITHOUT_EVENT, userPropertyCommand);
    }

    @Override
    public void clearUserProperty() {
        submit(MessageType.CLEAR_USER_PROPERTY);
    }

    @Override
    public void setCiProperty(String key, String value, AdBrixRm.SetCiProfileCallback callback) {
        submit(MessageType.SET_CI_PROPERTY, key, value, callback);
    }

    @Override
    public void getUserId(Completion<Result<String>> completion) {
        submit(MessageType.GET_USER_ID, completion);
    }

    @Override
    public void login(String userId, Completion<Result<Response>> completion) {
        submit(MessageType.LOGIN, userId, completion);
    }

    @Override
    public void logout(Completion<Result<Response>> completion) {
        submit(MessageType.LOGOUT, completion);
    }
    //옵저버 패턴 남용하던 부분 제거
    //옵저버 패턴은 구독과 활용부분이 중앙제어 되어야 하며, 각각은 리스너 형태로 전체 작동에 있어서 옵셔널해야 한다.
    //하지만, 현재 구조는 필수로직이기 때문에 옵저버를 제거하고 명시적 호출 구조로 변경함.
    /*
    @Override
    public void update(LogEventParameter logEventParameter) {
        submit(MessageType.LOG_EVENT, logEventParameter);
    }
    */

    @Override
    public void logEvent(LogEventParameter logEventParameter) {
        submit(MessageType.LOG_EVENT, logEventParameter);
    }

    @Override
    public void logSameEventWithPaging(String eventName, String group, List<JSONObject> eventParamList) {
        submit(MessageType.LOG_SAME_EVENT_WITH_PAGING, eventName, group, eventParamList);
    }

    @Override
    public void flushAllEvents(Completion<Result<Empty>> completion) {
        submit(MessageType.FLUSH_ALL_EVENTS, completion);
    }

    @Override
    public void onResume(Activity activity) {
        this.componentsFactory.createOrGetDFNSessionState().setInForeground(true);
        this.componentsFactory.getEventUploadIntervalManager().scheduleEventSendingJob();
        submit(MessageType.ON_RESUME, activity);
        if (repository != null) {
            if(activity != null){
                repository.updateCurrentActivity(activity);
            }
        }
    }

    @Override
    public void onPause(Activity activity) {
        if (repository != null) {
            repository.deleteCurrentActivity(activity);
        }
        this.componentsFactory.createOrGetDFNSessionState().setInForeground(false);
        this.componentsFactory.getEventUploadIntervalManager().setTimerToNull();
        submit(MessageType.ON_PAUSE, activity);
    }

    @Override
    public void onDestroy(Activity activity) {
        if (repository != null) {
            repository.updateCurrentActivity(activity);
        }
    }

    @Override
    public void onMessageReceived(Context context, RemoteMessage remoteMessage) {
        AbxLog.i("onMessageReceived() from: "+remoteMessage.getFrom()+" priority: "+remoteMessage.getPriority()+" original_priority: "+remoteMessage.getOriginalPriority(), true);
        Intent intent = new Intent(context, AbxPushReceiver.class);
        intent.setAction(CompatConstants.PUSH_REMOTE_MESSAGE_RECEIVE_FROM_FIREBASE_MESSAGING_SERVICE);
        Bundle bundle = CommonUtils.convertToBundle(remoteMessage.getData());
        intent.putExtras(bundle);
        context.sendBroadcast(intent);
    }

    @Override
    public void deeplink(Activity deeplinkActivity) {
        submit(MessageType.DEEPLINK, deeplinkActivity);
    }

    @Override
    public void deeplinkWithIntent(Intent deeplinkIntent) {
        submit(MessageType.DEEPLINK_WITH_INTENT, deeplinkIntent);
    }

    @Override
    public void gdprForgetMe() {
        submit(MessageType.GDPR_FORGET_ME);
    }

    @Override
    public void putDataRegistry(DataUnit dataUnit) {
        submit(MessageType.PUT_DATA_REGISTRY, dataUnit);
    }

    @Override
    public void runInBackGroundInOrder(Runnable runnable) {
        submit(MessageType.RUN_IN_BACKGROUND_IN_ORDER, runnable);
    }

    @Override
    public void runInBackGroundWithoutOrder(Runnable runnable) {
        submit(MessageType.RUN_IN_BACKGROUND_WITHOUT_ORDER, runnable);
    }

    @Override
    public void deleteUserDataAndStopSDK(String userId, Runnable onSuccess, Runnable onFail) {
        submit(MessageType.DELETE_USER_DATA_AND_STOP_SDK, userId, onSuccess, onFail);
    }

    @Override
    public void restartSDK(String userId, Runnable onSuccess, Runnable onFail) {
        submit(MessageType.RESTART_SDK, userId, onSuccess, onFail);
    }

    public DataRegistry getDataRegistry() throws NullPointerException {
        if (dataRegistry == null) {
            throw new NullPointerException();
        }

        return dataRegistry;
    }

    @Override
    public UserPropertyModel getCurrentUserPropertyModel() {
        try {
            if (componentsFactory == null) {
                AbxLog.d("controller.getCurrentUserPropertyModel is called before controller.startcontroller", true);
                return new UserPropertyModel(null, new HashMap<>());
            }

            UserPropertyManager userPropertyManager = componentsFactory.createOrGetUserPropertyManager();
            return userPropertyManager.getCurrentUserPropertyModel();
        } catch (Exception e) {
            AbxLog.e(e, true);
            return new UserPropertyModel(null, new HashMap<>());
        }
    }

    @Override
    public void triggerInAppMessage(String eventGroup, String eventName) {
        if (repository != null){
            repository.openInAppMessage(getInAppMessageEventName(eventGroup, eventName));
        }
    }

    @Override
    public void triggerInAppMessage(String eventGroup, String eventName, JSONObject eventParam) {
        if (repository != null){
            repository.openInAppMessage(getInAppMessageEventName(eventGroup, eventName), eventParam);
        }
    }

    @Override
    public void triggerInAppMessage(String eventGroup, String eventName, List<JSONObject> eventParamList) {
        if (repository != null){
            repository.openInAppMessage(getInAppMessageEventName(eventGroup, eventName), eventParamList);
        }
    }

    private String getInAppMessageEventName(String eventGroup, String eventName){
        StringBuilder stringBuilder = new StringBuilder(eventGroup.length()+eventName.length()+1);
        stringBuilder.append(eventGroup);
        stringBuilder.append(":");
        stringBuilder.append(eventName);
        return stringBuilder.toString();
    }

    @Override
    public void fetchActionHistoryFromServer(String token, ActionHistoryIdType idType, List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        submit(MessageType.FETCH_ACTION_HISTORY_FROM_SERVER, token, idType, actionType, completion);
    }

    @Override
    public void insertPushData(String pushDataString) {
        submit(MessageType.INSERT_PUSH_DATA, pushDataString);
    }

    @Override
    public void getActionHistory(int skip, int limit, List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        submit(MessageType.GET_ACTION_HISTORY, skip, limit, actionType, completion);
    }

    @Override
    public void getAllActionHistory(List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        submit(MessageType.GET_ALL_ACTION_HISTORY, actionType, completion);
    }

    @Override
    public void deleteActionHistory(
            @Nullable String token,
            String historyId,
            long timestamp,
            Completion<Result<Empty>> completion
    ) {
       submit(MessageType.DELETE_ACTION_HISTORY, token, historyId, timestamp, completion);
    }

    @Override
    public void deleteAllActionHistory(@Nullable String token, ActionHistoryIdType idType, Completion<Result<Empty>> completion) {
        submit(MessageType.DELETE_ALL_ACTION_HISTORY, token, idType, completion);
    }

    @Override
    public void clearSyncedActionHistoryInLocalDB(Completion<Result<Empty>> completion) {
        submit(MessageType.CLEAR_SYNCED_ACTION_HISTORY_IN_LOCAL_DB, completion);
    }

    @Override
    public void clearAllActionHistoryInLocalDB() {
        submit(MessageType.CLEAR_ALL_ACTION_HISTORY_IN_LOCAL_DB);
    }

    @Override
    public void fetchInAppMessage(boolean isDirectCall, Completion<Result<Empty>> completion) {
        submit(MessageType.FETCH_IN_APP_MESSAGE, isDirectCall, completion);
    }

    @Override
    public void getAllInAppMessage(Completion<Result<List<DfnInAppMessage>>> completion) {
        submit(MessageType.GET_ALL_IN_APP_MESSAGE, completion);
    }

    @Override
    public void openInAppMessage(String campaignId, Completion<Result<Empty>> completion) {
        submit(MessageType.OPEN_IN_APP_MESSAGE, campaignId, completion);
    }

    @Override
    public void deleteAllInAppMessageDBContents() {
        submit(MessageType.DELETE_ALL_IN_APP_MESSAGE_DB_CONTENTS);
    }

    private void submit(MessageType messageType, Object... args) {
        if (executorService == null || etcQueue == null) {
            AbxLog.d("controller.submit is called before controller.startcontroller", true);
            return;
        }

        if (executorService.isShutdown())
            executorService = Executors.newSingleThreadExecutor();

        if(isInitSubmitted && messageType == MessageType.REQUEST_INSTALL_REFERRER){
            submit(executorService, new MessageInvokeRunnable(messageType, args));
            return;
        }

        if (isInitSubmitted && messageType == MessageType.RUN_IN_BACKGROUND_WITHOUT_ORDER) {
            submit(executorService, new MessageInvokeRunnable(messageType, args));
            return;
        }

        if (isInitSubmitted && messageType == MessageType.RESTART_SDK) {
            submit(executorService, new MessageInvokeRunnable(messageType, args));
            return;
        }

        if (isInitSubmitted && messageType == MessageType.DELETE_USER_DATA_AND_STOP_SDK) {
            submit(executorService, new MessageInvokeRunnable(messageType, args));
            return;
        }

        if (isInitSubmitted && messageType == MessageType.INSERT_PUSH_DATA) {
            submit(executorService, new MessageInvokeRunnable(messageType, args));
            return;
        }

        if (isOnResumeSubmitted) {
            submit(executorService, new MessageInvokeRunnable(messageType, args));
            return;
        }

        if (messageType == MessageType.INITIALIZE) {
            submit(executorService, new MessageInvokeRunnable(messageType, args));
        } else if (messageType == MessageType.ON_RESUME){
            postponedOnResumeJob = new MessageInvokeRunnable(messageType, args);
            AbxLog.d(messageType.toString()+" is queued", true);
        } else {
            if(!messageType.equals(MessageType.ON_PAUSE)){
                Object[] convertedArgs = setTimeToNullOfLogEventParameter(messageType, args);
                etcQueue.offer(new MessageInvokeRunnable(messageType, convertedArgs));
                AbxLog.d(messageType.toString()+" is queued", true);
            }
        }
        flushQueuedEvent();
    }

    private void submit(ExecutorService executorService, MessageInvokeRunnable runnable){
        AbxLog.d("submit()->"+runnable.getMessage().messageType+" Thread: "+Thread.currentThread().getName(), true);
        executorService.submit(runnable);
        flushQueuedEvent();
    }

    private boolean isInstallReferrerCompleted(){
        boolean result = false;
        if(CommonUtils.isNull(componentsFactory)){
            return result;
        }
        GooglePlayReferrerProperties googlePlayReferrerProperties = new GooglePlayReferrerProperties(componentsFactory.getAndroidContext(), componentsFactory);
        result = googlePlayReferrerProperties.isRequestCompleted();
        return result;
    }

    public void onRestartCompleted() {
        isInitRestartProcessingNow = false;

        if (abxContext == null || componentsFactory == null || dataRegistry == null) {
            AbxLog.d("controller.onRestartCompleted is called before controller.startcontroller", true);
            return;
        }

        if (abxContext instanceof DisabledABXContext) {
            changeABXContext(new InitializingABXContext(componentsFactory, repository));

            Context context = null;
            try {
                context = componentsFactory.getAndroidContext();
            } catch (Exception e) {
                AbxLog.d(Arrays.toString(e.getStackTrace()), true);
            }
            String appKey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null);
            String secretKey = dataRegistry.safeGetString(DataRegistryKey.STRING_SECRETKEY, null);

            this.initialize(context, appKey, secretKey);

            //session 정보 초기화
            this.componentsFactory.createOrGetDFNSessionState().setIsSessionStarted(false);
            //이때 abxContext는 무조건 Active 이다. 이벤트 처리 전 바로 start_session 이벤트를 날리기 위함.
            this.onResume(null);
        }
    }

    @Override
    public void disableSDK(String disabledReason) {
        if (componentsFactory == null || repository == null) {
            AbxLog.d("controller.disableSDK is called before controller.startcontroller", true);
            return;
        }

        changeABXContext(new DisabledABXContext(disabledReason, componentsFactory, repository));
        componentsFactory.deactivate();
        if(isNeedRestoreUser()){
            restoreUser();
        }
    }

    @Override
    public void requestGetAttributionData(String logId, AdBrixRm.GetAttributionDataCallback callback) {
        submit(MessageType.GET_ATTRIBUTION_DATA, logId, callback);
    }

    @Override
    public void getSubscriptionStatus(AdBrixRm.GetSubscriptionStatusCallback callback) {
        submit(MessageType.GET_SUBSCRIPTION_STATUS, callback);
    }

    @Override
    public void setSubscriptionStatus(SubscriptionStatus subscriptionStatus, AdBrixRm.SetSubscriptionStatusCallback callback) {
        submit(MessageType.SET_SUBSCRIPTION_STATUS, subscriptionStatus, callback);
    }

    @Override
    public void flushQueuedEvent() {
//        AbxLog.d("flushQueuedEvent", true);
        if(!isInstallReferrerCompleted()){
            AbxLog.d("installReferrer is not completed", true);
            return;
        }
        if(!isInitSubmitted){
            AbxLog.d("initialize is not completed", true);
            return;
        }
        if(CommonUtils.notNull(postponedOnResumeJob)){
            isOnResumeSubmitted = true;
            MessageInvokeRunnable tempJob = postponedOnResumeJob;
            postponedOnResumeJob = null;
            submit(executorService, tempJob);
        }
        if(CommonUtils.isNull(etcQueue)){
            AbxLog.d("etcQueue is null", true);
            return;
        }
        if(!isOnResumeSubmitted){
            AbxLog.d("onResume is not completed", true);
            return;
        }
        if(!etcQueue.isEmpty()){
            MessageInvokeRunnable runnable = etcQueue.poll();
            if(runnable != null){
                MessageInvokeRunnable messageInvokeRunnable = updateTimeOfLogEventParameterIfTimeIsNull(runnable);
                submit(executorService, messageInvokeRunnable);
                return;
            }
        }
        if(!messageQueue.isEmpty()){
            Message message = messageQueue.poll();
            if(message != null){
                submit(executorService, new MessageInvokeRunnable(message.messageType, message.args));
                return;
            }
        }
    }

    /**
     * 2022.11.1 bobos
     * 2.3.1.6 버전의 DfnId API 호출 실패시 GDPR이 적용되는 이슈 대응
     * @see "https://github.com/IGAWorksDev/DFNGitOps/issues/110"
     * @return
     */
    public boolean isNeedRestoreUser(){
        boolean result = false;
        if(dataRegistry == null){
            AbxLog.d("dataRegistry is null", true);
            return result;
        }
        String dfnId = dataRegistry.safeGetString(DataRegistryKey.STRING_DFN_ID, null);
        if(!CommonUtils.isNullOrEmpty(dfnId)){
            return result;
        }
        boolean isGdprForgetMe = CoreUtils.isGdprForgetMe(dataRegistry, () -> {});
        if(!isGdprForgetMe){
            return result;
        }
        String sdkVersion = dataRegistry.safeGetString(DataRegistryKey.STRING_SDK_VERSION, null);
        if(!isGdPrIssueVersion(sdkVersion)){
            return result;
        }
        boolean isAdbrixPaused = CoreUtils.isAdbrixPause(dataRegistry);
        if(isAdbrixPaused){
            return result;
        }
        boolean isAdbrixStopped = CoreUtils.isAdbrixAllStop(dataRegistry);
        if(isAdbrixStopped){
            return result;
        }
        result = true;
        return result;
    }

    private boolean isGdPrIssueVersion(String sdkVersion){
        boolean result = false;
        if("2.3.1.6".equals(sdkVersion)){
            result = true;
            return result;
        }
        if("2.3.1.7".equals(sdkVersion)){
            result = true;
            return result;
        }
        if("2.3.1.8".equals(sdkVersion)){
            result = true;
            return result;
        }
        return result;
    }

    private void restoreUser(){
        AbxLog.d("restoreUser", true);
        if(dataRegistry == null){
            AbxLog.d("dataRegistry is null", true);
            return;
        }
        dataRegistry.putDataRegistry(
                new DataUnit(
                        DataRegistryKey.LONG_GDPR_FORGETME,
                        0L,
                        5,
                        this.getClass().getName(),
                        true
                )
        );
        dataRegistry.putDataRegistry(
                new DataUnit(
                        DataRegistryKey.LONG_GDPR_FORGETME_SERVER_SYNC,
                        0L,
                        5,
                        this.getClass().getName(),
                        true
                )
        );
        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_IS_SDK_STOPPED,
                false,
                5,
                this.getClass().getName(),
                true
        ));
        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_IS_SDK_STOPPED_SERVER_SYNC,
                false,
                5,
                this.getClass().getName(),
                true
        ));
        String dailyFirstOpenPrevDate = dataRegistry.safeGetString(DataRegistryKey.STRING_DAILY_FIRST_OPEN_PREV_DATE, null);
        if(CommonUtils.isNullOrEmpty(dailyFirstOpenPrevDate)){
            //2.3.1.6을 처음으로 사용한 사용자
            dataRegistry.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.STRING_LAST_FIRSTOPEN_ID,
                            null,
                            5,
                            this.getClass().getName(),
                            true
                    )
            );
        }
        try {
            onRestartCompleted();
        }catch (Exception e){
            AbxLog.e(e, true);
        }
    }

    private enum MessageType {
        CHANGE_ABX_CONTEXT,
        INITIALIZE,
        REQUEST_INSTALL_REFERRER,
        SAVE_USER_PROPERTY,
        SAVE_USER_PROPERTY_WITHOUT_EVENT,
        CLEAR_USER_PROPERTY,
        GET_USER_ID,
        LOGIN,
        LOGOUT,
        LOG_EVENT,
        LOG_SAME_EVENT_WITH_PAGING,
        FLUSH_ALL_EVENTS,
        ON_RESUME,
        ON_PAUSE,
        DEEPLINK,
        DEEPLINK_WITH_INTENT,
        GDPR_FORGET_ME,
        PUT_DATA_REGISTRY,
        RUN_IN_BACKGROUND_IN_ORDER,
        RUN_IN_BACKGROUND_WITHOUT_ORDER,
        DELETE_USER_DATA_AND_STOP_SDK,
        RESTART_SDK,
        FETCH_ACTION_HISTORY_FROM_SERVER,
        INSERT_PUSH_DATA,
        GET_ACTION_HISTORY,
        GET_ALL_ACTION_HISTORY,
        DELETE_ACTION_HISTORY,
        DELETE_ALL_ACTION_HISTORY,
        CLEAR_SYNCED_ACTION_HISTORY_IN_LOCAL_DB,
        CLEAR_ALL_ACTION_HISTORY_IN_LOCAL_DB,
        FETCH_IN_APP_MESSAGE,
        GET_ALL_IN_APP_MESSAGE,
        OPEN_IN_APP_MESSAGE,
        DELETE_ALL_IN_APP_MESSAGE_DB_CONTENTS,
        GET_ATTRIBUTION_DATA,
        GET_SUBSCRIPTION_STATUS,
        SET_SUBSCRIPTION_STATUS,
        SET_CI_PROPERTY,
    }

    private static class Singleton {
        private static final DefaultABXContextController INSTANCE = new DefaultABXContextController();
    }

    private static class Message {
        private MessageType messageType;
        private Object[] args;

        Message(MessageType messageType, Object... args) {
            this.messageType = messageType;
            this.args = args;
        }
    }

    private class MessageInvokeRunnable implements Runnable {
        private Message message;

        MessageInvokeRunnable(MessageType messageType, Object... args) {
            this.message = new Message(messageType, args);
        }

        public Message getMessage() {
            return message;
        }

        @Override
        public void run() {
            //메시지 처리
            try {
                if (message == null) {
                    idleTime();
                } else {
                    try {
                        processMessage(message);
                    } catch (BeforeInitializeException be) {
                        AbxLog.w(message.messageType + "\n" +
                                String.format(
                                        LogMessageFormat.DEFAULT_ABX_CONTEXT_BEFORE_INITIALIZE_EXCEPTION,
                                        be.toString()
                                ), true
                        );
                        if(message != null){
                            messageQueue.offer(message);
                            AbxLog.d(message.messageType.toString()+" is queued", true);
                        }
                    }
                }
            } catch (Exception e) {
                AbxLog.e(
                        String.format(
                                LogMessageFormat.DEFAULT_ABX_CONTEXT_CONTROLLER_QUEUE_EXCEPTION,
                                message.messageType.toString()
                        ), e, true
                );
            }
        }

        /**
         * message가 없을때 백그라운드 작업을 수행한다.
         */
        private void idleTime() {
            DefaultABXContextController.this.abxContext.runOnIdleTime();
        }

        /**
         * message를 처리한다.
         *
         * @param message
         */
        private void processMessage(Message message) {
            switch (message.messageType) {
                case CHANGE_ABX_CONTEXT:
                    IABXContext newContext = (IABXContext) message.args[0];
                    DefaultABXContextController.this.abxContext = newContext;
                    break;

                case INITIALIZE:
                    Context context = (Context) message.args[0];
                    String appkey = (String) message.args[1];
                    String secretkey = (String) message.args[2];
                    DefaultABXContextController.this.abxContext.initialize(context, appkey, secretkey, new ResultCallback<String>() {
                        @Override
                        public void callback(Status result, String data) {
                            isInitSubmitted = true;
                            AbxLog.d(data, true);
                            switch (result){
                                case SUCCESS:{
                                    changeABXContext(new ActiveABXContext(componentsFactory, repository));
                                    flushQueuedEvent();
                                    break;
                                }
                                case FAILURE:{
                                    disableSDK(data);
                                    break;
                                }
                            }
                            //onComplete
                            if (!isInstallReferrerCompleted()) {
                                requestInstallReferrer();
                            }
                        }
                    });
                    break;
                case REQUEST_INSTALL_REFERRER:
                    DefaultABXContextController.this.abxContext.getInstallReferrer();
                    break;
                case SAVE_USER_PROPERTY:
                    UserPropertyCommand userPropertyCommand1 = (UserPropertyCommand) message.args[0];
                    DefaultABXContextController.this.abxContext.saveUserProperty(userPropertyCommand1);
                    break;

                case SAVE_USER_PROPERTY_WITHOUT_EVENT:
                    UserPropertyCommand userPropertyCommand2 = (UserPropertyCommand) message.args[0];
                    DefaultABXContextController.this.abxContext.saveUserPropertyWithoutEvent(userPropertyCommand2);
                    break;

                case CLEAR_USER_PROPERTY:
                    DefaultABXContextController.this.abxContext.clearUserProperty();
                    break;
                case GET_USER_ID:{
                    Completion<Result<String>> completion = (Completion<Result<String>>) message.args[0];
                    DefaultABXContextController.this.abxContext.getUserId(completion);
                    break;
                }
                case LOGIN:{
                    String userId = (String) message.args[0];
                    Completion<Result<Response>> completion = null;
                    if(message.args.length == 2){
                        completion = (Completion<Result<Response>>) message.args[1];
                    }
                    DefaultABXContextController.this.abxContext.login(userId, completion);
                    break;
                }
                case LOGOUT:{
                    Completion<Result<Response>> completion = null;
                    if(message.args.length == 1){
                        completion = (Completion<Result<Response>>) message.args[0];
                    }
                    DefaultABXContextController.this.abxContext.logout(completion);
                    break;
                }
                case LOG_EVENT:
                    LogEventParameter logEventParameter = (LogEventParameter) message.args[0];

                    DefaultABXContextController.this.abxContext.logEvent(logEventParameter);
                    break;

                case LOG_SAME_EVENT_WITH_PAGING:
                    String eventName = (String) message.args[0];
                    String group = (String) message.args[1];

                    @SuppressWarnings("unchecked")
                    List<JSONObject> eventParamList = (List<JSONObject>) message.args[2];

                    DefaultABXContextController.this.abxContext.logSameEventWithPaging(eventName, group, eventParamList);
                    break;

                case FLUSH_ALL_EVENTS: {
                    @SuppressWarnings("unchecked")
                    Completion<Result<Empty>> completion = (Completion<Result<Empty>>) message.args[0];
                    DefaultABXContextController.this.abxContext.flushAllEvents(completion);
                    break;
                }

                case ON_RESUME:{
                    Activity activity = (Activity) message.args[0];
                    DefaultABXContextController.this.abxContext.onResume(activity);
                    isOnResumeSubmitted = true;
                    flushQueuedEvent();
                    break;
                }

                case ON_PAUSE: {
                    DefaultABXContextController.this.abxContext.onPause();
                    isOnResumeSubmitted = false;
                    break;
                }
                case DEEPLINK:
                    Activity deeplinkActivity = (Activity) message.args[0];
                    DefaultABXContextController.this.abxContext.deeplink(deeplinkActivity);
                    break;

                case DEEPLINK_WITH_INTENT:
                    Intent deeplinkIntent = (Intent) message.args[0];
                    DefaultABXContextController.this.abxContext.deeplinkWithIntent(deeplinkIntent);
                    break;

                case GDPR_FORGET_ME:
                    if (CoreUtils.isGdprForgetMe(dataRegistry, () -> { })) {
                        AbxLog.i("Set GDPR Sync set:: " + CoreUtils.isGdprForgetMeSync(dataRegistry), true);
                        if (CoreUtils.isGdprForgetMeSync(dataRegistry)) {
                            AbxLog.i("Set GDPR is already set the 'true'", true);
                            return;
                        }
                    }

                    new GdprForgetMeUseCase(repository).execute();

                    disableSDK("GDPR FORGET ME");
                    break;

                case PUT_DATA_REGISTRY:
                    DataUnit dataUnit = (DataUnit) message.args[0];
                    DefaultABXContextController.this.abxContext.putDataRegistry(dataUnit);
                    break;

                case RUN_IN_BACKGROUND_IN_ORDER:
                case RUN_IN_BACKGROUND_WITHOUT_ORDER:
                    Runnable runnable = (Runnable) message.args[0];
                    runnable.run();
                    break;

                case DELETE_USER_DATA_AND_STOP_SDK:

                    String deleteUserId = (String) message.args[0];
                    Runnable onDeleteSuccess = (Runnable) message.args[1];
                    Runnable onDeleteFail = (Runnable) message.args[2];

                    if (CoreUtils.isAdbrixPause(dataRegistry) ||
                            CoreUtils.isGdprForgetMe(dataRegistry, DefaultABXContextController.this::gdprForgetMe)) {
                        if (onDeleteFail != null)
                            onDeleteFail.run();

                        AbxLog.d("deleteUserDataAndStopSDK is failed due to (gdpr || pause || stop)", false);
                        return;
                    }

                    CoreUtils.printDRState(dataRegistry);
                    if (CoreUtils.getDRState(dataRegistry) == DRState.DELETE_SYNCED) {
                        AbxLog.d("Delete user data already synced!", true);
                        if (onDeleteSuccess != null)
                            onDeleteSuccess.run();
                        return;
                    }

                    if (isDeleteProcessingNow) {
                        AbxLog.d("Delete API is already processing!", true);
                        return;
                    }

                    isDeleteProcessingNow = true;
                    DefaultABXContextController.this.abxContext.deleteUserData(deleteUserId, new ResultCallback<String>() {
                        @Override
                        public void callback(Status result, String data) {
                            //onComplete
                            //Delete 종료
                            isDeleteProcessingNow = false;
                            //stop sdk.
                            disableSDK("SDK STOPPED");
                            switch (result){
                                case SUCCESS:{
                                    if (onDeleteSuccess != null)
                                        onDeleteSuccess.run();
                                    break;
                                }
                                case FAILURE:{
                                    if (onDeleteFail != null)
                                        onDeleteFail.run();
                                    break;
                                }
                            }
                        }
                    });
                    break;

                case RESTART_SDK:

                    String restartUserId = (String) message.args[0];
                    Runnable onRestartSuccess = (Runnable) message.args[1];
                    Runnable onRestartFail = (Runnable) message.args[2];

                    if (CoreUtils.isAdbrixPause(dataRegistry) ||
                            CoreUtils.isGdprForgetMe(dataRegistry, DefaultABXContextController.this::gdprForgetMe)) {
                        if (onRestartFail != null)
                            onRestartFail.run();

                        AbxLog.d("restartSDK is failed due to (gdpr || pause || stop)", false);
                        return;
                    }

                    CoreUtils.printDRState(dataRegistry);
                    if (CoreUtils.getDRState(dataRegistry) == DRState.INIT_RESTART_SYNCED) {
                        AbxLog.d("Restart already synced!", true);
                        if (onRestartSuccess != null)
                            onRestartSuccess.run();
                        return;
                    }

                    if (isInitRestartProcessingNow) {
                        AbxLog.d("InitRestart API is already processing!", true);
                        return;
                    }

                    isInitRestartProcessingNow = true;
                    if (abxContext instanceof DisabledABXContext){
                        isInitSubmitted = false;
                        isOnResumeSubmitted = false;
                    }

                    DefaultABXContextController.this.abxContext.restartSDK(restartUserId, new ResultCallback<String>() {
                        @Override
                        public void callback(Status result, String data) {
                            //onComplete
                            DefaultABXContextController.this.onRestartCompleted();
                            switch (result){
                                case SUCCESS:{
                                    if (onRestartSuccess != null)
                                        onRestartSuccess.run();
                                    break;
                                }
                                case FAILURE:{
                                    if (onRestartFail != null)
                                        onRestartFail.run();
                                    break;
                                }
                            }
                        }
                    });
                    break;

                case FETCH_ACTION_HISTORY_FROM_SERVER: {
                    String token = (String) message.args[0];
                    ActionHistoryIdType idType = (ActionHistoryIdType) message.args[1];
                    @SuppressWarnings("unchecked")
                    List<String> actionType = (List<String>) message.args[2];
                    @SuppressWarnings("unchecked")
                    Completion<Result<List<ActionHistory>>> completion = (Completion<Result<List<ActionHistory>>>) message.args[3];

                    DefaultABXContextController.this.abxContext.fetchActionHistoryFromServer(token, idType, actionType, completion);
                    break;
                }

                case INSERT_PUSH_DATA:
                    String pushDataString = (String) message.args[0];

                    DefaultABXContextController.this.abxContext.insertPushData(pushDataString);
                    break;

                case GET_ACTION_HISTORY: {
                    int skip = (int) message.args[0];
                    int limit = (int) message.args[1];
                    @SuppressWarnings("unchecked")
                    List<String> actionType = (List<String>) message.args[2];
                    @SuppressWarnings("unchecked")
                    Completion<Result<List<ActionHistory>>> completion = (Completion<Result<List<ActionHistory>>>) message.args[3];

                    DefaultABXContextController.this.abxContext.getActionHistory(skip, limit, actionType, completion);
                    break;
                }

                case GET_ALL_ACTION_HISTORY: {
                    @SuppressWarnings("unchecked")
                    List<String> actionType = (List<String>) message.args[0];
                    @SuppressWarnings("unchecked")
                    Completion<Result<List<ActionHistory>>> completion = (Completion<Result<List<ActionHistory>>>) message.args[1];

                    DefaultABXContextController.this.abxContext.getAllActionHistory(actionType, completion);
                    break;
                }

                case DELETE_ACTION_HISTORY: {
                    String token = (String) message.args[0];
                    String historyId = (String) message.args[1];
                    long timestamp = (long) message.args[2];
                    @SuppressWarnings("unchecked")
                    Completion<Result<Empty>> completion = (Completion<Result<Empty>>) message.args[3];

                    DefaultABXContextController.this.abxContext.deleteActionHistory(token, historyId, timestamp, completion);
                    break;
                }

                case DELETE_ALL_ACTION_HISTORY: {
                    String token = (String) message.args[0];
                    ActionHistoryIdType idType = (ActionHistoryIdType) message.args[1];
                    @SuppressWarnings("unchecked")
                    Completion<Result<Empty>> completion = (Completion<Result<Empty>>) message.args[2];

                    DefaultABXContextController.this.abxContext.deleteAllActionHistory(token, idType, completion);
                    break;
                }

                case CLEAR_SYNCED_ACTION_HISTORY_IN_LOCAL_DB: {
                    @SuppressWarnings("unchecked")
                    Completion<Result<Empty>> completion = (Completion<Result<Empty>>) message.args[0];

                    DefaultABXContextController.this.abxContext.clearSyncedActionHistoryInLocalDB(completion);
                    break;
                }

                case CLEAR_ALL_ACTION_HISTORY_IN_LOCAL_DB:

                    DefaultABXContextController.this.abxContext.clearAllActionHistoryInLocalDB();
                    break;

                case FETCH_IN_APP_MESSAGE: {
                    boolean isDirectCall = (boolean) message.args[0];
                    Completion<Result<Empty>> completion = (Completion<Result<Empty>>) message.args[1];

                    DefaultABXContextController.this.abxContext.fetchInAppMessage(isDirectCall, completion);
                    break;
                }


                case GET_ALL_IN_APP_MESSAGE: {
                    @SuppressWarnings("unchecked")
                    Completion<Result<List<DfnInAppMessage>>> completion =
                            (Completion<Result<List<DfnInAppMessage>>>) message.args[0];

                    DefaultABXContextController.this.abxContext.getAllInAppMessage(completion);
                    break;
                }

                case OPEN_IN_APP_MESSAGE: {
                    String campaignId = (String) message.args[0];
                    Completion<Result<Empty>> completion = (Completion<Result<Empty>>) message.args[1];

                    DefaultABXContextController.this.abxContext.openInAppMessage(campaignId, completion);
                    break;
                }

                case DELETE_ALL_IN_APP_MESSAGE_DB_CONTENTS:
                    DefaultABXContextController.this.abxContext.deleteAllInAppMessageDBContents();
                    break;
                case GET_ATTRIBUTION_DATA:{
                    String logId = (String) message.args[0];
                    AdBrixRm.GetAttributionDataCallback callback = (AdBrixRm.GetAttributionDataCallback) message.args[1];
                    DefaultABXContextController.this.abxContext.requestGetAttributionData(logId, callback, dataRegistry);
                    break;
                }
                case GET_SUBSCRIPTION_STATUS:{
                    AdBrixRm.GetSubscriptionStatusCallback callback = (AdBrixRm.GetSubscriptionStatusCallback) message.args[0];
                    DefaultABXContextController.this.abxContext.getSubscriptionStatus(callback, dataRegistry);
                    break;
                }
                case SET_SUBSCRIPTION_STATUS:{
                    SubscriptionStatus subscriptionStatus = (SubscriptionStatus) message.args[0];
                    AdBrixRm.SetSubscriptionStatusCallback callback = (AdBrixRm.SetSubscriptionStatusCallback) message.args[1];
                    DefaultABXContextController.this.abxContext.setSubscriptionStatus(subscriptionStatus, callback, dataRegistry);
                    break;
                }
                case SET_CI_PROPERTY:{
                    String key = (String) message.args[0];
                    String value = (String) message.args[1];
                    AdBrixRm.SetCiProfileCallback callback = (AdBrixRm.SetCiProfileCallback) message.args[2];
                    DefaultABXContextController.this.abxContext.setCiProperty(key, value, callback, dataRegistry);
                    break;
                }
                default:
                    throw new IllegalStateException("Unexpected value: " + message.messageType);
            }

            //메시지 처리 이후 백그라운드 작업을 수행한다.
            DefaultABXContextController.this.abxContext.runOnIdleTime();
        }
    }

    private Object[] setTimeToNullOfLogEventParameter(MessageType messageType, Object... args){
        Object[] result;
        if(messageType != MessageType.LOG_EVENT){
            result = args;
            return args;
        }
        LogEventParameter logEventParameter = (LogEventParameter) args[0];
        if(logEventParameter == null){
            result = args;
            return args;
        }
        logEventParameter.eventId = null;
        logEventParameter.eventDatetime = null;
        result = new Object[]{logEventParameter};
        return result;
    }

    private MessageInvokeRunnable updateTimeOfLogEventParameterIfTimeIsNull(MessageInvokeRunnable messageInvokeRunnable){
        MessageInvokeRunnable result;
        MessageType messageType = messageInvokeRunnable.message.messageType;
        if(messageType != MessageType.LOG_EVENT){
            result = messageInvokeRunnable;
            return result;
        }
        LogEventParameter logEventParameter = (LogEventParameter) messageInvokeRunnable.message.args[0];
        if(logEventParameter.eventId != null || logEventParameter.eventDatetime != null){
            result = messageInvokeRunnable;
            return result;
        }
        long currentTimeMillis = System.currentTimeMillis();
        logEventParameter.eventId = CommonUtils.randomUUIDWithCurrentTime(currentTimeMillis);;
        logEventParameter.eventDatetime = CommonUtils.getCurrentUTCInDBFormat(currentTimeMillis);;
        Object[] logEventArgs = new Object[]{logEventParameter};
        result = new MessageInvokeRunnable(messageType, logEventArgs);
        return result;
    }

}
