package io.adbrix.sdk.ui.inappmessage;

import static android.view.View.GONE;
import static android.view.View.VISIBLE;

import android.app.Activity;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.graphics.BlendMode;
import android.graphics.BlendModeColorFilter;
import android.graphics.Color;
import android.graphics.PorterDuff;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.GradientDrawable;
import android.os.Build;
import android.view.Gravity;
import android.view.MotionEvent;
import android.view.View;
import android.widget.Button;
import android.widget.FrameLayout;
import android.widget.ImageButton;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.ProgressBar;
import android.widget.RelativeLayout;
import android.widget.TextView;

import org.json.JSONException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import io.adbrix.sdk.R;
import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.IObserver;
import io.adbrix.sdk.configuration.IABXContextController;
import io.adbrix.sdk.domain.ABXConstants;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.DfnInAppMessage;
import io.adbrix.sdk.domain.model.IAMEnums;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.utils.CommonUtils;

public class ModalMessageViewFactory extends InAppMessageViewFactory {
    private List<IObserver<LogEventParameter>> observers = new ArrayList<>();
    private InAppMessageViewWrapper dialog;

    public ModalMessageViewFactory(InAppMessageDAO inAppMessageDAO, IABXContextController abxContextController) {
        super(inAppMessageDAO, abxContextController);
    }

    @Override
    public void createInAppMessage(Activity activity, DfnInAppMessage inAppMessage, boolean isCarried) {
        super.createInAppMessage(activity, inAppMessage, isCarried);
        activity.runOnUiThread(() -> {

            try {
                InAppMessageView view;
                dialog = new InAppMessageViewWrapper(
                        inAppMessage,
                        new InAppMessageViewListener() {
                            @Override
                            public void onClose(Activity attachedActivity, IAMEnums.CloseType action) {
                                AbxLog.i("inAppMessage onClose: "+action, true);
                                dialog.close(attachedActivity);
                            }
                        });

                boolean isPortrait = activity.getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT;
                Bitmap bitmapImage = null;
                if (IAMEnums.LayoutType.IMAGE_AND_TEXT.getValue().equals(inAppMessage.getLayout())) {
                    view = (InAppMessageView) activity.getLayoutInflater().inflate(R.layout.layout_modal_message_image_and_text, null);
                    bitmapImage = isPortrait ? inAppMessage.getPortraitImage() : inAppMessage.getLandScapeImage();
                    setImage(view, isPortrait, inAppMessage, dialog, bitmapImage);

                    setText(view, inAppMessage);
                } else if (IAMEnums.LayoutType.IMAGE.getValue().equals(inAppMessage.getLayout())) {
                    view = (InAppMessageView) activity.getLayoutInflater().inflate(R.layout.layout_modal_message_image, null);
                    bitmapImage = isPortrait ? inAppMessage.getPortraitImage() : inAppMessage.getLandScapeImage();
                    setImage(view, isPortrait, inAppMessage, dialog, bitmapImage);
                } else if (IAMEnums.LayoutType.TEXT.getValue().equals(inAppMessage.getLayout())) {
                    view = (InAppMessageView) activity.getLayoutInflater().inflate(R.layout.layout_modal_message_text, null);
                    setText(view, inAppMessage);
                } else {
                    AbxLog.d("createInAppMessage: unknown modal layout type", true);
                    return;
                }
                setOverlayColor(view, inAppMessage); // common
                FrameLayout buttonContainer = view.findViewById(R.id.com_adbrix_inappmessage_modal_button_layout);
                if(CommonUtils.notNull(buttonContainer)){
                    buttonContainer.setBackgroundColor(Color.parseColor(inAppMessage.getBgColor()));
                }

                //common
                switch (inAppMessage.getNumberOfButtons()) {
                    case 0:
                        setNoButton(view);
                        break;
                    case 1:
                        setSingleButton(view, dialog, inAppMessage, activity);
                        break;
                    case 2:
                        setDualButton(view, dialog, inAppMessage, activity);
                        break;
                    default:
                        setNoButton(view);
                        AbxLog.e("CreateInAppMessage : Invalid button number!", true);
                }

                //common
                setCancelButton(view, activity, inAppMessage);

                dialog.setContentView(view);
                dialog.setInAppMessageViewListener();

                //common
                open(activity, dialog, inAppMessage, isPortrait, isCarried);
            } catch (Exception e) {
                AbxLog.e("CreateInAppMessage : ",e, true);
                InAppMessageManager.getInstance().close(IAMEnums.CloseType.ERROR);
            }
        });
    }

    public void setOverlayColor(View view, DfnInAppMessage inAppMessage) {
        RelativeLayout modalContainer = view.findViewById(R.id.com_adbrix_inappmessage_modal_container);
        modalContainer.setBackgroundColor(Color.parseColor(inAppMessage.getOverlayColor()));
        modalContainer.setOnTouchListener(new View.OnTouchListener() {
            @Override
            public boolean onTouch(View view, MotionEvent motionEvent) {
                return true;
            }
        });
    }

    public void setImage(View view, boolean isPortrait, DfnInAppMessage inAppMessage, InAppMessageViewWrapper dialog, Bitmap bitmapImage) {
        if(CommonUtils.isNull(bitmapImage)){
            AbxLog.e("image is null", true);
            InAppMessageManager.getInstance().close(IAMEnums.CloseType.ERROR);
            return;
        }
        ImageView imageView = view.findViewById(R.id.com_adbrix_inappmessage_modal_imageview);
        ProgressBar progressBar = view.findViewById(R.id.com_adbirx_inappmessage_modal_progressbar);
        ImageView defaultImageView = view.findViewById(R.id.com_adbrix_inappmessage_modal_default_imageview);
        LinearLayout defaultImageViewLayout = view.findViewById(R.id.com_adbrix_inappmessage_modal_default_imageview_layout);
        FrameLayout progressBarLayout = view.findViewById(R.id.com_adbirx_inappmessage_modal_progressbar_layout);

        imageView.setVisibility(VISIBLE);
        progressBar.setVisibility(GONE);
        progressBarLayout.setVisibility(GONE);
        defaultImageViewLayout.setVisibility(GONE);

        imageView.setScaleType(ImageView.ScaleType.CENTER_CROP);
        imageView.setImageBitmap(bitmapImage);

        if (!CommonUtils.isNullOrEmpty(inAppMessage.getImageClickActionId())) {
            imageView.setOnClickListener(new IInAppMessageViewFactory.InAppMessageOnClickListener(
                    inAppMessage.getCampaignId(),
                    inAppMessage.getImageClickActionId(),
                    inAppMessage.getImageClickActionType(),
                    inAppMessage.getImageClickActionArg(),
                    dialog.getInAppMessageViewListener(),
                    inAppMessageDAO,
                    () -> {
                        Map<String, Object> eventParam = new HashMap<>();
                        try {
                            eventParam = CommonUtils.getMapFromJSONObject(inAppMessage.growthEventParam.getJson());
                        } catch (Exception e) {
                            AbxLog.w(e, true);
                        }

                        if (!eventParam.containsKey(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_ID)) {
                            eventParam.put(CoreConstants.IN_APP_MESSAGE_CAMPAIGN_ID_KEY, inAppMessage.getCampaignId());
                        }

                        eventParam.put(CoreConstants.IN_APP_MESSAGE_ACTION_ID_ABX_KEY, inAppMessage.getImageClickActionId());
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_EVENT_TYPE_ABX_KEY, IAMEnums.EventType.CLICK.getValue());

                        abxContextController.logEvent(new LogEventParameter(
                                CoreConstants.GROUP_ABX,
                                CoreConstants.EVENT_IN_APP_MESSAGE,
                                CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                                0,
                                0
                        ));
                        if (
                                IAMEnums.ActionType.CLOSE.getValue().equals(inAppMessage.getImageClickActionType())
                                        || IAMEnums.ActionType.WEBLINK_AND_CLOSE.getValue().equals(inAppMessage.getImageClickActionType())
                                        || IAMEnums.ActionType.DEEPLINK_AND_CLOSE.getValue().equals(inAppMessage.getImageClickActionType())
                                        || IAMEnums.ActionType.DONT_SHOW_ME_TODAY_AND_CLOSE.getValue().equals(inAppMessage.getImageClickActionType())
                        ) {

                        }
                    }
            ));
        }
    }

    public void setText(View view, DfnInAppMessage inAppMessage) {
        RelativeLayout textAndButtonLinearLayout = view.findViewById(R.id.com_adbirx_inappmessage_modal_text_and_button_layout);
        TextView titleText = view.findViewById(R.id.com_adbrix_inappmessage_modal_header_text);
        TextView bodyText = view.findViewById(R.id.com_adbrix_inappmessage_modal_message_text);

        titleText.setText(inAppMessage.getTitleText());
        titleText.setVisibility(VISIBLE);
        titleText.setTextColor(Color.parseColor(inAppMessage.getTitleTextColor()));

        bodyText.setText(inAppMessage.getBodyText());
        bodyText.setVisibility(VISIBLE);
        bodyText.setTextColor(Color.parseColor(inAppMessage.getBodyTextColor()));

        if (IAMEnums.TextAlign.CENTER.getValue().equals(inAppMessage.getTextAlign())) {
            titleText.setGravity(Gravity.CENTER);
            bodyText.setGravity(Gravity.CENTER);
        } else if (IAMEnums.TextAlign.RIGHT.getValue().equals(inAppMessage.getTextAlign())) {
            titleText.setGravity(Gravity.RIGHT);
            bodyText.setGravity(Gravity.RIGHT);
        } else if (IAMEnums.TextAlign.LEFT.getValue().equals(inAppMessage.getTextAlign())) {
            titleText.setGravity(Gravity.LEFT);
            bodyText.setGravity(Gravity.LEFT);
        }

        textAndButtonLinearLayout.setBackgroundColor(Color.parseColor(inAppMessage.getBgColor()));
    }

    private void setNoButton(View view) {
        LinearLayout singleButtonLayout = view.findViewById(R.id.com_adbrix_inappmessage_modal_button_layout_single);
        LinearLayout dualButtonLayout = view.findViewById(R.id.com_adbrix_inappmessage_modal_button_layout_dual);
        View buttonSpace = view.findViewById(R.id.com_adbrix_inappmessage_modal_view_button_space);
        if(CommonUtils.notNull(buttonSpace)) {
            buttonSpace.setVisibility(GONE);
        }
        singleButtonLayout.setVisibility(GONE);
        dualButtonLayout.setVisibility(GONE);
        FrameLayout buttonContainer = view.findViewById(R.id.com_adbrix_inappmessage_modal_button_layout);
        if(CommonUtils.notNull(buttonContainer)){
            buttonContainer.setVisibility(GONE);
        }
    }

    public void setSingleButton(View view, InAppMessageViewWrapper dialog, DfnInAppMessage inAppMessage, Activity activity) {
        LinearLayout singleButtonLayout = view.findViewById(R.id.com_adbrix_inappmessage_modal_button_layout_single);
        LinearLayout dualButtonLayout = view.findViewById(R.id.com_adbrix_inappmessage_modal_button_layout_dual);
        View buttonSpace = view.findViewById(R.id.com_adbrix_inappmessage_modal_view_button_space);
        if(CommonUtils.notNull(buttonSpace)) {
            buttonSpace.setVisibility(VISIBLE);
        }
        singleButtonLayout.setVisibility(VISIBLE);
        dualButtonLayout.setVisibility(GONE);

        Button singleButton = view.findViewById(R.id.com_adbrix_inappmessage_modal_button_single);
        setButton(dialog, singleButton, inAppMessage, 0, activity);
    }

    public void setDualButton(View view, InAppMessageViewWrapper dialog, DfnInAppMessage inAppMessage, Activity activity) {
        LinearLayout singleButtonLayout = view.findViewById(R.id.com_adbrix_inappmessage_modal_button_layout_single);
        LinearLayout dualButtonLayout = view.findViewById(R.id.com_adbrix_inappmessage_modal_button_layout_dual);
        View buttonSpace = view.findViewById(R.id.com_adbrix_inappmessage_modal_view_button_space);
        if(CommonUtils.notNull(buttonSpace)) {
            buttonSpace.setVisibility(VISIBLE);
        }
        singleButtonLayout.setVisibility(GONE);
        dualButtonLayout.setVisibility(VISIBLE);

        Button buttonFirst = view.findViewById(R.id.com_adbrix_inappmessage_modal_button_dual_first);
        Button buttonSecond = view.findViewById(R.id.com_adbrix_inappmessage_modal_button_dual_second);

        setButton(dialog, buttonFirst, inAppMessage, 0, activity);
        setButton(dialog, buttonSecond, inAppMessage, 1, activity);
    }

    public void setButton(InAppMessageViewWrapper dialog, Button button, DfnInAppMessage inAppMessage, int index, Activity activity) {
        //text
        button.setText(inAppMessage.getButtonText(index));
        button.setTextColor(Color.parseColor(inAppMessage.getButtonTextColor(index)));

        //background and border
        GradientDrawable buttonBackground = new GradientDrawable();
        buttonBackground.setShape(GradientDrawable.RECTANGLE);
        buttonBackground.setStroke(5, Color.parseColor(inAppMessage.getButtonBoarderColor(index)));
        buttonBackground.setColor(Color.parseColor(inAppMessage.getButtonBackgroundColor(index)));

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
            button.setBackground(buttonBackground);
        } else {
            button.setBackgroundDrawable(buttonBackground);
        }

        //action
        button.setOnClickListener(new IInAppMessageViewFactory.InAppMessageOnClickListener(
                inAppMessage.getCampaignId(),
                inAppMessage.getButtonActionID(index),
                inAppMessage.getButtonActionType(index),
                inAppMessage.getButtonActionArgument(index),
                dialog.getInAppMessageViewListener(),
                inAppMessageDAO,
                () -> {
                    Map<String, Object> eventParam = new HashMap<>();
                    try {
                        eventParam = CommonUtils.getMapFromJSONObject(inAppMessage.growthEventParam.getJson());
                    } catch (Exception e) {
                        AbxLog.w(e, true);
                    }

                    if (!eventParam.containsKey(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_ID)) {
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_CAMPAIGN_ID_KEY, inAppMessage.getCampaignId());
                    }
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_EVENT_TYPE_ABX_KEY, IAMEnums.EventType.CLICK.getValue());
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_ACTION_ID_ABX_KEY, inAppMessage.getButtonActionID(index));

                    abxContextController.logEvent(new LogEventParameter(
                            CoreConstants.GROUP_ABX,
                            CoreConstants.EVENT_IN_APP_MESSAGE,
                            CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                            0,
                            0
                    ));
                    if (
                            IAMEnums.ActionType.CLOSE.getValue().equals(inAppMessage.getButtonActionType(index))
                                    || IAMEnums.ActionType.WEBLINK_AND_CLOSE.getValue().equals(inAppMessage.getButtonActionType(index))
                                    || IAMEnums.ActionType.DEEPLINK_AND_CLOSE.getValue().equals(inAppMessage.getButtonActionType(index))
                                    || IAMEnums.ActionType.DONT_SHOW_ME_TODAY_AND_CLOSE.getValue().equals(inAppMessage.getButtonActionType(index))
                    ) {

                    }
                }
        ));
    }

    public void setCancelButton(View view, Activity activity, DfnInAppMessage inAppMessage) {
        ImageButton cancelButton = view.findViewById(R.id.com_adbrix_inappmessage_modal_close_button);

        cancelButton.setOnClickListener(v -> {
            InAppMessageManager.getInstance().close(IAMEnums.CloseType.ON_CLICK_FROM_CLOSE_BUTTON);
        });

        int closedButtonColor = Color.parseColor(inAppMessage.getDefaultCloseButtonColor());
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.Q) {
            cancelButton.setColorFilter(closedButtonColor, PorterDuff.Mode.SRC_IN);
        } else {
            cancelButton.setColorFilter(new BlendModeColorFilter(closedButtonColor, BlendMode.SRC_IN));
        }

        int closedButtonBgColor = Color.parseColor(inAppMessage.getDefaultCloseButtonBgColor());
        Drawable cancelButtonDrawable = view.getResources().getDrawable(R.drawable.ic_com_adbrix_inappmessage_cancel_button_border);

        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.Q) {
            cancelButtonDrawable.setColorFilter(closedButtonBgColor, PorterDuff.Mode.SRC_IN);
        } else {
            cancelButtonDrawable.setColorFilter(new BlendModeColorFilter(closedButtonBgColor, BlendMode.SRC_IN));
        }

        cancelButton.setBackgroundDrawable(cancelButtonDrawable);
    }

    public void open(Activity activity, InAppMessageViewWrapper dialog, DfnInAppMessage inAppMessage, boolean isPortrait, boolean isCarried) {
        dialog.open(activity, inAppMessage, isPortrait);

        if(isCarried){
            AbxLog.w("send impression is blocked because it is carried.", true);
            return;
        }

        Map<String, Object> eventParam = new HashMap<>();
        try {
            eventParam = CommonUtils.getMapFromJSONObject(inAppMessage.growthEventParam.getJson());
        } catch (Exception e) {
            AbxLog.w(e, true);
        }

        if (!eventParam.containsKey(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_ID)) {
            eventParam.put(CoreConstants.IN_APP_MESSAGE_CAMPAIGN_ID_KEY, inAppMessage.getCampaignId());
        }
        eventParam.put(CoreConstants.IN_APP_MESSAGE_EVENT_TYPE_ABX_KEY, IAMEnums.EventType.IMPRESSION.getValue());

        if (!inAppMessage.getLayout().equals(IAMEnums.LayoutType.TEXT.getValue())) {
            if (isPortrait && inAppMessage.getPortraitImage() != null) {
                eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_URL_ABX_KEY, inAppMessage.getPortraitImageURL());
                eventParam.put(CoreConstants.IN_APP_MESSAGE_IS_IMAGE_DOWNLOADED_ABX_KEY, inAppMessage.getsIsPortraitImageDownloaded());
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getPortraitImage().getAllocationByteCount());
                } else {
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getPortraitImage().getByteCount());
                }
            } else if (!isPortrait && inAppMessage.getLandScapeImage() != null) {
                eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_URL_ABX_KEY, inAppMessage.getLandscapeImageURL());
                eventParam.put(CoreConstants.IN_APP_MESSAGE_IS_IMAGE_DOWNLOADED_ABX_KEY, inAppMessage.getsIsLandscapeImageDownloaded());
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getLandScapeImage().getAllocationByteCount());
                } else {
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getLandScapeImage().getByteCount());
                }
            }
        }

        abxContextController.logEvent(new LogEventParameter(
                CoreConstants.GROUP_ABX,
                CoreConstants.EVENT_IN_APP_MESSAGE,
                CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                0,
                0
        ));
    }

    @Override
    public void dismissInAppMessage(Activity activity, IAMEnums.CloseType closeType) {
        super.dismissInAppMessage(activity, closeType);
        if (dialog != null){
            dialog.getInAppMessageViewListener().onClose(activity, closeType);
        }
    }
}
