package io.adbrix.sdk.ui.inappmessage;

import static android.view.View.GONE;

import android.app.Activity;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.graphics.BlendMode;
import android.graphics.BlendModeColorFilter;
import android.graphics.Color;
import android.graphics.Point;
import android.graphics.PorterDuff;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.GradientDrawable;
import android.os.Build;
import android.view.Display;
import android.view.Gravity;
import android.view.LayoutInflater;
import android.view.View;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.FrameLayout;
import android.widget.ImageButton;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.ProgressBar;
import android.widget.RelativeLayout;
import android.widget.ScrollView;
import android.widget.TextView;

import java.util.HashMap;
import java.util.Map;

import io.adbrix.sdk.R;
import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.TryOptional;
import io.adbrix.sdk.configuration.AbxFacade;
import io.adbrix.sdk.configuration.IABXContextController;
import io.adbrix.sdk.domain.ABXConstants;
import io.adbrix.sdk.domain.CompatConstants;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.DfnGrowthEventParam;
import io.adbrix.sdk.domain.model.DfnInAppMessage;
import io.adbrix.sdk.domain.model.IAMEnums;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.utils.CommonUtils;

public class FullScreenMessageViewFactory extends InAppMessageViewFactory {
    private InAppMessageViewWrapper dialog;

    public FullScreenMessageViewFactory(InAppMessageDAO inAppMessageDAO, IABXContextController abxContextController) {
        super(inAppMessageDAO, abxContextController);
    }

    @Override
    public void createInAppMessage(Activity activity, DfnInAppMessage inAppMessage, boolean isCarried) {
        super.createInAppMessage(activity, inAppMessage, isCarried);
        LayoutInflater inflater = activity.getLayoutInflater();
        InAppMessageView view;

        if (inAppMessage.getLayout().equals(IAMEnums.LayoutType.IMAGE.getValue()))
            view = (InAppMessageView) inflater.inflate(R.layout.layout_full_message_image, null);
        else if (inAppMessage.getLayout().equals(IAMEnums.LayoutType.SCROLLABLE_IMAGE.getValue()))
            view = (InAppMessageView) inflater.inflate(R.layout.layout_full_message_scrollable_image, null);
        else
            view = (InAppMessageView) inflater.inflate(R.layout.layout_full_message_image_with_text, null);

        activity.runOnUiThread(() -> {
            try {
                dialog = new InAppMessageViewWrapper(
                        inAppMessage,
                        new InAppMessageViewListener() {
                            @Override
                            public void onClose(Activity attachedActivity, IAMEnums.CloseType action) {
                                AbxLog.i("inAppMessage onClose: "+action, true);
                                dialog.close(attachedActivity);
                            }
                        });

                boolean isPortrait = activity.getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT;

                setImage(view, isPortrait, inAppMessage, dialog, activity);

                switch (inAppMessage.getNumberOfButtons()) {
                    case 0:
                        setNoButton(view);
                        break;
                    case 1:
                        setSingleButton(view, dialog, inAppMessage, activity);
                        break;
                    case 2:
                        setDualButton(view, dialog, inAppMessage, activity);
                        break;
                    default:
                        setNoButton(view);
                        AbxLog.e("createInAppMessage: Invalid button number!", true);
                }

                if (inAppMessage.getLayout().equals(IAMEnums.LayoutType.IMAGE_AND_TEXT.getValue()))
                    setText(view, inAppMessage);

                setCancelButton(view, activity, inAppMessage);
                dialog.setContentView(view);
                dialog.setInAppMessageViewListener();
                open(activity, dialog, inAppMessage, isPortrait, isCarried);
            } catch (Exception e) {
                AbxLog.e("createInAppMessage: ",e, true);
                InAppMessageManager.getInstance().close(IAMEnums.CloseType.ERROR);
            }
        });
    }

    private void open(Activity activity, InAppMessageViewWrapper dialog, DfnInAppMessage inAppMessage, boolean isPortrait, boolean isCarried) {
        dialog.open(activity, inAppMessage, isPortrait);

        if(isCarried){
            AbxLog.w("send impression is blocked because it is carried.", true);
            return;
        }
        Map<String, Object> eventParam = TryOptional.of(() -> inAppMessage.growthEventParam)
                .mapCatching(DfnGrowthEventParam::getJson)
                .map(CommonUtils::getMapFromJSONObject)
                .orElseGet(HashMap::new);

        if (!eventParam.containsKey(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_ID)) {
            eventParam.put(CoreConstants.IN_APP_MESSAGE_CAMPAIGN_ID_KEY, inAppMessage.getCampaignId());
        }
        eventParam.put(CoreConstants.IN_APP_MESSAGE_EVENT_TYPE_ABX_KEY, IAMEnums.EventType.IMPRESSION.getValue());

        if (isPortrait) {
            if (inAppMessage.getPortraitImage() != null) {
                eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_URL_ABX_KEY, inAppMessage.getPortraitImageURL());
                eventParam.put(CoreConstants.IN_APP_MESSAGE_IS_IMAGE_DOWNLOADED_ABX_KEY, inAppMessage.getsIsPortraitImageDownloaded());
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getPortraitImage().getAllocationByteCount());
                } else {
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getPortraitImage().getByteCount());
                }
            }
        } else {
            if (inAppMessage.getLandScapeImage() != null) {
                eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_URL_ABX_KEY, inAppMessage.getLandscapeImageURL());
                eventParam.put(CoreConstants.IN_APP_MESSAGE_IS_IMAGE_DOWNLOADED_ABX_KEY, inAppMessage.getsIsLandscapeImageDownloaded());
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getLandScapeImage().getAllocationByteCount());
                } else {
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getLandScapeImage().getByteCount());
                }
            }
        }

        this.abxContextController.logEvent(new LogEventParameter(
                CoreConstants.GROUP_ABX,
                CoreConstants.EVENT_IN_APP_MESSAGE,
                CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                0,
                0
        ));
    }

    private void setCancelButton(View view, Activity activity, DfnInAppMessage inAppMessage) {
        ImageButton cancelButton = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_close_button);
        cancelButton.setOnClickListener(button -> {
            InAppMessageManager.getInstance().close(IAMEnums.CloseType.ON_CLICK_FROM_CLOSE_BUTTON);
        });

        int closedButtonColor = Color.parseColor(inAppMessage.getDefaultCloseButtonColor());
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.Q) {
            cancelButton.setColorFilter(closedButtonColor, PorterDuff.Mode.SRC_IN);
        } else {
            cancelButton.setColorFilter(new BlendModeColorFilter(closedButtonColor, BlendMode.SRC_IN));
        }

        int closedButtonBgColor = Color.parseColor(inAppMessage.getDefaultCloseButtonBgColor());
        Drawable cancelButtonDrawable = view.getResources().getDrawable(R.drawable.ic_com_adbrix_inappmessage_cancel_button_border);

        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.Q) {
            cancelButtonDrawable.setColorFilter(closedButtonBgColor, PorterDuff.Mode.SRC_IN);
        } else {
            cancelButtonDrawable.setColorFilter(new BlendModeColorFilter(closedButtonBgColor, BlendMode.SRC_IN));
        }

        cancelButton.setBackgroundDrawable(cancelButtonDrawable);
    }

    private void setImage(View view, boolean isPortrait, DfnInAppMessage inAppMessage, InAppMessageViewWrapper dialog, Activity activity) {
        ImageView imageView = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_imageview);
        ProgressBar progressBar = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_progressbar);
        FrameLayout progressBarLayout = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_progressbar_layout);
        ImageView defaultImageView = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_default_imageview);
        LinearLayout defaultImageViewLayout = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_default_imageview_layout);

        Bitmap bitmapImage = isPortrait ? inAppMessage.getPortraitImage() : inAppMessage.getLandScapeImage();
        if(CommonUtils.isNull(bitmapImage)){
            AbxLog.e("image is null", true);
            InAppMessageManager.getInstance().close(IAMEnums.CloseType.ERROR);
            return;
        }
        progressBar.setVisibility(GONE);
        progressBarLayout.setVisibility(GONE);
        imageView.setVisibility(View.VISIBLE);
        defaultImageViewLayout.setVisibility(GONE);

        setImageScaleType(activity, view, inAppMessage, imageView, bitmapImage);

        if (inAppMessage.getImageClickActionId() != null) {
            imageView.setOnClickListener(new IInAppMessageViewFactory.InAppMessageOnClickListener(
                    inAppMessage.getCampaignId(),
                    inAppMessage.getImageClickActionId(),
                    inAppMessage.getImageClickActionType(),
                    inAppMessage.getImageClickActionArg(),
                    dialog.getInAppMessageViewListener(),
                    inAppMessageDAO,
                    () -> {
                        Map<String, Object> eventParam = TryOptional.of(() -> inAppMessage.growthEventParam)
                                .mapCatching(DfnGrowthEventParam::getJson)
                                .map(CommonUtils::getMapFromJSONObject)
                                .orElseGet(HashMap::new);

                        if (!eventParam.containsKey(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_ID)) {
                            eventParam.put(CoreConstants.IN_APP_MESSAGE_CAMPAIGN_ID_KEY, inAppMessage.getCampaignId());
                        }

                        eventParam.put(CoreConstants.IN_APP_MESSAGE_ACTION_ID_ABX_KEY, inAppMessage.getImageClickActionId());
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_EVENT_TYPE_ABX_KEY, IAMEnums.EventType.CLICK.getValue());

                        this.abxContextController.logEvent(new LogEventParameter(
                                CoreConstants.GROUP_ABX,
                                CoreConstants.EVENT_IN_APP_MESSAGE,
                                CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                                0,
                                0
                        ));

                        if (
                                IAMEnums.ActionType.CLOSE.getValue().equals(inAppMessage.getImageClickActionType())
                                        || IAMEnums.ActionType.WEBLINK_AND_CLOSE.getValue().equals(inAppMessage.getImageClickActionType())
                                        || IAMEnums.ActionType.DEEPLINK_AND_CLOSE.getValue().equals(inAppMessage.getImageClickActionType())
                                        || IAMEnums.ActionType.DONT_SHOW_ME_TODAY_AND_CLOSE.getValue().equals(inAppMessage.getImageClickActionType())
                        ) {

                        }
                    }
            ));
        }
    }

    private void setImageScaleType(Activity activity, View view, DfnInAppMessage inAppMessage, ImageView imageView, Bitmap bitmapImage) {
        if (!IAMEnums.LayoutType.SCROLLABLE_IMAGE.getValue().equals(inAppMessage.getLayout())) {
            imageView.setImageBitmap(bitmapImage);
            imageView.setScaleType(ImageView.ScaleType.CENTER_CROP);
            return;
        }

        WindowManager windowManager = (WindowManager) activity.getSystemService(Context.WINDOW_SERVICE);
        if (windowManager == null) {
            imageView.setImageBitmap(bitmapImage);
            imageView.setScaleType(ImageView.ScaleType.CENTER_CROP);
            return;
        }

        Display defaultDisplay = windowManager.getDefaultDisplay();
        Point point = new Point();
        defaultDisplay.getSize(point);

        double deviceRatio = (double) point.y / point.x;
        double imageRatio = (double) bitmapImage.getHeight() / bitmapImage.getWidth();

        if (imageRatio > deviceRatio) {
            imageView.setImageBitmap(bitmapImage);
            setImageViewGravity(imageView, Gravity.TOP);
            return;
        }

        if (IAMEnums.ScrollableImageOptionAlign.CENTER.getValue().equals(inAppMessage.getScrollableImageOptionAlign())) {
            setImageViewGravity(imageView, Gravity.CENTER);
        } else {
            setImageViewGravity(imageView, Gravity.TOP);
        }

        imageView.setImageBitmap(bitmapImage);
        ScrollView scrollView = (ScrollView) view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_scrollable_image_scrollview);
        scrollView.setBackgroundColor(Color.parseColor(inAppMessage.getBgColor()));
    }

    private void setImageViewGravity(ImageView imageView, int gravity) {
        FrameLayout.LayoutParams layoutParams = (FrameLayout.LayoutParams) imageView.getLayoutParams();
        layoutParams.gravity = gravity;
        imageView.setLayoutParams(layoutParams);
    }

    private void setImageScaleType2(Activity activity, View view, DfnInAppMessage inAppMessage, ImageView imageView, Bitmap bitmapImage) {
        WindowManager windowManager = (WindowManager) activity.getSystemService(Context.WINDOW_SERVICE);
        if (windowManager == null) {
            imageView.setImageBitmap(bitmapImage);
            imageView.setScaleType(ImageView.ScaleType.CENTER_CROP);
            return;
        }

        Display defaultDisplay = windowManager.getDefaultDisplay();
        Point point = new Point();
        defaultDisplay.getSize(point);

        double deviceRatio = (double) point.y / point.x;
        double imageRatio = (double) bitmapImage.getHeight() / bitmapImage.getWidth();

        if (imageRatio > deviceRatio) {
            imageView.setImageBitmap(bitmapImage);
            imageView.setScaleType(ImageView.ScaleType.FIT_START);
            return;
        }

        if (!inAppMessage.getLayout().equals(IAMEnums.LayoutType.SCROLLABLE_IMAGE.getValue())) {
            imageView.setImageBitmap(bitmapImage);
            imageView.setScaleType(ImageView.ScaleType.CENTER_CROP);
            return;
        }

//        ImageView notScrollableImageView = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_not_scrollable_imageview);
//        ScrollView scrollView = (ScrollView) view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_scrollable_image_scrollview);
//
//        notScrollableImageView.setImageBitmap(bitmapImage);
//        notScrollableImageView.setBackgroundColor(Color.parseColor(inAppMessage.getBgColor()));
//
//        if (IAMEnums.ScrollableImageOptionAlign.CENTER.getValue().equals(inAppMessage.getScrollableImageOptionAlign())) {
//            notScrollableImageView.setScaleType(ImageView.ScaleType.FIT_CENTER);
//        } else {
//            notScrollableImageView.setScaleType(ImageView.ScaleType.FIT_START);
//        }
//
//        scrollView.setVisibility(GONE);
//        notScrollableImageView.setVisibility(VISIBLE);
    }

    private void setText(View view, DfnInAppMessage inAppMessage) {
        RelativeLayout textLayout = view.findViewById(R.id.com_adbrix_inappmessage_text_layout);

        TextView titleText = view.findViewById(R.id.com_adbrix_inappmessage_title);
        titleText.setText(inAppMessage.getTitleText());
        titleText.setTextColor(Color.parseColor(inAppMessage.getTitleTextColor()));

        TextView bodyText = view.findViewById(R.id.com_adbrix_inappmessage_body);
        bodyText.setText(inAppMessage.getBodyText());
        bodyText.setTextColor(Color.parseColor(inAppMessage.getBodyTextColor()));

        if (IAMEnums.TextAlign.CENTER.getValue().equals(inAppMessage.getTextAlign())) {
            titleText.setGravity(Gravity.CENTER);
            bodyText.setGravity(Gravity.CENTER);
        } else if (IAMEnums.TextAlign.RIGHT.getValue().equals(inAppMessage.getTextAlign())) {
            titleText.setGravity(Gravity.RIGHT);
            bodyText.setGravity(Gravity.RIGHT);
        } else if (IAMEnums.TextAlign.LEFT.getValue().equals(inAppMessage.getTextAlign())) {
            titleText.setGravity(Gravity.LEFT);
            bodyText.setGravity(Gravity.LEFT);
        }

        textLayout.setBackgroundColor(Color.parseColor(inAppMessage.getBgColor()));
    }

    private void setNoButton(View view) {
        LinearLayout singleButtonLayout = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_button_layout_single);
        LinearLayout dualButtonLayout = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_button_layout_dual);

        singleButtonLayout.setVisibility(GONE);
        dualButtonLayout.setVisibility(GONE);
    }

    public void setSingleButton(View view, InAppMessageViewWrapper dialog, DfnInAppMessage inAppMessage, Activity activity) {
        LinearLayout singleButtonLayout = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_button_layout_single);
        LinearLayout doubleButtonsLayout = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_button_layout_dual);

        doubleButtonsLayout.setVisibility(View.GONE);
        singleButtonLayout.setVisibility(View.VISIBLE);

        Button singleButton = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_button_single);

        setButton(dialog, singleButton, inAppMessage, 0, activity);
    }

    public void setDualButton(View view, InAppMessageViewWrapper dialog, DfnInAppMessage inAppMessage, Activity activity) {
        LinearLayout singleButtonLayout = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_button_layout_single);
        LinearLayout doubleButtonsLayout = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_button_layout_dual);

        doubleButtonsLayout.setVisibility(View.VISIBLE);
        singleButtonLayout.setVisibility(View.GONE);

        Button buttonFirst = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_button_dual_first);
        Button buttonSecond = view.findViewById(R.id.com_adbrix_inappmessage_fullscreen_button_dual_second);

        setButton(dialog, buttonFirst, inAppMessage, 0, activity);
        setButton(dialog, buttonSecond, inAppMessage, 1, activity);
    }

    public void setButton(InAppMessageViewWrapper dialog, Button button, DfnInAppMessage inAppMessage, int index, Activity activity) {
        //text
        button.setText(inAppMessage.getButtonText(index));
        button.setTextColor(Color.parseColor(inAppMessage.getButtonTextColor(index)));

        //background and border
        GradientDrawable singleButtonBackground = new GradientDrawable();
        singleButtonBackground.setShape(GradientDrawable.RECTANGLE);
        singleButtonBackground.setStroke(5, Color.parseColor(inAppMessage.getButtonBoarderColor(index)));
        singleButtonBackground.setColor(Color.parseColor(inAppMessage.getButtonBackgroundColor(index)));

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
            button.setBackground(singleButtonBackground);
        } else {
            button.setBackgroundDrawable(singleButtonBackground);
        }

        button.setOnClickListener(new IInAppMessageViewFactory.InAppMessageOnClickListener(
                inAppMessage.getCampaignId(),
                inAppMessage.getButtonActionID(index),
                inAppMessage.getButtonActionType(index),
                inAppMessage.getButtonActionArgument(index),
                dialog.getInAppMessageViewListener(),
                inAppMessageDAO,
                () -> {
                    Map<String, Object> eventParam = TryOptional.of(() -> inAppMessage.growthEventParam)
                            .mapCatching(DfnGrowthEventParam::getJson)
                            .map(CommonUtils::getMapFromJSONObject)
                            .orElseGet(HashMap::new);

                    if (!eventParam.containsKey(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_ID)) {
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_CAMPAIGN_ID_KEY, inAppMessage.getCampaignId());
                    }
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_EVENT_TYPE_ABX_KEY, IAMEnums.EventType.CLICK.getValue());
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_ACTION_ID_ABX_KEY, inAppMessage.getButtonActionID(index));

                    this.abxContextController.logEvent(new LogEventParameter(
                            CoreConstants.GROUP_ABX,
                            CoreConstants.EVENT_IN_APP_MESSAGE,
                            CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                            0,
                            0
                    ));
                    if (
                            IAMEnums.ActionType.CLOSE.getValue().equals(inAppMessage.getButtonActionType(index))
                                    || IAMEnums.ActionType.WEBLINK_AND_CLOSE.getValue().equals(inAppMessage.getButtonActionType(index))
                                    || IAMEnums.ActionType.DEEPLINK_AND_CLOSE.getValue().equals(inAppMessage.getButtonActionType(index))
                                    || IAMEnums.ActionType.DONT_SHOW_ME_TODAY_AND_CLOSE.getValue().equals(inAppMessage.getButtonActionType(index))
                    ) {

                    }
                }
        ));
    }

    @Override
    public void dismissInAppMessage(Activity activity, IAMEnums.CloseType closeType) {
        super.dismissInAppMessage(activity, closeType);
        if (dialog != null){
            dialog.getInAppMessageViewListener().onClose(activity, closeType);
        }
    }

}
