package io.adbrix.sdk.data.actionhistory;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;

import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nullable;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.domain.model.ActionHistory;
import io.adbrix.sdk.utils.CommonUtils;


public class ActionHistorySqliteStore extends SQLiteOpenHelper {
    private static final int DATABASE_VERSION = 1;
    private static final String DATABASE_NAME = "ActionHistory.db";
    private ActionHistoryTable actionHistoryTable;

    public ActionHistorySqliteStore(@Nullable Context context) {
        super(context, DATABASE_NAME, null, DATABASE_VERSION);
        actionHistoryTable = new ActionHistoryTable();
    }

    @Override
    public void onDowngrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        try {
            db.execSQL(actionHistoryTable.getSqlDeleteEntries());
            db.execSQL(actionHistoryTable.getSqlCreateEntries());
            db.setVersion(DATABASE_VERSION);
        }catch (Exception e){
            AbxLog.e(e, true);
        }
    }

    @Override
    public void onCreate(SQLiteDatabase db) {
        db.execSQL(actionHistoryTable.getSqlCreateEntries());

        AbxLog.d("ActionHistory Database onCreate", true);
    }

    @Override
    public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        db.execSQL(actionHistoryTable.getSqlDeleteEntries());
        onCreate(db);

        AbxLog.d("ActionHistory Database onUpgrade", true);
    }

    public void deleteDB() {
        SQLiteDatabase db = this.getWritableDatabase();
        db.execSQL(actionHistoryTable.getSqlDeleteEntries());

        AbxLog.d("ActionHistory Database is deleted", true);
    }

    public void restartDB() {
        SQLiteDatabase db = this.getWritableDatabase();

        db.execSQL(actionHistoryTable.getSqlCreateEntries());

        AbxLog.d("ActionHistory Database is created", true);
    }

    public void insertActionHistory(
            String historyId,
            String actionType,
            String contents,
            long timestamp,
            String campaignId,
            int campaignRevisionNo,
            String stepId,
            String cycleTime,
            String group,
            boolean isServerSynced,
            boolean isRead
    ) throws Exception {
        SQLiteDatabase db = getWritableDatabase();

        ContentValues contentValues = new ContentValues();
        contentValues.put(ActionHistoryTable.COLUMN_NAME_HISTORY_ID, historyId);
        contentValues.put(ActionHistoryTable.COLUMN_NAME_HISTORY_ACTION_TYPE, actionType);
        contentValues.put(ActionHistoryTable.COLUMN_NAME_HISTORY_CONTENTS, contents);
        contentValues.put(ActionHistoryTable.COLUMN_NAME_HISTORY_TIMESTAMP, timestamp);
        contentValues.put(ActionHistoryTable.COLUMN_NAME_HISTORY_CAMPAIGN_ID, campaignId);
        contentValues.put(ActionHistoryTable.COLUMN_NAME_HISTORY_CAMPAIGN_REVISION_NO, campaignRevisionNo);
        contentValues.put(ActionHistoryTable.COLUMN_NAME_HISTORY_STEP_ID, stepId);
        contentValues.put(ActionHistoryTable.COLUMN_NAME_HISTORY_CYCLE_TIME, cycleTime);
        contentValues.put(ActionHistoryTable.COLUMN_NAME_HISTORY_GROUP, group);
        contentValues.put(ActionHistoryTable.COLUMN_NAME_HISTORY_IS_SERVER_SYNCED, isServerSynced);
        contentValues.put(ActionHistoryTable.COLUMN_NAME_HISTORY_IS_READ, isRead);

        long newRowId = db.insertWithOnConflict(ActionHistoryTable.TABLE_NAME, null, contentValues, SQLiteDatabase.CONFLICT_REPLACE);

        if (newRowId == -1) {
            AbxLog.d("Insert to ActionHistoryTable fail", true);
            throw new Exception("Insert to ActionHistoryTable fail");
        } else {
            AbxLog.d("Insert Success, newRowId in ActionHistoryTable : " + newRowId, true);
        }
    }

    public List<ActionHistory> getActionHistory(int skip, int limit, List<String> actionType) {
        SQLiteDatabase db = getReadableDatabase();
        StringBuilder stringBuilder = new StringBuilder();
        List<ActionHistory> actionHistories = new ArrayList<>();

        for (int i = 0; i < actionType.size(); i++) {
            String actionTypeString = actionType.get(i);

            stringBuilder.append("'")
                    .append(actionTypeString)
                    .append("'");

            if (i < actionType.size() - 1)
                stringBuilder.append(", ");
        }

        String getActionHistoryQuery = "SELECT * FROM "
                + ActionHistoryTable.TABLE_NAME
                + " WHERE " + ActionHistoryTable.COLUMN_NAME_HISTORY_ACTION_TYPE
                + " IN (" + stringBuilder.toString() + ")"
                + " LIMIT " + limit
                + " OFFSET " + skip;

        AbxLog.d("getActionHistoryQuery: " + getActionHistoryQuery, true);

        try (Cursor cursor = db.rawQuery(getActionHistoryQuery, null)) {
            while (cursor.moveToNext()) {
                actionHistories.add(
                    new ActionHistory(
                            cursor.getString(cursor.getColumnIndex(ActionHistoryTable.COLUMN_NAME_HISTORY_ID)),
                            cursor.getString(cursor.getColumnIndex(ActionHistoryTable.COLUMN_NAME_HISTORY_ACTION_TYPE)),
                            cursor.getString(cursor.getColumnIndex(ActionHistoryTable.COLUMN_NAME_HISTORY_CONTENTS)),
                            cursor.getLong(cursor.getColumnIndex(ActionHistoryTable.COLUMN_NAME_HISTORY_TIMESTAMP)),
                            CommonUtils.convertNullStringToNull(cursor.getString(cursor.getColumnIndex(ActionHistoryTable.COLUMN_NAME_HISTORY_GROUP))),
                            cursor.getInt(cursor.getColumnIndex(ActionHistoryTable.COLUMN_NAME_HISTORY_IS_SERVER_SYNCED)) == 1,
                            cursor.getInt(cursor.getColumnIndex(ActionHistoryTable.COLUMN_NAME_HISTORY_IS_READ)) == 1
                    )
                );
            }
        }

        return actionHistories;
    }

    public List<ActionHistory> getAllActionHistory(List<String> actionType) {
        SQLiteDatabase db = getReadableDatabase();
        StringBuilder stringBuilder = new StringBuilder();
        List<ActionHistory> actionHistories = new ArrayList<>();

        for (int i = 0; i < actionType.size(); i++) {
            String actionTypeString = actionType.get(i);

            stringBuilder.append("'")
                    .append(actionTypeString)
                    .append("'");

            if (i < actionType.size() - 1)
                stringBuilder.append(", ");
        }

        String getActionHistoryQuery = "SELECT * FROM "
                + ActionHistoryTable.TABLE_NAME
                + " WHERE " + ActionHistoryTable.COLUMN_NAME_HISTORY_ACTION_TYPE
                + " IN (" + stringBuilder.toString() + ")";

        AbxLog.d("getActionHistoryQuery: " + getActionHistoryQuery, true);

        try (Cursor cursor = db.rawQuery(getActionHistoryQuery, null)) {
            while (cursor.moveToNext()) {
                actionHistories.add(
                        new ActionHistory(
                                cursor.getString(cursor.getColumnIndex(ActionHistoryTable.COLUMN_NAME_HISTORY_ID)),
                                cursor.getString(cursor.getColumnIndex(ActionHistoryTable.COLUMN_NAME_HISTORY_ACTION_TYPE)),
                                cursor.getString(cursor.getColumnIndex(ActionHistoryTable.COLUMN_NAME_HISTORY_CONTENTS)),
                                cursor.getLong(cursor.getColumnIndex(ActionHistoryTable.COLUMN_NAME_HISTORY_TIMESTAMP)),
                                CommonUtils.convertNullStringToNull(cursor.getString(cursor.getColumnIndex(ActionHistoryTable.COLUMN_NAME_HISTORY_GROUP))),
                                cursor.getInt(cursor.getColumnIndex(ActionHistoryTable.COLUMN_NAME_HISTORY_IS_SERVER_SYNCED)) == 1,
                                cursor.getInt(cursor.getColumnIndex(ActionHistoryTable.COLUMN_NAME_HISTORY_IS_READ)) == 1
                        )
                );
            }
        }

        return actionHistories;
    }

    public void deleteActionHistory(String historyId, long timestamp) {
        SQLiteDatabase db = getWritableDatabase();

        int deletedRowCount = db.delete(
                ActionHistoryTable.TABLE_NAME,
                ActionHistoryTable.COLUMN_NAME_HISTORY_ID + " = '" + historyId + "'"
                + " AND " + ActionHistoryTable.COLUMN_NAME_HISTORY_TIMESTAMP + " = " + timestamp
                + " AND " + ActionHistoryTable.COLUMN_NAME_HISTORY_IS_SERVER_SYNCED + " = 1",
                null
        );

        AbxLog.d("deleteActionHistory: Deleted row count = " + deletedRowCount, true);
    }

    public void clearSyncedActionHistoryInLocalDB() {
        SQLiteDatabase db = getWritableDatabase();

        int deletedRowCount = db.delete(
                ActionHistoryTable.TABLE_NAME,
                ActionHistoryTable.COLUMN_NAME_HISTORY_IS_SERVER_SYNCED + " = 1",
                null
        );

        AbxLog.d("clearSyncedActionHistoryInLocalDB: Deleted row count = " + deletedRowCount, true);
    }

    public void clearAllActionHistoryInLocalDB() {
        SQLiteDatabase db = getWritableDatabase();

        int deletedRowCount = db.delete(
                ActionHistoryTable.TABLE_NAME,
                "1",
                null
        );

        AbxLog.d("clearAllActionHistoryInLocalDB: Deleted row count = " + deletedRowCount, true);
    }

    public long getLastServerTimeStamp() {
        SQLiteDatabase db = getReadableDatabase();

        String getLastServerTimeStampQuery = "SELECT MAX("
                + ActionHistoryTable.COLUMN_NAME_HISTORY_TIMESTAMP
                + ") FROM "
                + ActionHistoryTable.TABLE_NAME
                + " WHERE " + ActionHistoryTable.COLUMN_NAME_HISTORY_IS_SERVER_SYNCED + " = 1";

        AbxLog.d("getLastServerTimeStampQuery: " + getLastServerTimeStampQuery, true);

        try (Cursor cursor = db.rawQuery(getLastServerTimeStampQuery, null)) {
            if (cursor != null) {
                cursor.moveToFirst();
                return cursor.getLong(0);
            }

            return 0L;
        }
    }

    public void deleteOutdatedPushData(long lastTimestamp) {
        SQLiteDatabase db = getWritableDatabase();

        int deletedRowCount = db.delete(
                ActionHistoryTable.TABLE_NAME,
                ActionHistoryTable.COLUMN_NAME_HISTORY_IS_SERVER_SYNCED + " = 0"
                        + " AND " + ActionHistoryTable.COLUMN_NAME_HISTORY_TIMESTAMP + " < " + lastTimestamp,
                null
        );

        AbxLog.d("deleteOutdatedPushData: Deleted row count = " + deletedRowCount, true);
    }

    private static class ActionHistoryTable {
        public static final String TABLE_NAME = "ActionHistory";
        public static final String COLUMN_NAME_HISTORY_ID = "HistoryId";
        public static final String COLUMN_NAME_HISTORY_ACTION_TYPE = "ActionType";
        public static final String COLUMN_NAME_HISTORY_TIMESTAMP = "TimeStamp";
        public static final String COLUMN_NAME_HISTORY_STEP_ID = "StepId";
        public static final String COLUMN_NAME_HISTORY_CYCLE_TIME = "CycleTime";
        public static final String COLUMN_NAME_HISTORY_CONTENTS = "Contents";
        public static final String COLUMN_NAME_HISTORY_CAMPAIGN_REVISION_NO = "CampaignRevisionNO";
        public static final String COLUMN_NAME_HISTORY_CAMPAIGN_ID = "CampaignId";
        public static final String COLUMN_NAME_HISTORY_GROUP = "GroupCode";
        public static final String COLUMN_NAME_HISTORY_IS_SERVER_SYNCED = "IsServerSynced";
        public static final String COLUMN_NAME_HISTORY_IS_READ = "IsRead";

        public String getSqlCreateEntries() {
            return "CREATE TABLE IF NOT EXISTS " + TABLE_NAME + " (" +
                    COLUMN_NAME_HISTORY_ID + " TEXT UNIQUE, " +
                    COLUMN_NAME_HISTORY_ACTION_TYPE + " TEXT ," +
                    COLUMN_NAME_HISTORY_TIMESTAMP + " INT, " +
                    COLUMN_NAME_HISTORY_STEP_ID + " TEXT, " +
                    COLUMN_NAME_HISTORY_CYCLE_TIME + " TEXT, " +
                    COLUMN_NAME_HISTORY_CONTENTS + " TEXT, " +
                    COLUMN_NAME_HISTORY_CAMPAIGN_REVISION_NO + " INT, " +
                    COLUMN_NAME_HISTORY_CAMPAIGN_ID + " TEXT, " +
                    COLUMN_NAME_HISTORY_GROUP + " TEXT, " +
                    COLUMN_NAME_HISTORY_IS_SERVER_SYNCED + " INT DEFAULT 0, " +
                    COLUMN_NAME_HISTORY_IS_READ + " INT DEFAULT 0)";
        }

        public String getSqlDeleteEntries() {
            return "DROP TABLE IF EXISTS " + TABLE_NAME;
        }
    }
}
