package io.adbrix.sdk.component;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Queue;
import java.util.Timer;
import java.util.TimerTask;

import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.EventPackage;
import io.adbrix.sdk.utils.CommonUtils;

public class EventUploadIntervalManager implements IObservable<Void>, ITimerManager {
    private int countInterval = 30;
    private int timeInterval = 60_000;
    private Timer eventBusTimer;
    private WeakReference<Timer> eventBusTimerReference;
    private TimerTask eventBusTimerTask;
    private WeakReference<TimerTask> eventBusTimerTaskReference;
    private ArrayList<IObserver<Void>> observerArrayList = new ArrayList<>();

    public EventUploadIntervalManager() {

    }

    public void setCountInterval(int countInterval) {
        this.countInterval = countInterval;
    }

    public void setTimeInterval(int timeInterval) {
        this.timeInterval = timeInterval;
        setTimerToNull();
        scheduleEventSendingJob();
    }

    public void manageEventUploadInterval(Runnable eventSendingTask, EventPackage packageQueue) {
        if(packageQueue == null){
            return;
        }
        if(CommonUtils.isNullOrEmpty(packageQueue.eventModels)){
            return;
        }
        AbxLog.d("EventQueue size : " + packageQueue.eventModels.size(), true);
        scheduleEventSendingJob();
        countEventSendingJob(eventSendingTask, packageQueue);
    }

    private void scheduleEventSendingJob() {
        if(eventBusTimerReference!=null && eventBusTimerReference.get()!=null){
//            AbxLog.i("eventBusTimer is working!", true);
            return;
        }
        eventBusTimerReference = new WeakReference<>(new Timer());
        eventBusTimer = eventBusTimerReference.get();
        synchronized (eventBusTimer){
            eventBusTimerTaskReference = new WeakReference<>(new TimerTask() {
                @Override
                public void run() {
                    AbxLog.d("Trying to upload events every " + timeInterval / 1000 + " seconds.", true);
                    notifyObserver(null);
                }
            });
            eventBusTimerTask = eventBusTimerTaskReference.get();
            if(eventBusTimer!=null){
                try {
                    eventBusTimer.schedule(eventBusTimerTask, this.timeInterval, this.timeInterval);
                }catch (Exception e){
                    AbxLog.w(e, true);
                    setTimerToNull();
                }
            }
        }
    }

    private void countEventSendingJob(Runnable eventSendingTask, EventPackage packageQueue) {
        if (packageQueue == null)
            return;
        if(CommonUtils.isNullOrEmpty(packageQueue.eventModels)){
            AbxLog.d("packageQueue.eventModels is null or empty", true);
            return;
        }
        if(packageQueue.eventModels.size() >= countInterval){
            AbxLog.d("Trying to upload " + countInterval + " events.", true);
            eventSendingTask.run();
        }
    }

    @Override
    public void add(IObserver<Void> observer) {
        observerArrayList.add(observer);
    }

    @Override
    public void delete(IObserver<Void> observer) {
        observerArrayList.remove(observer);
    }

    public void deleteAllObservers() {
        observerArrayList.clear();
    }

    @Override
    public void notifyObserver(Void v) {
        for (IObserver<Void> observer : observerArrayList) {
            observer.update(v);
        }
    }

    @Override
    public void setTimerToNull() {
        AbxLog.w("eventBusTimer is cancelled", true);
        try {
            if(eventBusTimerTask == null){
                if(eventBusTimerTaskReference!=null){
                    eventBusTimerTask = eventBusTimerTaskReference.get();
                }
            }
            if(eventBusTimerTask!=null){
                eventBusTimerTask.cancel();
                eventBusTimerTask = null;
                eventBusTimerTaskReference = null;
            }
            if(eventBusTimer==null){
                if(eventBusTimerReference!=null){
                    eventBusTimer = eventBusTimerReference.get();
                }
            }
            if(eventBusTimer!=null){
                synchronized (eventBusTimer){
                    if(eventBusTimer!=null){
                        eventBusTimer.cancel();
                        eventBusTimer.purge();
                        eventBusTimer = null;
                        eventBusTimerReference = null;
                    }
                }
            }
        }catch (Exception e){
            AbxLog.w(e, true);
        }
    }

    public void deactivate(){
        setTimerToNull();
        deleteAllObservers();
    }

    public int getCountInterval() {
        return countInterval;
    }

    public int getTimeInterval() {
        return timeInterval;
    }
}
