package io.adbrix.sdk.component;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashSet;
import java.util.Locale;

import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.EventSamplingModel;
import io.adbrix.sdk.domain.model.EventSamplingType;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.utils.CoreUtils;

public class EventSamplingFilter {
    private final HashSet<String> eventsThatShouldNotBeSampled;
    private final DataRegistry dataRegistry;
    private EventSamplingModel eventSamplingModel;

    public EventSamplingFilter(DataRegistry dataRegistry) {
        this.dataRegistry = dataRegistry;

        JSONObject eventSamplingDataJson;
        String eventSamplingDataString = dataRegistry.safeGetString(DataRegistryKey.STRING_S3_CONFIG_EVENT_SAMPLING_DATA, "");

        try {
            eventSamplingDataJson = new JSONObject(eventSamplingDataString);
        } catch (JSONException e) {
            eventSamplingDataJson = new JSONObject();
        }

        this.eventSamplingModel = EventSamplingModel.createInstance(eventSamplingDataJson);

        dataRegistry.registerDataUnitEventListener(DataRegistryKey.STRING_S3_CONFIG_EVENT_SAMPLING_DATA, (oldValue, newValue) -> {
            try {
                String oldString = oldValue != null ? oldValue.getString() : "";
                String newString = newValue.getString();

                if (newString != null && !newString.equals(oldString)) {
                    this.eventSamplingModel = EventSamplingModel.createInstance(new JSONObject(newString));
                }
            } catch (DataUnit.DataUnitInvalidTypeCastingException | JSONException e) {
                AbxLog.e(e, true);
            }
        });

        this.eventsThatShouldNotBeSampled = new HashSet<>();
        this.eventsThatShouldNotBeSampled.add("abx:first_open");
        this.eventsThatShouldNotBeSampled.add("abx:deeplink_open");
        this.eventsThatShouldNotBeSampled.add("abx:daily_first_open");
        this.eventsThatShouldNotBeSampled.add("abx:set_push");
        this.eventsThatShouldNotBeSampled.add("abx:login");
        this.eventsThatShouldNotBeSampled.add("abx:user_property_changed");
    }

    public boolean isEventBlockedBySamplingFilter(String eventNameWithGroup) {

        if (eventsThatShouldNotBeSampled.contains(eventNameWithGroup)) {
            AbxLog.d(eventNameWithGroup + " should not be blocked by sampling filter!", true);
            return false;
        }

        int probability = eventSamplingModel.getProbability(eventNameWithGroup);

        if (probability == -1) {
            return false;
        }

        if (eventSamplingModel.type == EventSamplingType.ADID) {
            String adid = dataRegistry.safeGetString(DataRegistryKey.STRING_ADID, null);

            if (adid != null) {
                int remainder = CoreUtils.getSHA256HashToInt(adid, 7) % 100;

                if (probability <= remainder) {
                    AbxLog.d(String.format(Locale.US, "%s is blocked by sampling filter! (remainder = %d%%, probability = %d%%)",
                            eventNameWithGroup, remainder, probability), true);
                    return true;
                }
            }
        }
//        else if (eventSamplingModel.type == EventSamplingType.LOG_ID) {
//            //Not implement yet.
//        }

        return false;
    }

    public boolean isEventBlockedBySamplingFilter(LogEventParameter logEventParameter) {
        return isEventBlockedBySamplingFilter(getEventNameWithGroup(logEventParameter));
    }

    private String getEventNameWithGroup(LogEventParameter logEventParameter) {
        String eventNameWithGroup;

        if (CoreConstants.EVENT_START_SESSION.equals(logEventParameter.eventName)) {
            eventNameWithGroup = Optional.ofNullable(logEventParameter.eventParam)
                    .map(param -> (String) param.get("abx:open_type"))
                    .map(openType -> openType.replace("string:", ""))
                    .map(replacedOpenType -> logEventParameter.group + ":" + replacedOpenType)
                    .orElse("");
        } else {
            eventNameWithGroup = logEventParameter.group + ":" + logEventParameter.eventName;
        }
        return eventNameWithGroup;
    }
}
