package com.igaworks.v2.core.push.notification;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.os.Build;
import android.os.Bundle;

import org.json.JSONException;
import org.json.JSONObject;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.domain.ABXConstants;
import io.adbrix.sdk.domain.CompatConstants;
import io.adbrix.sdk.ui.push.components.PushController;
import io.adbrix.sdk.ui.push.utils.PushUtils;
import io.adbrix.sdk.utils.CommonUtils;

public class AbxPushReceiver extends BroadcastReceiver {
    private enum ActionType{
        LOCAL_PUSH_RECEIVE,
        LOCAL_PUSH_CLICK,
        REMOTE_PUSH_RECEIVE,
        REMOTE_PUSH_CLICK,
        NONE
    }
    @Override
    public void onReceive(final Context context, final Intent intent) {
        if(intent == null){
            AbxLog.w("intent is null", false);
            return;
        }
        String action = intent.getAction();
        if(CommonUtils.isNullOrEmpty(action)){
            AbxLog.w("intent action is empty", false);
            return;
        }
        Bundle bundle = intent.getExtras();
        if(CommonUtils.isNullOrEmpty(bundle)){
            AbxLog.w("extras is empty", false);
            return;
        }
        AbxLog.d("AbxPushReceiver onReceive() "+intent.toString(), false);
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.JELLY_BEAN) {
            AbxLog.e("App Version is " + Build.VERSION.SDK_INT + ". Should Be higher than " + Build.VERSION_CODES.JELLY_BEAN, true);
            return;
        }
        if(!PushController.getInstance().isPushAvailable()){
            return;
        }
        ActionType actionType = getActionType(intent.getAction());
        if(actionType == ActionType.NONE){
            AbxLog.w("Doesn't match in AbxPushReceiver's actions", false);
            return;
        }
        if(!isPushDataAvailable(actionType, bundle)){
            AbxLog.w("push data is not available", false);
            return;
        }
        if(isPushOpenAction(actionType)){
            PushController.getInstance().processPushOpen(intent);
            return;
        }
        if(!isPushReceivedAction(actionType)){
            return;
        }
        if(!isCanShowNotificationWhenForeground(actionType, bundle)){
            AbxLog.w("push can't show when foreground state because alwaysIsShown value is false", false);
            return;
        }
        if(isDuplicateNotification(actionType, bundle)){
            return;
        }
        PushController.getInstance().showNotification(context, intent);
        insertPushData(actionType, bundle);
    }

    private ActionType getActionType(String action){
        if(action.equals(CompatConstants.LOCAL_PUSH_OPEN_CLCKED)){
            return ActionType.LOCAL_PUSH_CLICK;
        }
        if(action.equals(CompatConstants.REMOTE_PUSH_OPEN_CLICKED)){
            return ActionType.REMOTE_PUSH_CLICK;
        }
        if(action.equals(CompatConstants.PUSH_LOCAL_MESSAGE_RECEIVE)){
            return ActionType.LOCAL_PUSH_RECEIVE;
        }
        if(action.equals(CompatConstants.PUSH_REMOTE_MESSAGE_RECEIVE)){
            return ActionType.REMOTE_PUSH_RECEIVE;
        }
        if(action.equals(CompatConstants.PUSH_REMOTE_MESSAGE_RECEIVE_FROM_FIREBASE_MESSAGING_SERVICE)){
            return ActionType.REMOTE_PUSH_RECEIVE;
        }
        return ActionType.NONE;
    }

    private boolean isPushOpenAction(ActionType actionType){
        switch (actionType){
            case LOCAL_PUSH_CLICK:
            case REMOTE_PUSH_CLICK:
                return true;
            default:
                return false;
        }
    }
    private boolean isPushReceivedAction(ActionType actionType){
        switch (actionType){
            case LOCAL_PUSH_RECEIVE:
            case REMOTE_PUSH_RECEIVE:
                return true;
            default:
                return false;
        }
    }
    private boolean isPushDataAvailable(ActionType actionType, Bundle bundle){
        boolean result = false;
        switch (actionType){
            case LOCAL_PUSH_RECEIVE:{
                String style = bundle.getString(CompatConstants.PUSH_PROP_STYLE);
                if(CommonUtils.isNullOrEmpty(style)){
                    return result;
                }
                if(CompatConstants.PUSH_STYLE_BIG_TEXT.equals(style)){
                    String properties = bundle.getString(CompatConstants.PUSH_TEXT_PROPERTIES);
                    if(CommonUtils.isNullOrEmpty(properties)){
                        return result;
                    }
                }
                if(CompatConstants.PUSH_STYLE_BIG_PICTURE.equals(style)){
                    String properties = bundle.getString(CompatConstants.PUSH_PICTURE_PROPERTIES);
                    if(CommonUtils.isNullOrEmpty(properties)){
                        return result;
                    }
                }
                break;
            }
            case REMOTE_PUSH_RECEIVE:{
                String properties = bundle.getString(ABXConstants.PUSH_REMOTE_FCM_KEY);
                if(CommonUtils.isNullOrEmpty(properties)){
                    return result;
                }
                break;
            }
        }
        result = true;
        return result;
    }

    private boolean isCanShowNotificationWhenForeground(ActionType actionType, Bundle bundle){
        if(!PushController.getInstance().isForeground()){
            return true;
        }
        boolean alwaysIsShown = true;
        if(actionType == ActionType.LOCAL_PUSH_RECEIVE){
            alwaysIsShown = bundle.getBoolean(CompatConstants.PUSH_USER_INPUT_ALWAYS_IS_SHOWN, true);
        }
        else if(actionType == ActionType.REMOTE_PUSH_RECEIVE){
            String receivedJSONString = bundle.getString(ABXConstants.PUSH_REMOTE_FCM_KEY);
            JSONObject objFromServer = null;
            try {
                objFromServer = new JSONObject(receivedJSONString);
                if (objFromServer.has(ABXConstants.PUSH_REMOTE_KEY_PUSH_ALWAYS_SHOWN)) {
                    if (PushUtils.checkHasVaildKey(objFromServer, ABXConstants.PUSH_REMOTE_KEY_PUSH_ALWAYS_SHOWN) && objFromServer.get(ABXConstants.PUSH_REMOTE_KEY_PUSH_ALWAYS_SHOWN) instanceof Boolean) {
                        alwaysIsShown = objFromServer.getBoolean(ABXConstants.PUSH_REMOTE_KEY_PUSH_ALWAYS_SHOWN);
                    }
                }
            } catch (Exception e) {
                AbxLog.w(e, true);
                return false;
            }
        }
        if (!alwaysIsShown){
            return false;
        }
        return true;
    }

    private boolean isDuplicateNotification(ActionType actionType, Bundle bundle){
        boolean result = false;
        if(actionType != ActionType.REMOTE_PUSH_RECEIVE){
            return result;
        }
        String receivedJSONString = bundle.getString(ABXConstants.PUSH_REMOTE_FCM_KEY);
        JSONObject jsonObject = new JSONObject();
        try {
            jsonObject = new JSONObject(receivedJSONString);
        } catch (JSONException e) {
            AbxLog.w("isPushAvailable() parse error: "+e.toString(), true);
            return result;
        }
        if(!PushUtils.checkHasVaildKey(jsonObject, ABXConstants.PUSH_REMOTE_KEY_NOTIFICATION_ID)){
            AbxLog.w("there is no PUSH_REMOTE_KEY_NOTIFICATION_ID key", true);
            return result;
        }
        int notificationId = -1;
        try {
            notificationId = jsonObject.getInt(ABXConstants.PUSH_REMOTE_KEY_NOTIFICATION_ID);
        } catch (JSONException e) {
            AbxLog.w("isPushAvailable() parse error: "+e.toString(), true);
            return result;
        }
        result = PushController.getInstance().isDuplicatedNotification(notificationId);
        return result;
    }

    private void insertPushData(ActionType actionType, Bundle bundle){
        if(actionType != ActionType.REMOTE_PUSH_RECEIVE){
            return;
        }
        String receivedJSONString = bundle.getString(ABXConstants.PUSH_REMOTE_FCM_KEY);
        if(CommonUtils.isNullOrEmpty(receivedJSONString)){
            AbxLog.w("abx:gf:fcm data is empty", false);
            return;
        }
        PushController.getInstance().insertPushData(receivedJSONString);
    }
}
