package io.adbrix.sdk.ui.inappmessage;

import android.app.Activity;
import android.content.Context;
import android.os.Build;
import android.os.Bundle;
import android.os.ResultReceiver;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewTreeObserver;
import android.view.Window;
import android.view.inputmethod.InputMethodManager;
import android.widget.FrameLayout;

import io.adbrix.sdk.R;
import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.domain.model.DfnInAppMessage;
import io.adbrix.sdk.utils.CommonUtils;

public class InAppMessageViewWrapper{
    private final DfnInAppMessage inAppMessage;
    private InAppMessageView contentView;
    private final InAppMessageViewListener inAppMessageViewListener;
    private View previouslyFocusedView = null;
    private boolean wasUsingKeypad = false;

    public InAppMessageViewWrapper(DfnInAppMessage inAppMessage, InAppMessageViewListener inAppMessageViewListener) {
        this.inAppMessage = inAppMessage;
        this.inAppMessageViewListener = inAppMessageViewListener;
    }

    public void setContentView(InAppMessageView view){
        this.contentView = view;
    }

    public void setInAppMessageViewListener(){
        if(CommonUtils.isNull(this.contentView)){
            AbxLog.d("setInAppMessageViewListener contentView is null", true);
            return;
        }
        this.contentView.setInAppMessageViewListener(inAppMessageViewListener);
    }

    public DfnInAppMessage getInAppMessage() {
        return inAppMessage;
    }

    public void open(Activity activity) {
        if(CommonUtils.isNull(this.contentView)){
            AbxLog.d("view is null", true);
            InAppMessageManager.getInstance().setIsInProgress(false);
            return;
        }
//        final ViewGroup parentViewGroup = parentView;
        final ViewGroup parentViewGroup = getParentViewGroup(activity);
        if(CommonUtils.isNull(parentViewGroup)){
            AbxLog.d("parentViewGroup is null", true);
            InAppMessageManager.getInstance().setIsInProgress(false);
            return;
        }
        AbxLog.d("parentViewGroup : "  + parentViewGroup.getChildCount(), true);
        this.previouslyFocusedView = activity.getCurrentFocus();
        final int parentViewGroupHeight = parentViewGroup.getHeight();
        if(parentViewGroupHeight == 0){
            ViewTreeObserver viewTreeObserver = parentViewGroup.getViewTreeObserver();
            if(viewTreeObserver.isAlive()){
                viewTreeObserver.addOnGlobalLayoutListener(
                        new ViewTreeObserver.OnGlobalLayoutListener() {
                            @Override
                            public void onGlobalLayout() {
                                AbxLog.d("Detected root view height of "+parentViewGroup.getHeight() + " in onGlobalLayout", true);
                                parentViewGroup.removeView(contentView);
                                addInAppMessageViewToViewGroup(activity, parentViewGroup, contentView);
                                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
                                    parentViewGroup.getViewTreeObserver().removeOnGlobalLayoutListener(this);
                                } else{
                                    parentViewGroup.getViewTreeObserver().removeGlobalOnLayoutListener(this);
                                }
                            }
                        });
            }
        } else {
            AbxLog.d("Detected root view height of "+parentViewGroup.getHeight() + " in onGlobalLayout", true);
            addInAppMessageViewToViewGroup(activity, parentViewGroup, contentView);
        }
    }

    private void addInAppMessageViewToViewGroup(Activity activity,
                                                ViewGroup parentViewGroup,
                                                View view){
//        AbxLog.d("adding inAppMessageView to parentViewGroup", true);
        try{
            parentViewGroup.removeView(view);
            if(CommonUtils.notNull(view.getParent())){
                AbxLog.d("view's parent is removed. because view already has a parent.", true);
                ((ViewGroup)view.getParent()).removeView((ViewGroup)view);
            }
            if(CommonUtils.notNull(view.getParent())){
                AbxLog.d("view already has a parent", true);
            } else{
                parentViewGroup.post(new Runnable() {
                    @Override
                    public void run() {
                        if(InAppMessageManager.getInstance().isShowing()){
                            AbxLog.d("InAppMessage is already shown", true);
                            InAppMessageManager.getInstance().setIsInProgress(false);
                            return;
                        }
                        parentViewGroup.addView(view, getLayoutParams(activity));
                        AbxLog.d("add view: "+view+"\nfrom parent: "+parentViewGroup+"\n"+activity, true);
                        view.invalidate();
                        ViewCompat.requestApplyInsets(parentViewGroup);
                        ViewCompat.setOnApplyWindowInsetsListener(parentViewGroup, new OnApplyWindowInsetsListener() {
                            @Override
                            public WindowInsetsCompat onApplyWindowInsets(View v, WindowInsetsCompat insets) {
                                if(CommonUtils.isNull(insets)){
                                    return insets;
                                }
                                applyWindowInsets(view, insets);
                                return insets;
                            }
                        });
                        finalizeViewBeforeDisplay(activity, view);
                        InAppMessageManager.getInstance().setIsInProgress(false);
                        InAppMessageManager.getInstance().setShowing(true);
                    }
                });
            }
        } catch(Exception e){
            AbxLog.e( "addInAppMessageViewToViewGroup failed. ", e, true);
            InAppMessageManager.getInstance().setIsInProgress(false);
            return;
        }
    }

    private ViewGroup.LayoutParams getLayoutParams(Activity activity){
        FrameLayout.LayoutParams layoutParams = new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.MATCH_PARENT);
        if(CommonUtils.isNull(inAppMessage)){
            AbxLog.d("getLayoutParams inAppMessage is null", true);
            return layoutParams;
        }
        if(CommonUtils.isNull(inAppMessage.getType())){
            AbxLog.d("getLayoutParams inAppMessageType is null", true);
            return layoutParams;
        }
        if(InAppMessageFactoryContainer.STICKY_BANNER.equals(inAppMessage.getType())){
            layoutParams = new FrameLayout.LayoutParams(FrameLayout.LayoutParams.MATCH_PARENT, FrameLayout.LayoutParams.WRAP_CONTENT);
            layoutParams = setStickyBannerLayoutGravity(layoutParams);
//            layoutParams = setStickyBannerLayoutMargins(activity, layoutParams);
        }
        return layoutParams;
    }

    private FrameLayout.LayoutParams setStickyBannerLayoutGravity(FrameLayout.LayoutParams layoutParams){
        layoutParams.gravity = Gravity.TOP;
        String stickyBannerAlign = inAppMessage.getStickyBannerAlign();
        if(CommonUtils.isNullOrEmpty(stickyBannerAlign)){
            return layoutParams;
        }
        if(StickyBannerMessageViewFactory.ALIGN_TOP.equals(stickyBannerAlign)){
            layoutParams.gravity = Gravity.TOP;
        } else{
            layoutParams.gravity = Gravity.BOTTOM;
        }
        return layoutParams;
    }

    private FrameLayout.LayoutParams setStickyBannerLayoutMargins(Activity activity, FrameLayout.LayoutParams layoutParams){
        int margins = (int)activity.getResources().getDimension(R.dimen.InAppMessage_SlideUp_Portrait_BaseLayout_Margins);
        layoutParams.setMargins(margins, margins, margins, margins);
        return layoutParams;
    }

    /**
     * 2022.11.08 bobos
     * 자동으로 최상위 view를 찾는 코드
     * @param activity
     * @return
     */
    private ViewGroup getParentViewGroup(Activity activity){
        Window window = activity.getWindow();
        if(CommonUtils.isNull(window)){
            AbxLog.d("parentWindow is null", true);
            return null;
        }
        View decorView = window.getDecorView();
        if(CommonUtils.isNull(decorView)){
            AbxLog.d("parentDecorView is null", true);
            return null;
        }
        ViewGroup viewGroup = decorView.findViewById(android.R.id.content);
        return viewGroup;
    }

    public ViewGroup getParentViewGroup(Activity activity, ViewGroup viewGroup){
        if(CommonUtils.isNull(viewGroup)){
            AbxLog.d("parentView is null, use android.R.id.content", true);
            return getParentViewGroup(activity);
//            return null;
        }
//        AbxLog.d("parentViewId is "+viewId);
//        ViewGroup viewGroup = activity.findViewById(viewId);
        return viewGroup;
    }

    public void close(Activity activity){
        AbxLog.d("closeInAppMessageView", true);
        if(CommonUtils.isNull(activity)){
            AbxLog.d("activity is null", true);
            InAppMessageManager.getInstance().setIsInProgress(false);
            return;
        }
        if(CommonUtils.isNull(contentView)){
            AbxLog.d("view is null", true);
            InAppMessageManager.getInstance().setIsInProgress(false);
            return;
        }
        InAppMessageManager.getInstance().setShowing(false);
        InAppMessageManager.getInstance().setIsInProgress(false);
        InAppMessageManager.getInstance().setIsInAnimating(false);
        activity.runOnUiThread(new Runnable() {
            @Override
            public void run() {
                boolean isSuccess = removeViewFromParent(activity, contentView);
                if(isSuccess){
                    if(CommonUtils.notNull(previouslyFocusedView)){
                        AbxLog.d("Returning focus to previouslyFocusedView. View: "+previouslyFocusedView, true);
                        previouslyFocusedView.requestFocus();
                        if(wasUsingKeypad){
                            showKeypad(activity);
                            wasUsingKeypad = false;
                        }
                    }
                    finishView();
                }
                InAppMessageManager.getInstance().setShowing(false);
                InAppMessageManager.getInstance().setIsInProgress(false);
            }
        });
    }

    private boolean removeViewFromParent(Activity activity, View view){
        boolean result = true;
        ViewGroup parentView = getParentViewGroup(activity);

        if(CommonUtils.isNull(parentView)){
            AbxLog.d("parentView is null", true);
            result = false;
            return result;
        }
        try {
            parentView.removeView(view);
            AbxLog.d("Removed view: "+view+"\nfrom parent: "+parentView+"\n"+activity, true);
        }catch (IllegalStateException e){
            AbxLog.e(e, true);
            result = false;
            return result;
        }
        return result;
    }

    public InAppMessageViewListener getInAppMessageViewListener(){
        return this.inAppMessageViewListener;
    }

    public void clearInAppMessageViewAnimation(){
        this.contentView.clearAnimation();
    }
    private void finishView(){
        contentView.finishView();
    }

    private void applyWindowInsets(View view, WindowInsetsCompat insets){
        ViewGroup.LayoutParams layoutParams = view.getLayoutParams();
        if (layoutParams == null || !(layoutParams instanceof ViewGroup.MarginLayoutParams)) {
            AbxLog.d("Not applying window insets.", true);
            return;
        }
        ViewGroup.MarginLayoutParams marginLayoutParams;
        try {
            marginLayoutParams = (ViewGroup.MarginLayoutParams)layoutParams;
        }catch (Exception e){
            AbxLog.w(e, true);
            return;
        }
        if(InAppMessageFactoryContainer.STICKY_BANNER.equals(inAppMessage.getType())){
            marginLayoutParams.setMargins(getMaxSafeLeftInset(insets), getMaxSafeTopInset(insets), getMaxSafeRightInset(insets), getMaxSafeBottomInset(insets));
            return;
        }
//        marginLayoutParams.setMargins(0, getMaxSafeTopInset(insets), 0, 0);
    }

    private void finalizeViewBeforeDisplay(Activity activity, final View view) {
        if(CommonUtils.isNull(inAppMessage.getType())){
            setFocusableInTouchModeAndRequestFocus(view);
            return;
        }
        switch (inAppMessage.getType()){
            case InAppMessageFactoryContainer.MODAL:
            case InAppMessageFactoryContainer.FULL_SCREEN:
                setFocusableInTouchModeAndRequestFocus(view);
                hideKeypad(activity);
                break;
            case InAppMessageFactoryContainer.STICKY_BANNER:
                setFocusableInTouchModeAndRequestFocus(view);
                break;
        }
    }

    private int getMaxSafeLeftInset(WindowInsetsCompat windowInsets) {
        if (windowInsets.getDisplayCutout() != null) {
            final DisplayCutoutCompat displayCutout = windowInsets.getDisplayCutout();
            return Math.max(displayCutout.getSafeInsetLeft(), windowInsets.getSystemWindowInsetLeft());
        } else {
            return windowInsets.getSystemWindowInsetLeft();
        }
    }

    private int getMaxSafeRightInset(WindowInsetsCompat windowInsets) {
        if (windowInsets.getDisplayCutout() != null) {
            final DisplayCutoutCompat displayCutout = windowInsets.getDisplayCutout();
            return Math.max(displayCutout.getSafeInsetRight(), windowInsets.getSystemWindowInsetRight());
        } else {
            return windowInsets.getSystemWindowInsetRight();
        }
    }

    private int getMaxSafeTopInset(WindowInsetsCompat windowInsets) {
        if (windowInsets.getDisplayCutout() != null) {
            final DisplayCutoutCompat displayCutout = windowInsets.getDisplayCutout();
            return Math.max(displayCutout.getSafeInsetTop(), windowInsets.getSystemWindowInsetTop());
        } else {
            return windowInsets.getSystemWindowInsetTop();
        }
    }

    private int getMaxSafeBottomInset(WindowInsetsCompat windowInsets) {
        if (windowInsets.getDisplayCutout() != null) {
            final DisplayCutoutCompat displayCutout = windowInsets.getDisplayCutout();
            return Math.max(displayCutout.getSafeInsetBottom(), windowInsets.getSystemWindowInsetBottom());
        } else {
            return windowInsets.getSystemWindowInsetBottom();
        }
    }

    private boolean isDeviceNotInTouchMode(View view) {
        return !view.isInTouchMode();
    }

    private void setFocusableInTouchModeAndRequestFocus(View view) {
        try {
            view.setFocusableInTouchMode(true);
            view.requestFocus();
        } catch (Exception e) {
            AbxLog.e( "Caught exception while setting view to focusable in touch mode and requesting focus.", e, true);
        }
    }

    private void hideKeypad(Activity activity){
        if(this.previouslyFocusedView == null){
            return;
        }
        InputMethodManager inputMethodManager = (InputMethodManager)activity.getSystemService(Context.INPUT_METHOD_SERVICE);
        inputMethodManager.hideSoftInputFromWindow(this.previouslyFocusedView.getWindowToken(), InputMethodManager.HIDE_NOT_ALWAYS, new ResultReceiver(null){
            @Override
            protected void onReceiveResult(int resultCode, Bundle resultData) {
                super.onReceiveResult(resultCode, resultData);
                if (resultCode == InputMethodManager.RESULT_UNCHANGED_HIDDEN || resultCode == InputMethodManager.RESULT_HIDDEN){
                    wasUsingKeypad = true;
                }
            }
        });
    }
    private void showKeypad(Activity activity){
        InputMethodManager inputMethodManager = (InputMethodManager)activity.getSystemService(Context.INPUT_METHOD_SERVICE);
        inputMethodManager.showSoftInput(previouslyFocusedView, InputMethodManager.SHOW_IMPLICIT);
    }
}
