package io.adbrix.sdk.data.repository;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Arrays;
import java.util.ConcurrentModificationException;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.ReentrantReadWriteLock;

import javax.annotation.Nullable;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.ILogger;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.repository.datasource.IDataContext;
import io.adbrix.sdk.domain.CoreConstants;

public class DataRegistry {

    private final IDataContext dataContext;
    private final ILogger logger;
    private final ReentrantReadWriteLock readWriteLock;
    private ConcurrentHashMap<DataRegistryKey, DataUnit> registry;
    private HashMap<DataRegistryKey, List<IDataUnitEventListener>> eventListenerTable;
    private boolean isListeningOnDataUnitChange = false;
    /**
     * isPersistence = true 로 셋팅된 dataUnit이 put되었을때 true로 변경되며, archiving 판단기준으로 사용한다.
     */
    private boolean saveSwitch = false;

    public DataRegistry(IDataContext dataContext, ILogger logger) {
        this.registry = new ConcurrentHashMap<>();
        this.dataContext = dataContext;
        this.logger = logger;
        this.eventListenerTable = new HashMap<>();
        this.readWriteLock = new ReentrantReadWriteLock();

        //데이터 컨텍스트에서 기존에 저장된 레지스트리를 가져와서 최초 설정한다.
        String archivingJson = this.dataContext.getValueOrNull(CoreConstants.DATA_REGISTRY_ARCHIVING_KEY);
        if (archivingJson == null) {
            logger.debug("DataRegistry 데이터가 없습니다. 앱을 최초 실행했거나, 사용자가 데이터를 삭제할경우 발생합니다.");
        } else {
            fillRegistryFromJsonString(archivingJson);
        }
    }

    private void fillRegistryFromJsonString(String jsonString) {
        try {
            JSONObject jsonObject = new JSONObject(jsonString);
            JSONArray jsonArray = jsonObject.getJSONArray("dataUnits");

            for (int i = 0; i < jsonArray.length(); i++) {
                JSONObject dataUnitJson = jsonArray.getJSONObject(i);

                Object srcValue;
                String provider;

                try {
                    srcValue = dataUnitJson.getString("src_value");
                    provider = dataUnitJson.getString("provider");
                } catch (JSONException e) {
                    srcValue = null;
                    provider = null;
                }

                this.registry.put(
                        DataRegistryKey.parseEnum(dataUnitJson.getInt("key")),
                        new DataUnit(
                                DataRegistryKey.parseEnum(dataUnitJson.getInt("key")),
                                srcValue,
                                dataUnitJson.getInt("priority"),
                                provider,
                                dataUnitJson.getBoolean("is_persistence")
                        ));
            }
        } catch (Exception e) {
            AbxLog.e(e, true);
            this.logger.error("DataRegistry에서 전체 항목을 가져오는데 실패했습니다. 원본 JSON : " + jsonString);
        }
    }

    public void saveRegistry() {
        if (saveSwitch) {
            try {
                this.dataContext.setValue(CoreConstants.DATA_REGISTRY_ARCHIVING_KEY, this.toJsonString());
            } catch (JSONException e) {
                logger.debug("DataRegistry 데이터를 저장하지 못했습니다. 이런 경우는 발생해서는 안됩니다.");
            }

            saveSwitch = false;
        }
    }

    public void clearRegistry() {
        this.registry = new ConcurrentHashMap<>();
        this.eventListenerTable = new HashMap<>();
        this.dataContext.clear();
    }

    private String toJsonString() throws JSONException {
        JSONObject root = new JSONObject();
        try {
            JSONArray dataUnits = new JSONArray();
            for (DataUnit dataUnit : this.registry.values()) {
                dataUnits.put(dataUnit.getJson());
            }
            root.put("dataUnits", dataUnits);
        } catch (ConcurrentModificationException concurrentModificationException) {
            AbxLog.d("DataRegistry.toJsonString exception: " + Arrays.toString(concurrentModificationException.getStackTrace()), true);
            return root.toString();
        }
        return root.toString();
    }

    public void putDataRegistry(DataUnit newValue) {
        readWriteLock.writeLock().lock();
        if (newValue == null || newValue.key == null)
            return;

        try {
            DataUnit oldValue = registry.get(newValue.key);

            if (oldValue == null ||
                    newValue.priority >= oldValue.priority) {
                registry.put(newValue.key, newValue);
                onDataUnitChanged(oldValue, newValue);
                saveSwitch = newValue.isPersistence | saveSwitch;
            }
        } finally {
            readWriteLock.writeLock().unlock();
            saveRegistry();
        }
    }

    public void putDataRegistry(List<DataUnit> in) {
        for (DataUnit dataUnit : in) {
            putDataRegistry(dataUnit);
        }
    }

    private DataUnit get(DataRegistryKey key) throws CanNotFindDataRegistryKeyException {
        if (registry.containsKey(key))
            return registry.get(key);
        else
            throw new CanNotFindDataRegistryKeyException();
    }

    public int getInt(DataRegistryKey key) throws DataUnit.DataUnitInvalidTypeCastingException, CanNotFindDataRegistryKeyException {
        return this.get(key).getInt();
    }

    public String getString(DataRegistryKey key) throws DataUnit.DataUnitInvalidTypeCastingException, CanNotFindDataRegistryKeyException {
        return this.get(key).getString();
    }

    public boolean getBoolean(DataRegistryKey key) throws DataUnit.DataUnitInvalidTypeCastingException, CanNotFindDataRegistryKeyException {
        return this.get(key).getBoolean();
    }

    @Nullable
    public Boolean getBooleanOrNull(DataRegistryKey key) {
        try {
            return getBoolean(key);
        } catch (DataUnit.DataUnitInvalidTypeCastingException e) {
            return null;
        } catch (DataRegistry.CanNotFindDataRegistryKeyException e) {
            return null;
        }
    }

    public float getFloat(DataRegistryKey key) throws DataUnit.DataUnitInvalidTypeCastingException, CanNotFindDataRegistryKeyException {
        return this.get(key).getFloat();
    }

    public float getLong(DataRegistryKey key) throws DataUnit.DataUnitInvalidTypeCastingException, CanNotFindDataRegistryKeyException {
        return this.get(key).getLong();
    }

    public int safeGetInt(DataRegistryKey key, int defaultValue) {
        readWriteLock.readLock().lock();
        try {
            DataUnit dataUnit = registry.get(key);
            if (dataUnit != null) {
                return dataUnit.getInt();
            }
        } catch (Exception e) {
            AbxLog.e(e, true);
        } finally {
            readWriteLock.readLock().unlock();
        }

        return defaultValue;
    }

    public String safeGetString(DataRegistryKey key, String defaultValue) {
        readWriteLock.readLock().lock();
        try {
            DataUnit dataUnit = registry.get(key);
            if (dataUnit != null) {
                return dataUnit.getString();
            }
        } catch (Exception e) {
            AbxLog.e(e, true);
        } finally {
            readWriteLock.readLock().unlock();
        }

        return defaultValue;
    }

    public boolean safeGetBoolean(DataRegistryKey key, boolean defaultValue) {
        readWriteLock.readLock().lock();
        try {
            DataUnit dataUnit = registry.get(key);
            if (dataUnit != null) {
                return dataUnit.getBoolean();
            }
        } catch (Exception e) {
            AbxLog.e(e, true);
        } finally {
            readWriteLock.readLock().unlock();
        }

        return defaultValue;
    }

    public float safeGetFloat(DataRegistryKey key, float defaultValue) {
        readWriteLock.readLock().lock();
        try {
            DataUnit dataUnit = registry.get(key);
            if (dataUnit != null) {
                return dataUnit.getFloat();
            }
        } catch (Exception e) {
            AbxLog.e(e, true);
        } finally {
            readWriteLock.readLock().unlock();
        }

        return defaultValue;
    }

    public double safeGetDouble(DataRegistryKey key, double defaultValue) {
        readWriteLock.readLock().lock();
        try {
            DataUnit dataUnit = registry.get(key);
            if (dataUnit != null) {
                return dataUnit.getDouble();
            }
        } catch (Exception e) {
            AbxLog.e(e, true);
        } finally {
            readWriteLock.readLock().unlock();
        }

        return defaultValue;
    }

    public long safeGetLong(DataRegistryKey key, long defaultValue) {
        readWriteLock.readLock().lock();
        try {
            DataUnit dataUnit = registry.get(key);
            if (dataUnit != null) {
                return dataUnit.getLong();
            }
        } catch (Exception e) {
            AbxLog.e(e, true);
        } finally {
            readWriteLock.readLock().unlock();
        }

        return defaultValue;
    }

    public long incrementAndGetLong(DataRegistryKey key) {
        long current = safeGetLong(key, 0L);
        long next = current + 1L;

        putDataRegistry(new DataUnit(
                key,
                next,
                5,
                this.getClass().getName(),
                true
        ));

        return next;
    }

    public void registerDataUnitEventListener(DataRegistryKey key, IDataUnitEventListener eventListener) {
        if (!eventListenerTable.containsKey(key)) {
            eventListenerTable.put(key, Arrays.asList(eventListener));
        } else {
            List<IDataUnitEventListener> eventListeners = eventListenerTable.get(key);
            if (eventListeners != null) {
                eventListeners.add(eventListener);
            }
        }
    }

    public void registerDataUnitEventListener(List<DataRegistryKey> keys, IDataUnitEventListener eventListener) {
        for (DataRegistryKey key : keys) {
            registerDataUnitEventListener(key, eventListener);
        }
    }

    private void onDataUnitChanged(@Nullable DataUnit oldValue, DataUnit newValue) {
        if (isListeningOnDataUnitChange) {
            List<IDataUnitEventListener> eventListeners = eventListenerTable.get(newValue.key);
            if (eventListeners != null) {
                for (IDataUnitEventListener eventListener : eventListeners) {
                    eventListener.onChange(oldValue, newValue);
                }
            }
        }
    }

    public void setListenOnChange(boolean bool) {
        isListeningOnDataUnitChange = bool;
    }

    public void unregisterAllDataUnitEventListener() {
        eventListenerTable.clear();
    }

    public static class CanNotFindDataRegistryKeyException extends Exception {
    }
}
