package io.adbrix.sdk.configuration;

import android.app.Activity;
import android.content.Context;
import android.content.Intent;

import com.igaworks.v2.core.AdBrixRm;
import com.igaworks.v2.core.result.GetAttributionDataResult;

import org.json.JSONObject;

import java.util.HashMap;
import java.util.List;

import javax.annotation.Nullable;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.GooglePlayReferrerProperties;
import io.adbrix.sdk.component.IABXComponentsFactory;
import io.adbrix.sdk.component.TryOptional;
import io.adbrix.sdk.data.SdkVersion;
import io.adbrix.sdk.data.dataprovider.DeviceRealtimeDataProvider;
import io.adbrix.sdk.data.dataprovider.DeviceStaticDataProvider;
import io.adbrix.sdk.data.dataprovider.IDataProviderResultListener;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.net.IApiConnection;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.LogMessageFormat;
import io.adbrix.sdk.domain.Repository;
import io.adbrix.sdk.domain.exception.BeforeInitializeException;
import io.adbrix.sdk.domain.function.Completion;
import io.adbrix.sdk.domain.interactor.GdprForgetMeUseCase;
import io.adbrix.sdk.domain.model.ActionHistory;
import io.adbrix.sdk.domain.model.ActionHistoryIdType;
import io.adbrix.sdk.domain.model.DfnInAppMessage;
import io.adbrix.sdk.domain.model.Empty;
import io.adbrix.sdk.domain.model.Error;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.domain.model.Response;
import io.adbrix.sdk.domain.model.Result;
import io.adbrix.sdk.domain.model.Success;
import io.adbrix.sdk.domain.model.UserPropertyCommand;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.CoreUtils;

public class InitializingABXContext implements IABXContext {
    private IABXComponentsFactory componentsFactory;
    private Repository repository;

    public InitializingABXContext(IABXComponentsFactory componentsFactory, Repository repository) {
        this.componentsFactory = componentsFactory;
        this.repository = repository;
    }

    @Override
    public Result<String> initialize(Context context, String appkey, String secretKey) {
        AbxLog.d("Trying to initialize Adbrix Remastered SDK", true);
        AbxLog.d("SDK Version :: " + SdkVersion.SDK_VERSION, true);

        if (componentsFactory == null) {
            return Error.of("InitializngABXContext.initialize is called before startcontroller");
        }

        //1. 컴포넌트 팩토리 android Context 설정
        componentsFactory.setAndroidContext(context);

        //2. 앱키 시크릿키 설정
        if (!this.setupAppkeyAndSecretKey(appkey, secretKey)) {
            return Error.of("Error occurred while processing appKey / secretKey.");
        }

        //3.앱 설정 refresh
        //TODO timeout 5초
        //new InitDefaultConfigValueUseCase(repository).execute();
        //new SyncApplicationConfigureUseCase(repository).execute();
        componentsFactory.createOrGetRemoteConfigProvider().syncApplicationConfigure();
        boolean isAdbrixDisabled = false;
        DataRegistry dataRegistry = componentsFactory.createOrGetDataRegistry();
        DefaultABXContextController controller = (DefaultABXContextController) componentsFactory.getAbxContextController();
        if(controller.isNeedRestoreUser()){
            isAdbrixDisabled = CoreUtils.isAdbrixDisabled(dataRegistry, ()->{});
        } else{
            isAdbrixDisabled = CoreUtils.isAdbrixDisabled(dataRegistry, new GdprForgetMeUseCase(repository)::execute);
        }

        if (isAdbrixDisabled) {
            return Error.of("Adbrix Disabled");
        }

        //4. 프로바이더 분류 후 초기값 설정 및 실행
        this.initializeProviders();

        return Success.of("Result of initializing Adbrix SDK : Success");
    }

    @Override
    public void getInstallReferrer() {
        GooglePlayReferrerProperties googlePlayReferrerProperties = new GooglePlayReferrerProperties(componentsFactory.getAndroidContext(), componentsFactory);
        if(googlePlayReferrerProperties.needRefresh()){
            googlePlayReferrerProperties.saveInstallReferrerCompleted(false);
            googlePlayReferrerProperties.startConnection(new GooglePlayReferrerProperties.Callback() {
                @Override
                public void getReferrerMap(HashMap<String, Object> referrerMap, GooglePlayReferrerProperties googlePlayReferrerProperties) {
                    AbxLog.d("getInstallReferrer success: "+referrerMap.toString(), true);
                    googlePlayReferrerProperties.endConnection();
                    componentsFactory.getAbxContextController().flushQueuedEvent();
                }
            });
        }
    }

    private boolean setupAppkeyAndSecretKey(String appkey, String secretKey) {
        // AppKey, SecretKey가 22자 이상인지

        if (CommonUtils.isNullOrEmpty(appkey) || appkey.length() != 22) {
            AbxLog.e("AppKey가 길이 22의 문자열이 아닙니다.", true);
            return false;
        }

        if (CommonUtils.isNullOrEmpty(secretKey) || secretKey.length() != 22) {
            AbxLog.e("SecretKey가 길이 22의 문자열이 아닙니다.", true);
            return false;
        }

        AbxLog.d("Both AppKey and SecretKey are valid", true);

        TryOptional.of(componentsFactory::createOrGetDataRegistry).ifPresent(it -> {

            it.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.STRING_APPKEY,
                            appkey,
                            5,
                            "Initializing",
                            false
                    )
            );

            it.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.STRING_SECRETKEY,
                            secretKey,
                            5,
                            "Initializing",
                            false
                    )
            );
        });

        return true;
    }

    /**
     * 초기화중에도 이벤트는 받을 수 있어야 합니다.
     */
    @Override
    public void logEvent(LogEventParameter logEventParameter) {
    }

    @Override
    public void logSameEventWithPaging(String eventName, String group, List<JSONObject> eventParamList) {

    }

    @Override
    public void flushAllEvents(Completion<Result<Empty>> completion) {
        if(CommonUtils.notNull(completion)) {
            completion.handle(Error.of(String.format(LogMessageFormat.INVALID_CONTEXT, "InitializingABXContext", "flushAllEvents")));
        }
        throw new BeforeInitializeException();
    }

    /**
     * 초기화 중에도 사용자 속성은 설정할 수 있어야 합니다.
     */
    @Override
    public void saveUserProperty(UserPropertyCommand userPropertyCommand) {
    }

    @Override
    public Boolean saveUserPropertyWithoutEvent(UserPropertyCommand userPropertyCommand) {
        return null;
    }

    @Override
    public void clearUserProperty() {

    }

    @Override
    public void getUserId(Completion<Result<String>> completion) {
        if(CommonUtils.notNull(completion)) {
            completion.handle(Error.of(String.format(LogMessageFormat.INVALID_CONTEXT, "InitializingABXContext", "getUserId")));
        }
    }

    @Override
    public void login(String userId, Completion<Result<Response>> completion) {
        throw new BeforeInitializeException();
    }

    @Override
    public void logout(Completion<Result<Response>> completion) {
        throw new BeforeInitializeException();
    }

    private void initializeProviders() {
        try {
            //필수적 프로바이더 최초값 설정
            DeviceStaticDataProvider deviceStaticDataProvider = componentsFactory.createOrGetDeviceStaticDataProvider();
            DeviceRealtimeDataProvider deviceRealtimeDataProvider = componentsFactory.createOrGetDeviceRealtimeDataProvider();
            deviceStaticDataProvider.provideDefaultValues();
            AbxLog.d("프로바이더 초기화에 성공했습니다.-" + deviceStaticDataProvider.getClass().getName(), true);
            deviceRealtimeDataProvider.provideDefaultValues();
            deviceRealtimeDataProvider.asyncProvide(new IDataProviderResultListener() {
                @Override
                public void onSuccess() {
                    AbxLog.d("프로바이더 초기화에 성공했습니다.-" + deviceRealtimeDataProvider.getClass().getName(), true);
                }

                @Override
                public void onFail(Exception e) {
                    AbxLog.e("프로바이더 초기화에 실패했습니다.-" + deviceRealtimeDataProvider.getClass().getName(),e, true);
                }
            });
        } catch (Exception e) {
            AbxLog.e("컴포넌트를 만들지 못했습니다.",e, true);
        }
    }

    @Override
    public void runOnIdleTime() {
        //무시
    }

    @Override
    public void onResume(Activity activity) {
        throw new BeforeInitializeException();
    }

    @Override
    public void onPause() {
        throw new BeforeInitializeException();
    }

    @Override
    public void deeplink(Activity deeplinkActivity) {
        throw new BeforeInitializeException();
    }

    @Override
    public void deeplinkWithIntent(Intent deeplinkIntent) {
        throw new BeforeInitializeException();
    }

    @Override
    public void putDataRegistry(DataUnit dataUnit) {
        throw new BeforeInitializeException();
    }

    @Override
    public Result<Empty> deleteUserData(String userId) {
        return Error.of(String.format(LogMessageFormat.INVALID_CONTEXT, "InitializingABXContext", "deleteUserData"));
    }

    @Override
    public Result<Empty> restartSDK(String userId) {
        return Error.of(String.format(LogMessageFormat.INVALID_CONTEXT, "InitializingABXContext", "restartSDK"));
    }

    @Override
    public void fetchActionHistoryFromServer(@Nullable String token, ActionHistoryIdType idType, List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        if(CommonUtils.notNull(completion)) {
            completion.handle(Error.of(String.format(LogMessageFormat.INVALID_CONTEXT, "InitializingABXContext", "fetchActionHistoryFromServer")));
        }
        throw new BeforeInitializeException();
    }

    @Override
    public void insertPushData(String pushDataString) {
        throw new BeforeInitializeException();
    }

    @Override
    public void getActionHistory(int skip, int limit, List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        if(CommonUtils.notNull(completion)) {
            completion.handle(Error.of(String.format(LogMessageFormat.INVALID_CONTEXT, "InitializingABXContext", "getActionHistory")));
        }
        throw new BeforeInitializeException();
    }

    @Override
    public void getAllActionHistory(List<String> actionType, Completion<Result<List<ActionHistory>>> completion) {
        if(CommonUtils.notNull(completion)) {
            completion.handle(Error.of(String.format(LogMessageFormat.INVALID_CONTEXT, "InitializingABXContext", "getAllActionHistory")));
        }
        throw new BeforeInitializeException();
    }

    @Override
    public void deleteActionHistory(@Nullable String token, String historyId, long timestamp, Completion<Result<Empty>> completion) {
        if(CommonUtils.notNull(completion)) {
            completion.handle(Error.of(String.format(LogMessageFormat.INVALID_CONTEXT, "InitializingABXContext", "deleteActionHistory")));
        }
        throw new BeforeInitializeException();
    }

    @Override
    public void deleteAllActionHistory(@Nullable String token, @Nullable ActionHistoryIdType idType, Completion<Result<Empty>> completion) {
        if(CommonUtils.notNull(completion)) {
            completion.handle(Error.of(String.format(LogMessageFormat.INVALID_CONTEXT, "InitializingABXContext", "deleteAllActionHistory")));
        }
        throw new BeforeInitializeException();
    }

    @Override
    public void clearSyncedActionHistoryInLocalDB(Completion<Result<Empty>> completion) {
        if(CommonUtils.notNull(completion)) {
            completion.handle(Error.of(String.format(LogMessageFormat.INVALID_CONTEXT, "InitializingABXContext", "clearSyncedActionHistoryInLocalDB")));
        }
        throw new BeforeInitializeException();
    }

    @Override
    public void clearAllActionHistoryInLocalDB() {
        throw new BeforeInitializeException();
    }

    @Override
    public void fetchInAppMessage(Completion<Result<Empty>> completion) {
        if(CommonUtils.notNull(completion)) {
            completion.handle(Error.of(String.format(LogMessageFormat.INVALID_CONTEXT, "InitializingABXContext", "fetchInAppMessage")));
        }
        throw new BeforeInitializeException();
    }

    @Override
    public void getAllInAppMessage(Completion<Result<List<DfnInAppMessage>>> completion) {
        if(CommonUtils.notNull(completion)) {
            completion.handle(Error.of(String.format(LogMessageFormat.INVALID_CONTEXT, "InitializingABXContext", "getAllInAppMessage")));
        }
        throw new BeforeInitializeException();
    }

    @Override
    public void openInAppMessage(String campaignId, Completion<Result<Empty>> completion) {
        if(CommonUtils.notNull(completion)) {
            completion.handle(Error.of(String.format(LogMessageFormat.INVALID_CONTEXT, "InitializingABXContext", "openInAppMessage")));
        }
        throw new BeforeInitializeException();
    }

    @Override
    public void deleteAllInAppMessageDBContents() {
        throw new BeforeInitializeException();
    }

    @Override
    public void requestGetAttributionData(String logId, AdBrixRm.GetAttributionDataCallback callback, DataRegistry dataRegistry) {
        if(CommonUtils.notNull(callback)){
            callback.onCallback(new GetAttributionDataResult(IApiConnection.RESPONSE_CODE_ERROR, GetAttributionDataResult.Result.UNKNOWN_ERROR.getValue()));
        }
        throw new BeforeInitializeException();
    }
}
