package io.adbrix.sdk.component.autoEvent;

import android.app.Activity;
import android.content.Intent;
import android.net.Uri;
import android.os.Bundle;
import android.os.Handler;

import com.igaworks.v2.core.AdBrixRm;
import com.igaworks.v2.core.push.notification.AbxPushReceiver;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.DeeplinkParameterSet;
import io.adbrix.sdk.component.DeeplinkPostingObservable;
import io.adbrix.sdk.component.GooglePlayReferrerProperties;
import io.adbrix.sdk.component.IObservable;
import io.adbrix.sdk.component.IObserver;
import io.adbrix.sdk.component.InAppMessageAutoFetchObservable;
import io.adbrix.sdk.component.UserPropertyManager;
import io.adbrix.sdk.component.migration_v1.V1MigrationManager;
import io.adbrix.sdk.data.DFNSessionState;
import io.adbrix.sdk.data.RemoteConfigProvider;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.ABXConstants;
import io.adbrix.sdk.domain.CompatConstants;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.Repository;
import io.adbrix.sdk.domain.function.Completion;
import io.adbrix.sdk.domain.model.Empty;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.domain.model.Result;
import io.adbrix.sdk.ui.push.models.AbxRemotePushModel;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.CoreUtils;

public class SessionAutoEventGenerator implements IObservable<LogEventParameter> {

    private static final String BASIC_OPEN = "basic_open";
    private static final String FIRST_OPEN = "first_open";
    private static final String DEEPLINK_OPEN = "deeplink_open";
    private static final String PUSH_OPEN = "push_open";

    private DataRegistry dataRegistry;
    private FirstOpenAutoEventGenerator firstOpenAutoEventGenerator;

    private String lastCreatedSessionId;
    private String lastEndSessionId;
    private long lastSessionCreatedTime;
    private long lastSessionEndTime;
    private boolean isEndSessionSentLast;
    private Repository repository;
    private DeeplinkParameterSet deeplinkParameterSet;
    private V1MigrationManager v1MigrationManager;
    private List<IObserver<LogEventParameter>> observers;
    private DFNSessionState dfnSessionState;
    private RemoteConfigProvider remoteConfigProvider;

    private Timer timer;
    private TimerTask timerTask;

    public SessionAutoEventGenerator(
            DataRegistry dataRegistry,
            FirstOpenAutoEventGenerator firstOpenAutoEventGenerator,
            Repository repository,
            DeeplinkParameterSet deeplinkParameterSet,
            V1MigrationManager v1MigrationManager,
            DFNSessionState dfnSessionState,
            RemoteConfigProvider remoteConfigProvider
    ) {

        this.dataRegistry = dataRegistry;
        this.firstOpenAutoEventGenerator = firstOpenAutoEventGenerator;
        this.repository = repository;
        this.deeplinkParameterSet = deeplinkParameterSet;
        this.v1MigrationManager = v1MigrationManager;
        this.dfnSessionState = dfnSessionState;
        this.remoteConfigProvider = remoteConfigProvider;
        observers = new ArrayList<>();

        lastCreatedSessionId = dataRegistry.safeGetString(DataRegistryKey.STRING_LAST_CREATED_SESSION_ID, null);
        lastEndSessionId = dataRegistry.safeGetString(DataRegistryKey.STRING_LAST_END_SESSION_ID, null);
        lastSessionCreatedTime = dataRegistry.safeGetLong(DataRegistryKey.LONG_LAST_SESSION_CREATED_TIME, 0);
        lastSessionEndTime = dataRegistry.safeGetLong(DataRegistryKey.LONG_LAST_SESSION_END_TIME, 0);
        isEndSessionSentLast = dataRegistry.safeGetBoolean(DataRegistryKey.BOOLEAN_IS_END_SESSION_SENT_LAST, false);
    }

    @Override
    public void add(IObserver<LogEventParameter> observer) {
        observers.add(observer);
    }

    @Override
    public void delete(IObserver<LogEventParameter> observer) {
        observers.remove(observer);
    }

    @Override
    public void notifyObserver(LogEventParameter logEventParameter) {
        for (IObserver<LogEventParameter> observer : observers) {
            observer.update(logEventParameter);
        }
    }

    public void onResume(final Activity activity) {
        boolean isPushOpenClickIntentActionOnResume = isPushOpenClickIntentActionOnResume(activity);
        boolean isRemotePushOpenClick = false;
        Map<String, Object> pushOpenEventParam = new HashMap<>();
        if(isPushOpenClickIntentActionOnResume){
            pushOpenEventParam = getPushOpenParamOnIntent(activity.getIntent());
            broadcastToAbxPushReceiver(activity);
            isRemotePushOpenClick = isRemotePushOpenClickIntentActionOnResume(activity, pushOpenEventParam);
            removePushData(activity);
        }

        if (timer != null) {
            // Cancel the app background event from being sent,
            // it hasn't been enough time between previous Session end and this start
            cancelEndSessionSchedule();
            if(isPushOpenClickIntentActionOnResume){
                removePushData(activity);
            }
            return;
        }

        if (this.dfnSessionState.isSessionStarted())
            return;

        if (!CommonUtils.isNullOrEmpty(lastCreatedSessionId) && // First StartSession
                lastCreatedSessionId != lastEndSessionId) {
            LogEventParameter eventParameter = new LogEventParameter(
                    CoreConstants.GROUP_ABX,
                    CoreConstants.EVENT_END_SESSION,
                    null,
                    0,
                    getSessionLength()
            );

            processEndSession(eventParameter);
        }

        if (firstOpenAutoEventGenerator.isTrigger()) {

            if (v1MigrationManager.isMigrationCompleted()) {
                processStartSession(BASIC_OPEN, null);
                return;
            }
            Map<String, Object> referrerMap = firstOpenAutoEventGenerator.getGooglePlayReferrerPropertiesOrNull();
            AbxLog.d("FirstOpen start", true);
            LogEventParameter startSessionLogEventParameter = processStartSession(FIRST_OPEN, referrerMap);
            AbxLog.d("FirstOpen finished successfully", true);
            firstOpenAutoEventGenerator.runDeferredDeeplinkProcess(startSessionLogEventParameter);
            CoreUtils.clearAdbrixTrackingParamFromCurrentActivity(activity, dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null));
        } else if (CoreUtils.isActivityStartedByDeeplink(activity, dataRegistry)) {
            Uri decodedUri = CoreUtils.getDeeplinkUriFromCurrentActivity(activity);

            if (decodedUri == null) {
                AbxLog.d("deeplink uri is null", true);
                processStartSession(BASIC_OPEN, null);
                return;
            }

            String deeplinkUrl = decodedUri.toString();

            Map<String, Object> deeplinkEventParam = CoreUtils.getDeeplinkParameterFromUriString(deeplinkUrl);

            if (deeplinkParameterSet.isAlreadyUsedDeeplinkParameter(deeplinkEventParam)) {
                processStartSession(BASIC_OPEN, null);
                return;
            }

            processStartSession(DEEPLINK_OPEN, deeplinkEventParam);
            Uri uri = CoreUtils.getDeeplinkUriFromCurrentActivity(activity, false);
            DeeplinkPostingObservable.getInstance().notifyObserver(uri.toString());

            CoreUtils.clearAdbrixTrackingParamFromCurrentActivity(activity, dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null));
        } else if (isRemotePushOpenClick) {
            processStartSession(PUSH_OPEN, pushOpenEventParam);
        } else {
            processStartSession(BASIC_OPEN, null);
        }
    }

    // Schedule timer to send an end_session event after SESSION_TIMEOUT ms have passed
    public void onPause() {
        cancelEndSessionSchedule();

        long currentTimeMillis = System.currentTimeMillis();
        updateLastSessionEndTime(currentTimeMillis);

        String eventId = CommonUtils.randomUUIDWithCurrentTime(currentTimeMillis);
        String eventDateTime = CommonUtils.getCurrentUTCInDBFormat(currentTimeMillis);

        LogEventParameter eventParameter = new LogEventParameter(
                eventId,
                null,
                CoreConstants.GROUP_ABX,
                CoreConstants.EVENT_END_SESSION,
                null,
                0,
                getSessionLength(),
                eventDateTime
        );
        timer = new Timer();
        synchronized (timer){
            timerTask = new TimerTask() {
                @Override
                public void run() {
                    AbxLog.w("SessionAutoEventTimer is completed", true);
                    timer = null;
                    processEndSession(eventParameter);
                }
            };
            timer.schedule(timerTask, CoreConstants.SESSION_TIMEOUT);
        }
    }

    public void ping() {
        // updateElapsedTimeInForeground(SystemClock.elapsedRealtime());
    }

    public boolean isStartSessionByDeeplink(Activity deeplinkActivity) {
        return CoreUtils.isActivityStartedByDeeplink(deeplinkActivity, dataRegistry) && !this.dfnSessionState.isSessionStarted();
    }

    public boolean isStartSessionByDeeplinkIntent(Intent deeplinkIntent) {
        return CoreUtils.isStartedByDeeplinkIntent(deeplinkIntent, dataRegistry) && !this.dfnSessionState.isSessionStarted();
    }

    private boolean isPushOpenClickIntentActionOnResume(Activity activity){
        boolean isOpenClick = false;
        if(activity == null){
            AbxLog.i("activity is null", true);
            return isOpenClick;
        }
        Intent intent = activity.getIntent();
        if(intent == null){
            AbxLog.i("intent is null", true);
            return isOpenClick;
        }
        String action = intent.getAction();
        if(CommonUtils.isNullOrEmpty(action)) {
            AbxLog.i("intent action is null", true);
            return isOpenClick;
        }
        if(CompatConstants.REMOTE_PUSH_OPEN_CLICKED.equals(action)){
            isOpenClick = true;
            return isOpenClick;
        }
        if(CompatConstants.LOCAL_PUSH_OPEN_CLCKED.equals(action)){
            isOpenClick = true;
            return isOpenClick;
        }

        return isOpenClick;
    }

    private boolean isRemotePushOpenClickIntentActionOnResume(Activity activity, Map<String, Object> pushOpenParam){
        boolean result = false;
        Intent intent = activity.getIntent();
        String action = intent.getAction();
        if(CompatConstants.REMOTE_PUSH_OPEN_CLICKED.equals(action)){
            if(!CommonUtils.isNullOrEmpty(pushOpenParam)){
                result = true;
                return result;
            }
        }

        return result;
    }

    private void broadcastToAbxPushReceiver(Activity activity){
        Intent intent = activity.getIntent();
        final String action = intent.getAction();
        intent.setClass(activity.getApplicationContext(), AbxPushReceiver.class);
        intent.setAction(action);
        activity.sendBroadcast(intent);
        AbxLog.i("broadcast to AbxPushReceiver", true);
    }

    private void removePushData(Activity activity){
        Intent intent = activity.getIntent();
        intent.removeExtra(ABXConstants.PUSH_REMOTE_FCM_KEY);
        intent.removeExtra(CompatConstants.PUSH_TEXT_PROPERTIES);
        intent.removeExtra(CompatConstants.PUSH_PICTURE_PROPERTIES);
        activity.setIntent(intent);
    }

    private Map<String, Object> getPushOpenParamOnIntent(Intent intent){
        Map<String, Object> result = new HashMap<>();
        if(intent == null){
            AbxLog.i("intent is null", true);
            return result;
        }
        Bundle intentExtra = intent.getExtras();
        if(intentExtra == null){
            AbxLog.i("intent extras is null", true);
            return result;
        }
        String temp = intent.getExtras().getString(ABXConstants.PUSH_REMOTE_FCM_KEY);
        if(CommonUtils.isNullOrEmpty(temp)){
            AbxLog.i("PUSH_REMOTE_DATA is null or empty", true);
            return result;
        }
        String campaignId = "";
        String stepId = "";
        String cycleTime = "";
        int revisionNo = -1;

        JSONObject pushJsonObject;
        try {
            pushJsonObject = new JSONObject(temp);
            if (pushJsonObject.has(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_ID)) {
                campaignId = pushJsonObject.getString(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_ID);
            }
            if (pushJsonObject.has(ABXConstants.GROWTH_EVENT_KEY_CYCLE_TIME)) {
                cycleTime = pushJsonObject.getString(ABXConstants.GROWTH_EVENT_KEY_CYCLE_TIME);
            }
            if (pushJsonObject.has(ABXConstants.GROWTH_EVENT_KEY_STEP_ID)) {
                stepId = pushJsonObject.getString(ABXConstants.GROWTH_EVENT_KEY_STEP_ID);
            }
            if (pushJsonObject.has(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_REVISION_NO)) {
                revisionNo = pushJsonObject.getInt(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_REVISION_NO);
            }
        } catch (JSONException e) {
            AbxLog.e(e, true);
            return result;
        }
        JSONObject pushOpenParam = new JSONObject();
        try {
            pushOpenParam
                .put(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_ID, campaignId)
                .put(ABXConstants.GROWTH_EVENT_KEY_STEP_ID, stepId)
                .put(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_REVISION_NO, revisionNo)
                .put(ABXConstants.GROWTH_EVENT_KEY_CYCLE_TIME, cycleTime);
        } catch (JSONException e) {
            AbxLog.e(e, true);
            return result;
        }
        result = CommonUtils.getMapFromJSONObject(pushOpenParam);
        return result;
    }

    //Used for Start Session Event only
    private long getSessionInterval() {
        return lastSessionEndTime == 0 ? 0 : System.currentTimeMillis() - lastSessionEndTime;
    }

    //Used for EndSession Event only
    private long getSessionLength() {
        return lastSessionEndTime - lastSessionCreatedTime;
    }

    private LogEventParameter processStartSession(String openType, Map<String, Object> eventParam) {
        if (this.dfnSessionState.isSessionStarted()) {
            return null;
        }

        updateLastCreatedSessionId(CommonUtils.randomUUIDWithCurrentTime());
        updateLastSessionCreatedTime(System.currentTimeMillis());

        //InAppMessage
        repository.resetInAppMessageFrequencySession();

        int minutesToExpiry = dataRegistry.safeGetInt(DataRegistryKey.INT_IN_APP_MESSAGE_MINUTES_TO_EXPIRY, -1);
        long inAppMessageLastResponseTime = dataRegistry.safeGetLong(DataRegistryKey.LONG_IN_APP_MESSAGE_LAST_RESPONSE_TIME, -1);

        if (minutesToExpiry == -1 || inAppMessageLastResponseTime == -1 //first inAppMessage request
                || inAppMessageLastResponseTime + minutesToExpiry * 60 * 1000 < System.currentTimeMillis()//cache expired
        ) {
            AdBrixRm.fetchInAppMessage(new Completion<Result<Empty>>() {
                @Override
                public void handle(Result<Empty> result) {
                    AbxLog.d("SessionAutoEventGenerator fetchInAppMessage completed: " + result, true);
                    InAppMessageAutoFetchObservable.getInstance().notifyObserver(result);
                }
            });
        }
        //InAppMessage end

        //dfnId Load
        String dfnId = dataRegistry.safeGetString(DataRegistryKey.STRING_DFN_ID, null);
        if(CommonUtils.isNullOrEmpty(dfnId)){
            repository.loadDFNId();
        }
        //dfnId Load End

        startTvAttributionBeacon(dfnId);

        if (eventParam == null) {
            eventParam = new HashMap<>();
        }
        eventParam.put("abx:open_type", openType);

        LogEventParameter eventParameter = new LogEventParameter(
                CoreConstants.GROUP_ABX,
                CoreConstants.EVENT_START_SESSION,
                CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                getSessionInterval(),
                0
        );

        if (openType.equals(FIRST_OPEN)) {
            //DataRegistry에 last_first_open_event_id 설정
            dataRegistry.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.STRING_LAST_FIRSTOPEN_ID,
                            eventParameter.eventId,
                            5,
                            this.getClass().getName(),
                            true
                    ));

            //DataRegistry에 last_open_id 설정
            dataRegistry.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.STRING_LAST_OPEN_ID,
                            eventParameter.eventId,
                            5,
                            getClass().getName(),
                            true
                    ));
        } else if (openType.equals(DEEPLINK_OPEN)) {
            dataRegistry.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.STRING_LAST_DEEPLINK_ID,
                            eventParameter.eventId,
                            5,
                            this.getClass().getName(),
                            true
                    ));

            dataRegistry.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.STRING_LAST_OPEN_ID,
                            eventParameter.eventId,
                            5,
                            this.getClass().getName(),
                            true
                    ));
        }

        //start_session 시 session_order_no 초기화
        dataRegistry.putDataRegistry(
                new DataUnit(
                        DataRegistryKey.LONG_SESSION_ORDER_NO,
                        0L,
                        5,
                        this.getClass().getName(),
                        true
                ));

        notifyObserver(eventParameter);
        this.dfnSessionState.setIsSessionStarted(true);

        isEndSessionSentLast = false;

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_IS_END_SESSION_SENT_LAST,
                false,
                5,
                this.getClass().getName(),
                true
        ));
        return eventParameter;
    }

    private void processEndSession(LogEventParameter eventParameter) {
        if (isEndSessionSentLast)
            return;

        updateLastEndSessionId(lastCreatedSessionId);

        notifyObserver(eventParameter);
        this.dfnSessionState.setIsSessionStarted(false);

        isEndSessionSentLast = true;

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_IS_END_SESSION_SENT_LAST,
                true,
                5,
                this.getClass().getName(),
                true
        ));
    }

    public void cancelEndSessionSchedule() {
        AbxLog.w("SessionAutoEventTimer is cancelled", true);
        CommonUtils.cancelTimerTask(timerTask);
        timerTask = null;
        CommonUtils.cancelTimer(timer);
        timer = null;
    }

    private void updateLastCreatedSessionId(String id) {
        lastCreatedSessionId = id;
        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_LAST_CREATED_SESSION_ID,
                id,
                5,
                this.getClass().getName(),
                true));
    }

    private void updateLastEndSessionId(String id) {
        lastEndSessionId = id;
        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_LAST_END_SESSION_ID,
                id,
                5,
                this.getClass().getName(),
                true));
    }

    private void updateLastSessionCreatedTime(long time) {

        lastSessionCreatedTime = time;
        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.LONG_LAST_SESSION_CREATED_TIME,
                time,
                5,
                this.getClass().getName(),
                true));
    }

    private void updateLastSessionEndTime(long time) {
        lastSessionEndTime = time;
        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.LONG_LAST_SESSION_END_TIME,
                time,
                5,
                this.getClass().getName(),
                true));
    }

    private void startTvAttributionBeacon(String dfnId){
        repository.startTvAttributionBeacon(dfnId);
    }

    public void deactivate(){
        updateLastCreatedSessionId(null);
        updateLastEndSessionId(null);
        updateLastSessionEndTime(0);
        observers.clear();
    }
}
