package io.adbrix.sdk.utils;

import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Color;

import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.security.KeyManagementException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLSocketFactory;
import javax.net.ssl.TrustManager;
import javax.net.ssl.TrustManagerFactory;
import javax.net.ssl.X509TrustManager;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.net.ApiConnection;
import io.adbrix.sdk.data.net.TlsSocketFactory;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;

public class HttpConnectionUtils {
    public static HttpURLConnection createConnection(String urlString, ApiConnection.RequestMethod requestMethod, DataRegistry dataRegistry) throws IOException{
        URL url = new URL(urlString);
        HttpURLConnection connection = (HttpURLConnection) url.openConnection();
        connection.setReadTimeout(CoreConstants.NETWORK_TIMEOUT);
        connection.setConnectTimeout(CoreConstants.NETWORK_TIMEOUT);
        if(isDeferredDeeplinkRequest(urlString, dataRegistry)){
            connection.setReadTimeout(CoreConstants.NETWORK_TIMEOUT_FOR_DEFERRED_DEEPLINK);
            connection.setConnectTimeout(CoreConstants.NETWORK_TIMEOUT_FOR_DEFERRED_DEEPLINK);
        }

        if (urlString.startsWith("https")) {
            configureHttpsConnection((HttpsURLConnection) connection);
        }
        connection.setRequestMethod(requestMethod.getMethodString());
        connection.setInstanceFollowRedirects(true);

        return connection;
    }

    private static boolean isDeferredDeeplinkRequest(String urlString, DataRegistry dataRegistry){
        boolean result = false;
        if(dataRegistry == null){
            return result;
        }
        String appKey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null);
        String deferredDeeplinkUrlString = String.format(CoreConstants.deferredDeeplinkUrlReq, appKey);
        if(urlString.equals(deferredDeeplinkUrlString)){
            result = true;
            return result;
        }
        return result;
    }

    private static void configureHttpsConnection(HttpsURLConnection connection) {
        /**
         * 2022.07.11 bobos
         * 앱에서 setDefaultSSLSocketFactory를 호출했을 경우 통신에 문제가 생기는 것을 방지하기 위함
         */
        SSLSocketFactory sslSocketFactory = getSSLSocketFactory();
        if(sslSocketFactory == null){
            return;
        }
        connection.setSSLSocketFactory(sslSocketFactory);
    }

    public static SSLSocketFactory getSSLSocketFactory() {
        SSLSocketFactory sslSocketFactory = null;
        TrustManager[] trustManagers = getPlatformTrustManager();
        if(trustManagers == null){
            return null;
        }
        try {
            sslSocketFactory = new TlsSocketFactory(trustManagers, new java.security.SecureRandom());
        } catch (KeyManagementException e) {
            AbxLog.e(e, true);
        } catch (NoSuchAlgorithmException e) {
            AbxLog.e(e, true);
        }
        return sslSocketFactory;
    }

    /**
     * 2022.07.11 bobos
     * OS 기본 값을 가져와서 Google Play 안전하지 않은 TrustManager 경고 대응
     * @see "https://github.com/square/okhttp/blob/3ad1912f783e108b3d0ad2c4a5b1b89b827e4db9/okhttp/src/jvmMain/kotlin/okhttp3/internal/platform/Platform.kt"
     * @return
     */
    public static TrustManager[] getPlatformTrustManager() {
        TrustManagerFactory trustManagerFactory = null;
        try {
            trustManagerFactory = TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
            trustManagerFactory.init((KeyStore) null);
        } catch (NoSuchAlgorithmException e) {
            AbxLog.e(e, true);
        } catch (KeyStoreException e) {
            AbxLog.e(e, true);
        }
        if(trustManagerFactory == null){
            AbxLog.e("trustManagerFactory is null", true);
            return null;
        }
        TrustManager[] trustManagers = trustManagerFactory.getTrustManagers();
        if (trustManagers.length != 1 || !(trustManagers[0] instanceof X509TrustManager)) {
            AbxLog.e("Unexpected default trust managers:" + Arrays.toString(trustManagers), true);
            return null;
        }
        return trustManagers;
    }
    public static Bitmap getBitmapFromURL(String imageUrl, int attempt) {
        return getBitmapFromURL(imageUrl, CoreConstants.NETWORK_TIMEOUT, attempt);
    }

    public static Bitmap getBitmapFromURL(String imageUrl, int timeout, int attempt) {
        HttpURLConnection connection = null;
        InputStream input = null;
        Bitmap bitmap = null;
        if(CommonUtils.isNullOrEmpty(imageUrl)){
            AbxLog.w("getBitmapFromURL is fail. imageUrl is null or empty", false);
            return bitmap;
        }
        try {
            connection = createConnection(imageUrl, ApiConnection.RequestMethod.GET, null);
            connection.setConnectTimeout(timeout);
            connection.setReadTimeout(timeout);
            connection.setDoInput(true);
            connection.setUseCaches(false);
            connection.connect();
            if(connection.getResponseCode() == HttpURLConnection.HTTP_OK){
                input = connection.getInputStream();
                bitmap = BitmapFactory.decodeStream(input);
            }
            AbxLog.i("getBitmapFromURL HttpURLConnection responseCode: "+connection.getResponseCode()+" url: "+imageUrl,true);
            if(input != null){
                input.close();
            }
            if(connection != null){
                connection.disconnect();
            }
            if(!isColoredImage(bitmap)){
                if(attempt < 1){
                    AbxLog.d("getBitmapFromURL isWhiteImage reDownload..", true);
                    bitmap = null;
                    bitmap = getBitmapFromURL(imageUrl, timeout,1);
                }
            }
        } catch (Exception e) {
            AbxLog.w("getBitmapFromURL is fail. url: "+imageUrl,e, false);
        }
        return bitmap;
    }

    private static boolean isColoredImage(Bitmap bitmap){
        boolean result = true;
        if(bitmap == null){
            return result;
        }
        try {
            for (int x = 0; x< bitmap.getWidth(); x++) {
                for (int y = 0; y < bitmap.getHeight(); y++) {
                    int pixel = bitmap.getPixel(x,y);
                    int redValue = Color.red(pixel);
                    if(redValue != 255){
                        return result;
                    }
                    int blueValue = Color.blue(pixel);
                    if(blueValue != 255){
                        return result;
                    }
                    int greenValue = Color.green(pixel);
                    if(greenValue != 255){
                        return result;
                    }
                }
            }
        } catch (Exception e){
            AbxLog.e(e, true);
            return result;
        }
        result = false;
        return result;
    }
}
