package io.adbrix.sdk.ui.push.components;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteException;
import android.database.sqlite.SQLiteOpenHelper;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.utils.CommonUtils;

public class PushEventSqliteStore extends SQLiteOpenHelper implements IPushEventStore{

    private static PushEventSqliteStore instance;
    private static final String DEFAULT_INSTANCE = "default_instance";
    private static final String DATABASE_NAME = "abrix.v2.sql";
    private static final int DATABASE_VERSION = 2;

    private static final String PUSH_DATA_STORE_TABLE_NAME = "push_store";
    private static final String PUSH_LIST_TITLE_FIELD = "title";
    private static final String PUSH_LIST_CONTENT_FIELD = "content";
    private static final String PUSH_LIST_TIME_FIELD = "reserved_time";
    private static final String PUSH_LIST_EVENT_ID_FIELD = "event_id";
    private static final String PUSH_LIST_ID_FIELD = "id";
    private static final String PUSH_LIST_STACKED_FIELD = "is_stacked";
    private static final String CREATE_PUSH_LIST_STROE_TABLE = "CREATE TABLE IF NOT EXISTS "
            + PUSH_DATA_STORE_TABLE_NAME + " (" + PUSH_LIST_ID_FIELD + " INTEGER PRIMARY KEY AUTOINCREMENT NOT NULL,"
            + PUSH_LIST_EVENT_ID_FIELD + " INTEGER,"
            + PUSH_LIST_TIME_FIELD + " INTEGER,"
            + PUSH_LIST_STACKED_FIELD + " INTEGER,"
            + PUSH_LIST_TITLE_FIELD + " TEXT,"
            + PUSH_LIST_CONTENT_FIELD + " TEXT);";

    private File file;

    private PushEventSqliteStore(Context context, String instance) {
        super(context, getDatabaseName(instance), null, DATABASE_VERSION);
        file = context.getDatabasePath(getDatabaseName(instance));
    }

    public static synchronized PushEventSqliteStore createAndGetPushEventSqliteStore(Context context) {
        if (instance == null) {
            instance = new PushEventSqliteStore(context.getApplicationContext(), DEFAULT_INSTANCE);
        }
        return instance;
    }

    /*
    public static synchronized PushEventSqliteStore getInstance() {
        return instance;
    }
     */

    private static String getDatabaseName(String instance) {
        return (CommonUtils.isNullOrEmpty(instance) || instance.equals(DEFAULT_INSTANCE)) ? DATABASE_NAME : DATABASE_NAME + "_" + instance;
    }


    private static void convertIfCursorWindowException(RuntimeException e) {
        String message = e.getMessage();
        if (!CommonUtils.isNullOrEmpty(message) && message.startsWith("Cursor window allocation of")) {
            throw new CursorWindowAllocationException(message);
        } else {
            throw e;
        }
    }

    public long insertOrReplacePushEventToTableAndGetId(String title, String content, int eventId, long time, int isStacked) {
        String table = PUSH_DATA_STORE_TABLE_NAME;

        if (title == null) return -1;
        long result = -1;
        try {
            SQLiteDatabase db = getWritableDatabase();
            ContentValues contentValues = new ContentValues();
            contentValues.put(PUSH_LIST_EVENT_ID_FIELD, eventId);
            contentValues.put(PUSH_LIST_TIME_FIELD, time);
            contentValues.put(PUSH_LIST_TITLE_FIELD, title);
            contentValues.put(PUSH_LIST_CONTENT_FIELD, content);
            contentValues.put(PUSH_LIST_STACKED_FIELD, isStacked);
            result = db.insertWithOnConflict(
                    table,
                    null,
                    contentValues,
                    SQLiteDatabase.CONFLICT_REPLACE
            );
            if (result == -1) {
                AbxLog.w("insertOrReplacePushEventToTable Failed", true);
            }
        } catch (SQLiteException e) {
            AbxLog.w(String.format("insertOrReplacePushEvent in %s failed: %s", table, e.getMessage()), true);
            // Hard to recover from SQLiteExceptions, just start fresh
            delete();
        } catch (StackOverflowError e) {
            AbxLog.w(String.format("insertOrReplacePushEvent in %s failed: %s", table, e.getMessage()), true);
            // potential stack overflow error when getting database on custom Android versions
            delete();
        } finally {
            close();
        }
        return result;
    }


    public synchronized JSONArray getPushEventList() throws JSONException {
        return getPushEventFromTable(PUSH_DATA_STORE_TABLE_NAME);
    }

    public synchronized JSONArray getPushEventList(int isStacked) throws JSONException {
        return getPushEventFromTable(PUSH_DATA_STORE_TABLE_NAME, isStacked);
    }

    public synchronized long updatePushEventState(int id, boolean state) {
        return updatePushEventStateFromTable(PUSH_DATA_STORE_TABLE_NAME, id, state);
    }

    public synchronized long deletePushEvent(int eventId) {
        return deletePushEventFromTable(PUSH_DATA_STORE_TABLE_NAME, eventId);
    }


    @Override
    public void onCreate(SQLiteDatabase sqLiteDatabase) {
        // INTEGER PRIMARY KEY AUTOINCREMENT guarantees that all generated values
        // for the field will be monotonically increasing and unique over the
        // lifetime of the table, even if rows get removed
        sqLiteDatabase.execSQL(CREATE_PUSH_LIST_STROE_TABLE);
    }

    @Override
    public void onUpgrade(SQLiteDatabase sqLiteDatabase, int oldVersion, int newVersion) {
        if (oldVersion > newVersion) {
            AbxLog.e("onUpgrade() with invalid oldVersion and newVersion", true);
            resetDatabase(sqLiteDatabase);
            return;
        }

        if (newVersion <= 1) {
            return;
        }

        resetDatabase(sqLiteDatabase);
    }

    @Override
    public void onDowngrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        try {
            resetDatabase(db);
            db.setVersion(DATABASE_VERSION);
        }catch (Exception e){
            AbxLog.e(e, true);
        }
    }

    private void resetDatabase(SQLiteDatabase db) {
        db.execSQL("DROP TABLE IF EXISTS " + PUSH_DATA_STORE_TABLE_NAME);
        onCreate(db);
    }



    private void delete() {
        try {
            close();
            file.delete();
        } catch (SecurityException e) {
            AbxLog.e(e, true);
        }
    }


    private synchronized JSONArray getPushEventFromTable(String table) throws JSONException {
        return getPushEventFromTable(table, 0);
    }

    private synchronized JSONArray getPushEventFromTable(String table, int isStacked) throws JSONException {
        JSONArray listOfPush = new JSONArray(); // JSON ARRAY 로 교체
        Cursor cursor = null;
        try {
            SQLiteDatabase db = getReadableDatabase();
            cursor = db.query(table, new String[]{PUSH_LIST_ID_FIELD, PUSH_LIST_EVENT_ID_FIELD, PUSH_LIST_TIME_FIELD, PUSH_LIST_TITLE_FIELD, PUSH_LIST_CONTENT_FIELD, PUSH_LIST_STACKED_FIELD},
                    null, null, null, null,
                    PUSH_LIST_ID_FIELD + " ASC", null
            );

            while (cursor.moveToNext()) {
                int id = cursor.getInt(0);
                int eventId = cursor.getInt(1);
                long time = cursor.getLong(2);
                String title = cursor.getString(3);
                String content = cursor.getString(4);
                int notificationSuccess = cursor.getInt(5);
                if (isStacked == 1 && notificationSuccess == 0) {
                    continue;
                }
                JSONObject temp = new JSONObject();
                temp.put("id", id);
                temp.put("eventId", eventId);
                temp.put("notification_time", time);
                temp.put("title", title);
                temp.put("content", content);
//                temp.put("isStacked", notification_success == 1);
                listOfPush.put(temp);
            }
        } catch (SQLiteException e) {
            AbxLog.w(String.format("getEvents from %s failed: %s", table, e.getMessage()), true);
            // Hard to recover from SQLiteExceptions, just start fresh
            delete();
        } catch (StackOverflowError e) {
            AbxLog.w(String.format("getEvents from %s failed: %s", table, e.getMessage()), true);
            // potential stack overflow error when getting database on custom Android versions
            delete();
        } catch (RuntimeException e) {
            convertIfCursorWindowException(e);
        } finally {
            if (cursor != null) {
                cursor.close();
            }
            close();
        }
        return listOfPush;
    }



    private synchronized long updatePushEventStateFromTable(String table, int id, boolean state) {

        long result = -1;
        try {
            SQLiteDatabase db = getWritableDatabase();
            ContentValues contentValues = new ContentValues();
            if (state) contentValues.put(PUSH_LIST_STACKED_FIELD, 1);
            else contentValues.put(PUSH_LIST_STACKED_FIELD, 0);
            result = db.update(
                    table,
                    contentValues,
                    PUSH_LIST_ID_FIELD + "=" + id, null
            );
            if (result == -1) {
                AbxLog.w("insertOrReplacePushEventToTable Failed", true);
            }
        } catch (SQLiteException e) {
            AbxLog.w(String.format("insertOrReplacePushEvent in %s failed: %s", table, e.getMessage()), true);
            // Hard to recover from SQLiteExceptions, just start fresh
            delete();
        } catch (StackOverflowError e) {
            AbxLog.w(String.format("insertOrReplacePushEvent in %s failed: %s", table, e.getMessage()), true);
            // potential stack overflow error when getting database on custom Android versions
            delete();
        } finally {
            close();
        }
        return result;
    }



    private synchronized long deletePushEventFromTable(String table, int eventId) {
        long result = -1;
        try {
            SQLiteDatabase db = getWritableDatabase();
            result = db.delete(table, PUSH_LIST_EVENT_ID_FIELD + "=?", new String[]{String.valueOf(eventId)});
        } catch (SQLiteException e) {
            AbxLog.w(String.format("deletePushEvent from %s failed: %s", table, e.getMessage()), true);
            // Hard to recover from SQLiteExceptions, just start fresh
            delete();
        } catch (StackOverflowError e) {
            AbxLog.w(String.format("deletePushEvent from %s failed: %s", table, e.getMessage()), true);
            // potential stack overflow error when getting database on custom Android versions
            delete();
        } finally {
            close();
        }
        return result;
    }

    static class CursorWindowAllocationException extends RuntimeException {
        CursorWindowAllocationException(String description) {
            super(description);
        }
    }
}
