package io.adbrix.sdk.data;

import android.content.Context;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Arrays;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.IABXComponentsFactory;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.net.ApiConnection;
import io.adbrix.sdk.data.net.ApiConnectionManager;
import io.adbrix.sdk.data.net.IApiConnection;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.CoreUtils;

public class RemoteConfigProvider {
    private JSONArray configBlockedEventList;
    private long configAppScanPeriod;
    private boolean configAppScanStop;
    private int configPropertyMaxSize = ConfigConstants.CONFIG_DEFAULT_PROPERTY_MAX_SIZE;
    private String configSkIpPort = ConfigConstants.CONFIG_DEFAULT_SK_IP_PORT;
    private int configSkPingVersion = ConfigConstants.CONFIG_DEFAULT_SK_PING_VERSION;
    private int configSkPingPeriodSecond = ConfigConstants.CONFIG_DEFAULT_SK_PING_PERIOD_SECOND;
    private boolean configInAppMessageActive = false;
    private boolean configAdbrixPause;
    private boolean configAdbrixAllStop;
    private JSONObject configEventSamplings;
    private boolean configTVAttributionActive = false;

    private DataRegistry dataRegistry;
    private Context context;
    private IApiConnection apiConnection;

    public RemoteConfigProvider(DataRegistry dataRegistry, Context context, IApiConnection apiConnection){
        this.dataRegistry = dataRegistry;
        this.context = context;
        this.apiConnection = apiConnection;
        initDefaultConfigValue();
        //syncApplicationConfigure();
    }

    public void initDefaultConfigValue() {
        long tempLong;
        String tempStringArray;
        String tempString;

        tempStringArray = dataRegistry.safeGetString(DataRegistryKey.STRING_S3_CONFIG_BLOCKED_EVTLIST, null);
        if (tempStringArray != null) {
            try {
                configBlockedEventList = new JSONArray(tempStringArray);
            } catch (JSONException e) {
                AbxLog.e(e, true);
            }
        } else configBlockedEventList = ConfigConstants.CONFIG_DEFAULT_BLOCKED_EVENT_LIST;

        tempLong = dataRegistry.safeGetLong(DataRegistryKey.LONG_S3_CONFIG_APP_SCAN_PERIOD, -1);
        if (tempLong == -1)
            configAppScanPeriod = ConfigConstants.CONFIG_DEFAULT_APP_SCAN_PERIOD * 3600 * 1000;
        else configAppScanPeriod = (int) tempLong * 3600 * 1000;

        tempLong = dataRegistry.safeGetLong(DataRegistryKey.LONG_S3_CONFIG_APP_SCAN_STOP, -1);
        if (tempLong == -1) configAppScanStop = ConfigConstants.CONFIG_DEFAULT_APP_SCAN_ON_OFF;
        else configAppScanStop = tempLong == 1;

        tempLong = dataRegistry.safeGetLong(DataRegistryKey.LONG_S3_CONFIG_PROPERTY_KEY_MAX, -1);
        if (tempLong == -1)
            configPropertyMaxSize = ConfigConstants.CONFIG_DEFAULT_PROPERTY_MAX_SIZE;
        else configPropertyMaxSize = (int) tempLong;

        tempLong = dataRegistry.safeGetLong(DataRegistryKey.LONG_S3_CONFIG_ADBRIX_PAUSE, -1);
        if (tempLong == -1) configAdbrixPause = ConfigConstants.CONFIG_DEFAULT_ADBRIX_PAUSE;
        else configAdbrixPause = tempLong == 1;

        tempLong = dataRegistry.safeGetLong(DataRegistryKey.LONG_S3_CONFIG_ADBRIX_ALL_STOP, -1);
        if (tempLong == -1)
            configAdbrixAllStop = ConfigConstants.CONFIG_DEFAULT_ADBRIX_ALL_STOP;
        else configAdbrixAllStop = tempLong == 1;

        if (isConfigAdbrixAllStop()) configAdbrixPause = true;

        tempString = dataRegistry.safeGetString(DataRegistryKey.STRING_S3_CONFIG_SK_IP_PORT, null);
        if (tempString != null) {
            configSkIpPort = tempString;
        } else configSkIpPort = ConfigConstants.CONFIG_DEFAULT_SK_IP_PORT;

        tempLong = dataRegistry.safeGetInt(DataRegistryKey.INT_S3_CONFIG_SK_PING_VERSION, -1);
        if (tempLong == -1) {
            configSkPingVersion = ConfigConstants.CONFIG_DEFAULT_SK_PING_VERSION;
        } else configSkPingVersion = (int) tempLong;

        tempLong = dataRegistry.safeGetInt(DataRegistryKey.INT_S3_CONFIG_SK_PING_PERIOD_SECOND, -1);
        if (tempLong == -1)
            configSkPingPeriodSecond = ConfigConstants.CONFIG_DEFAULT_SK_PING_PERIOD_SECOND * 1000;
        else configSkPingPeriodSecond = (int) tempLong * 1000;

        tempLong = dataRegistry.safeGetLong(DataRegistryKey.LONG_S3_CONFIG_IN_APP_MESSAGE_ACTIVE, -1);
        if (tempLong == -1)
            configInAppMessageActive = ConfigConstants.CONFIG_DEFAULT_IN_APP_MESSAGE_ACTIVE;
        else configInAppMessageActive = tempLong == 1;

        tempString = dataRegistry.safeGetString(DataRegistryKey.STRING_S3_CONFIG_EVENT_SAMPLING_DATA, null);
        if (tempString != null) {
            try {
                configEventSamplings = new JSONObject(tempString);
            } catch (JSONException e) {
                AbxLog.e(e, true);
            }
        } else configEventSamplings = new JSONObject();

        tempLong = dataRegistry.safeGetLong(DataRegistryKey.LONG_S3_CONFIG_TV_ATTRIBUTION_WITHOUT_SUPPORT_LIBRARY_ACTIVE, -1);
        if (tempLong == -1)
            configTVAttributionActive = ConfigConstants.CONFIG_DEFAULT_TV_ATTRIBUTION_ACTIVE;
        else configTVAttributionActive = tempLong == 1;


        AbxLog.d("Initialized default value - adbrix pause       : " + isConfigAdbrixPause(), true);
        AbxLog.d("Initialized default value - banned event list  : " + getConfigBlockedEventList(), true);
//        AbxLog.d("Initialized default value - app scan period    : " + config_appScanPeriod/3600/1000,    true);
//        AbxLog.d("Initialized default value - app scan stop      : " + config_appScanStop,                true);
        AbxLog.d("Initialized default value - property max size  : " + getConfigPropertyMaxSize(), true);
        AbxLog.d("Initialized default value - adbrix all stop    : " + isConfigAdbrixAllStop(), true);
        AbxLog.d("Initialized default value - sk ip port         : " + getConfigSkIpPort(), true);
        AbxLog.d("Initialized default value - sk ping version    : " + getConfigSkPingVersion(), true);
        AbxLog.d("Initialized default value - sk ping period sec : " + getConfigSkPingPeriodSecond() / 1000, true);
        AbxLog.d("Initialized default value - inAppMessage active : " + isConfigInAppMessageActive(), true);
        AbxLog.d("Initialized default value - eventSamplings : " + getConfigEventSamplings().toString(), true);
        AbxLog.d("Initialized default value - TVAttribution(without support library) active : " + configTVAttributionActive, true);

    }

    public void syncApplicationConfigure() {

        if (CoreUtils.isGdprForgetMe(dataRegistry, () -> { })) return;
        if (CoreUtils.isAdbrixAllStop(dataRegistry)) {
            AbxLog.d("ADBRIX ALL STOP :: don't synchronize with server", false);
            return;
        }

        AbxLog.d("Trying to get application settings..", false);
        AbxLog.d("requestAppConfig API called!", true);
        try {
            ApiConnectionManager<Void> apiConnectionManager = new ApiConnectionManager<>(new ApiConnectionManager.Result<Void>() {
                @Override
                public void connectSuccess(String responseString, int responseCode, Void unused) {
//                    AbxLog.d("App-Config : connectSuccess ! : " + responseString, true);
                    parseResponseOfS3Config(responseString);
                    initDefaultConfigValue();
                    updateRemoteConfigUpdateDate();
                }

                @Override
                public void connectFail(int responseCode, Void unused) {
                    AbxLog.w("AppSyncConfig Failed : Can not get S3 config.json file....  responseCode : " + responseCode, true);
                }
            }, null);

            String appKey = dataRegistry.safeGetString(DataRegistryKey.STRING_APPKEY, null);
            String urlString = String.format(CoreConstants.configRequestUrl, appKey);

            IApiConnection apiConnection = this.apiConnection.get(urlString);

            apiConnectionManager.executeWithRetry(apiConnection);

        } catch (Exception e) {
            AbxLog.e(e, false);
        }
    }

    private void parseResponseOfS3Config(String responseString) {

        try {
            int temp;
            JSONObject jsonObjectFromS3Server;

            if (!CommonUtils.isNullOrEmpty(responseString))
                jsonObjectFromS3Server = new JSONObject(responseString);
            else return;

            temp = getDecryptedConfigValue(jsonObjectFromS3Server, ConfigConstants.S3_CONFIG_ADBRIX_PAUSE);
            if (temp != -1) {
                configAdbrixPause = temp == 1;

                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.LONG_S3_CONFIG_ADBRIX_PAUSE,
                                (long) temp,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("CONFIG::ADBRIX PAUSE : " + isConfigAdbrixPause(), true);
            }

            temp = getDecryptedConfigValue(jsonObjectFromS3Server, ConfigConstants.S3_CONFIG_ADBRIX_ALL_STOP);
            if (temp != -1) {
                configAdbrixAllStop = temp == 1;

                if (isConfigAdbrixAllStop()) {
                    configAdbrixPause = true;

                    dataRegistry.putDataRegistry(
                            new DataUnit(
                                    DataRegistryKey.LONG_S3_CONFIG_ADBRIX_PAUSE,
                                    (long) temp,
                                    5,
                                    this.getClass().getName(),
                                    true
                            )
                    );
                }

                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.LONG_S3_CONFIG_ADBRIX_ALL_STOP,
                                (long) temp,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("CONFIG::ADBRIX ALL STOP : " + isConfigAdbrixAllStop(), true);
            }

            temp = getDecryptedConfigValue(jsonObjectFromS3Server, ConfigConstants.S3_CONFIG_APP_SCAN_PERIOD);
            if (temp != -1) {
                configAppScanPeriod = temp * 3600 * 1000; // period : per hour

                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.LONG_S3_CONFIG_APP_SCAN_PERIOD,
                                (long) temp,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
//                AbxLog.d("CONFIG::APPSCAN PERIOD : " + temp, true);

                AbxLog.d("AbxApplicationScan:: DEPRECATED!!!", true);
            }

            temp = getDecryptedConfigValue(jsonObjectFromS3Server, ConfigConstants.S3_CONFIG_APP_SCAN_STOP);
            if (temp != -1) {
                configAppScanStop = temp == 1;

                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.LONG_S3_CONFIG_APP_SCAN_STOP,
                                (long) temp,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
//                AbxLog.d("CONFIG::APPSCAN STOP : " + config_appScanStop, true);
                AbxLog.d("AbxApplicationScan:: DEPRECATED!!!", true);
            }

            temp = getDecryptedConfigValue(jsonObjectFromS3Server, ConfigConstants.S3_CONFIG_PROPERTY_KEY_MAX);
            if (temp != -1) {
                configPropertyMaxSize = temp;

                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.LONG_S3_CONFIG_PROPERTY_KEY_MAX,
                                (long) temp,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("CONFIG::PROPERTY MAX SIZE : " + getConfigPropertyMaxSize(), true);
            }

            if (jsonObjectFromS3Server.has(ConfigConstants.S3_CONFIG_BLOCKED_EVTLIST)) {
                if (jsonObjectFromS3Server.get(ConfigConstants.S3_CONFIG_BLOCKED_EVTLIST) instanceof JSONArray) {
                    configBlockedEventList = (JSONArray) jsonObjectFromS3Server.get(ConfigConstants.S3_CONFIG_BLOCKED_EVTLIST);

                    dataRegistry.putDataRegistry(
                            new DataUnit(
                                    DataRegistryKey.STRING_S3_CONFIG_BLOCKED_EVTLIST,
                                    getConfigBlockedEventList().toString(),
                                    5,
                                    this.getClass().getName(),
                                    true
                            )
                    );
                    AbxLog.d("CONFIG::BANNED LIST : " + getConfigBlockedEventList().toString(), true);
                }
            }

            if (jsonObjectFromS3Server.has(ConfigConstants.S3_CONFIG_SK_IP_PORT)) {
                if (jsonObjectFromS3Server.get(ConfigConstants.S3_CONFIG_SK_IP_PORT) instanceof String) {
                    configSkIpPort = (String) jsonObjectFromS3Server.get(ConfigConstants.S3_CONFIG_SK_IP_PORT);
                    if (getConfigSkIpPort().length() != 17) {
                        AbxLog.d("CONFIG::SK IP PORT : " + "Wrong SK_IP_PORT!!", true);
                    } else {
                        dataRegistry.putDataRegistry(
                                new DataUnit(
                                        DataRegistryKey.STRING_S3_CONFIG_SK_IP_PORT,
                                        getConfigSkIpPort(),
                                        5,
                                        this.getClass().getName(),
                                        true
                                )
                        );
                        AbxLog.d("CONFIG::SK IP PORT : " + getConfigSkIpPort(), true);
                    }
                }
            }

            temp = getDecryptedConfigValue(jsonObjectFromS3Server, ConfigConstants.S3_CONFIG_SK_PING_VERSION);
            if (temp != -1) {
                configSkPingVersion = temp;

                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.INT_S3_CONFIG_SK_PING_VERSION,
                                temp,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("CONFIG::SK PING VER : " + getConfigSkPingVersion(), true);
            }

            temp = getDecryptedConfigValue(jsonObjectFromS3Server, ConfigConstants.S3_CONFIG_SK_PING_PERIOD_SECOND);
            if (temp != -1) {
                configSkPingPeriodSecond = temp;

                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.INT_S3_CONFIG_SK_PING_PERIOD_SECOND,
                                temp,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("CONFIG::SK PING PERIOD SEC : " + getConfigSkPingPeriodSecond(), true);
            }

            temp = getDecryptedConfigValue(jsonObjectFromS3Server, ConfigConstants.S3_CONFIG_IN_APP_MESSAGE_ACTIVE);
            if (temp != -1) {
                configInAppMessageActive = temp == 1;

                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.LONG_S3_CONFIG_IN_APP_MESSAGE_ACTIVE,
                                (long) temp,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("CONFIG::InAppMessage Active : " + isConfigInAppMessageActive(), true);
            }

            if (jsonObjectFromS3Server.optJSONObject(ConfigConstants.S3_CONFIG_EVENT_SAMPLING) != null) {
                JSONObject eventSamplingJson = jsonObjectFromS3Server.getJSONObject(ConfigConstants.S3_CONFIG_EVENT_SAMPLING);

                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.STRING_S3_CONFIG_EVENT_SAMPLING_DATA,
                                eventSamplingJson.toString(),
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("CONFIG::EVENT SAMPLING FILTER : " + eventSamplingJson.toString(), true);
            }

            temp = getDecryptedConfigValue(jsonObjectFromS3Server, ConfigConstants.S3_CONFIG_TV_ATTRIBUTION_WITHOUT_SUPPORT_LIBRARY_ACTIVE);
            if (temp != -1) {
                configTVAttributionActive = temp == 1;

                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.LONG_S3_CONFIG_TV_ATTRIBUTION_WITHOUT_SUPPORT_LIBRARY_ACTIVE,
                                (long) temp,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("CONFIG::TVAttribution Active : " + configTVAttributionActive, true);
            }

        } catch (JSONException e) {
            AbxLog.d("CONFIG : The format received from server is not valid. Can't create new JSONObject.", true);
            AbxLog.d(Arrays.toString(e.getStackTrace()), true);
        }
    }

    private int getDecryptedConfigValue(JSONObject objFromServer, String type) {
        String md5Hash;
        if (objFromServer.has(type)) {
            try {
                Object temp = objFromServer.get(type);
                String tempToString = "";
                if (temp instanceof String) tempToString = (String) temp;
                md5Hash = CoreUtils.get02XMD5(tempToString);
                if (ConfigConstants.S3_KEY.containsKey(md5Hash)) {
                    return ConfigConstants.S3_KEY.get(md5Hash);
                }
            } catch (JSONException e) {
                AbxLog.e(e, true);
                return -1;
            }
        }
        return -1;
    }

    public boolean isBlockedEvent(String eventName, String category) {

        if (getConfigBlockedEventList() != null && getConfigBlockedEventList().length() != 0) {
            for (int i = 0; i < getConfigBlockedEventList().length(); i++) {
                try {
                    String bannedEvent = getConfigBlockedEventList().getString(i);
                    String[] evtSplitByCategory = bannedEvent.split(":");

                    if (evtSplitByCategory.length != 2) continue;
                    if (!evtSplitByCategory[0].equals("abx") && !evtSplitByCategory[0].equals("custom"))
                        continue;
                    if (category.equals(CoreConstants.GROUP_CUSTOM) && evtSplitByCategory[0].equals("custom")) {
                        if (evtSplitByCategory[1].equals(eventName)) {
                            AbxLog.d("S3ConfigHandler :: custom event is blocked ! : " + eventName, true);
                            return true;
                        }
                    } else if (category.equals(CoreConstants.GROUP_ABX) && evtSplitByCategory[0].equals("abx")) {
                        if (evtSplitByCategory[1].equals(eventName)) {
                            AbxLog.d("S3ConfigHandler :: ABX event is blocked ! : " + eventName, true);
                            return true;
                        }
                    }
                } catch (JSONException e) {
                    AbxLog.e("Blocked Event Exception: ",e, true);
                }
            }
        }
        return false;
    }

    public JSONArray getConfigBlockedEventList() {
        return configBlockedEventList;
    }

    public long getConfigAppScanPeriod() {
        return configAppScanPeriod;
    }

    public boolean isConfigAppScanStop() {
        return configAppScanStop;
    }

    public int getConfigPropertyMaxSize() {
        return configPropertyMaxSize;
    }

    public String getConfigSkIpPort() {
        return configSkIpPort;
    }

    public int getConfigSkPingVersion() {
        return configSkPingVersion;
    }

    public int getConfigSkPingPeriodSecond() {
        return configSkPingPeriodSecond;
    }

    public boolean isConfigInAppMessageActive() {
        return configInAppMessageActive;
    }

    public boolean isConfigAdbrixPause() {
        return configAdbrixPause;
    }

    public boolean isConfigAdbrixAllStop() {
        return configAdbrixAllStop;
    }

    public JSONObject getConfigEventSamplings() {
        return configEventSamplings;
    }

    private void updateRemoteConfigUpdateDate(){
        String nowDate = CoreUtils.getDateViaGMTDateFormat(CoreConstants.ABX_DATE_FORMAT_SIMPLE);
        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_REMOTE_CONFIG_UPDATE_DATE,
                nowDate,
                5,
                this.getClass().getName(),
                true));
    }
}
