package io.adbrix.sdk.component;

import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.os.RemoteException;

import com.android.installreferrer.api.InstallReferrerClient;
import com.android.installreferrer.api.InstallReferrerStateListener;
import com.android.installreferrer.api.ReferrerDetails;

import java.util.HashMap;

import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.utils.CommonUtils;

public class GooglePlayReferrerProperties {
    public static final String MARKET_INSTALL_BTN_CLICKED = "abx:market_install_btn_clicked";
    public static final String APP_INSTALL_START = "abx:app_install_start";
    public static final String APP_INSTALL_COMPLETED = "abx:app_install_completed";
    public static final String REFERRER = "abx:referrer";
    public static final String APP_FIRST_OPEN = "abx:app_first_open";
    public static final String CLICK_ADKEY = "abx:click_adkey";

    private Context context;
    private IABXComponentsFactory componentsFactory;
    private InstallReferrerClient referrerClient;

    public GooglePlayReferrerProperties(Context context, IABXComponentsFactory componentsFactory){
        this.context = context;
        this.componentsFactory = componentsFactory;
        this.referrerClient = InstallReferrerClient.newBuilder(context).build();
    }

    public boolean isRequestCompleted(){
        DataRegistry dataRegistry = componentsFactory.createOrGetDataRegistry();
        boolean result = dataRegistry.safeGetBoolean(DataRegistryKey.BOOLEAN_IS_INSTALL_REFERRER_COMPLETED, false);
        return result;
    }

    public boolean needRefresh(){
        boolean result = false;
        DataRegistry dataRegistry = componentsFactory.createOrGetDataRegistry();
        String referrer = dataRegistry.safeGetString(DataRegistryKey.STRING_REFERRER, null);
        if(CommonUtils.isNullOrEmpty(referrer)){
            result = true;
            return result;
        }
        return result;
    }

    public void saveInstallReferrerCompleted(boolean isCompleted){
        DataRegistry dataRegistry = componentsFactory.createOrGetDataRegistry();
        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.BOOLEAN_IS_INSTALL_REFERRER_COMPLETED,
                isCompleted,
                5,
                this.getClass().getName(),
                true));
    }

    public HashMap<String, Object> getReferrerMap(){
        HashMap<String, Object> referrerMap = new HashMap<>();
        DataRegistry dataRegistry = componentsFactory.createOrGetDataRegistry();
        long marketInstallBtnClicked = dataRegistry.safeGetLong(DataRegistryKey.LONG_MARKET_INSTALL_BTN_CLICKED, -1);
        long appInstallStart = dataRegistry.safeGetLong(DataRegistryKey.LONG_APP_INSTALL_START, -1);
        int appInstallCompleted = dataRegistry.safeGetInt(DataRegistryKey.INT_APP_INSTALL_COMPLETED, 0);
        String referrer = dataRegistry.safeGetString(DataRegistryKey.STRING_REFERRER, null);
        int appFirstOpen = dataRegistry.safeGetInt(DataRegistryKey.INT_APP_FIRST_OPEN, 0);
        String clickAdkey = dataRegistry.safeGetString(DataRegistryKey.STRING_CLICK_ADKEY, null);
        referrerMap.put(GooglePlayReferrerProperties.MARKET_INSTALL_BTN_CLICKED, marketInstallBtnClicked);
        referrerMap.put(GooglePlayReferrerProperties.APP_INSTALL_START, appInstallStart);
        referrerMap.put(GooglePlayReferrerProperties.APP_INSTALL_COMPLETED, appInstallCompleted);
        referrerMap.put(GooglePlayReferrerProperties.REFERRER, referrer);
        referrerMap.put(GooglePlayReferrerProperties.APP_FIRST_OPEN, appFirstOpen);
        referrerMap.put(GooglePlayReferrerProperties.CLICK_ADKEY, clickAdkey);
        return referrerMap;
    }

    public void startConnection(Callback callback) {
        HashMap<String, Object> referrerMap = new HashMap<>();
        if(componentsFactory == null){
            AbxLog.w("ComponentsFactory is null", true);
            saveInstallReferrerCompleted(true);
            callback.getReferrerMap(referrerMap, this);
            return;
        }
        if(CommonUtils.isNull(context)){
            AbxLog.w("Context is null", true);
            saveInstallReferrerCompleted(true);
            callback.getReferrerMap(referrerMap, this);
            return;
        }
        try {
            referrerClient.startConnection(new InstallReferrerStateListener() {
                @Override
                public void onInstallReferrerSetupFinished(int responseCode) {
                    switch (responseCode) {
                        case InstallReferrerClient.InstallReferrerResponse.OK:
                            onInstallReferrerResponseOK(referrerMap, referrerClient);
                            break;
                        default:
                            // API not available on the current Play Store app. or Connection couldn't be established.
                            onInstallReferrerResponseError(responseCode);
                            break;
                    }
                    saveInstallReferrerCompleted(true);
                    callback.getReferrerMap(referrerMap, GooglePlayReferrerProperties.this);
                }

                @Override
                public void onInstallReferrerServiceDisconnected() {
                    // Try to restart the connection on the next request to
                    // Google Play by calling the startConnection() method.
                    onInstallReferrerResponseError(InstallReferrerClient.InstallReferrerResponse.SERVICE_DISCONNECTED);
                    saveInstallReferrerCompleted(true);
                    callback.getReferrerMap(referrerMap, GooglePlayReferrerProperties.this);
                }
            });
        } catch (Exception e) {
            AbxLog.w(e, true);
            saveInstallReferrerCompleted(true);
            callback.getReferrerMap(referrerMap, this);
        }
    }

    public void endConnection(){
        referrerClient.endConnection();
        referrerClient = null;
    }

    private void onInstallReferrerResponseOK(HashMap<String, Object> referrerMap, InstallReferrerClient referrerClient) {
        try {
            ReferrerDetails response = referrerClient.getInstallReferrer();
            String referrerUrl = null;
            String clickAdKey = null;
            long referrerClickTime = -1;
            long appInstallTime = -1;
            if (response != null) {
                referrerUrl = response.getInstallReferrer();
                if (!CommonUtils.isNullOrEmpty(referrerUrl)) {
                    try {
                        String abxTid = Uri.parse("http://referrer.ad-brix.com?" + referrerUrl).getQueryParameter("abx_tid");
                        if (!CommonUtils.isNullOrEmpty(abxTid)) {
                            clickAdKey = abxTid;
                        }
                    } catch (Exception e) {
                        AbxLog.e(e, true);
                    }
                }

                long referrerClickTimestampSeconds = response.getReferrerClickTimestampSeconds();
                if (referrerClickTimestampSeconds != 0) {
                    referrerClickTime = referrerClickTimestampSeconds;
                }

                long installBeginTimestampSeconds = response.getInstallBeginTimestampSeconds();
                if (installBeginTimestampSeconds != 0) {
                    appInstallTime = installBeginTimestampSeconds;
                }
            }
            DataRegistry dataRegistry = componentsFactory.createOrGetDataRegistry();
            referrerMap.put(GooglePlayReferrerProperties.MARKET_INSTALL_BTN_CLICKED, referrerClickTime);
            dataRegistry.putDataRegistry(new DataUnit(
                    DataRegistryKey.LONG_MARKET_INSTALL_BTN_CLICKED,
                    referrerClickTime,
                    5,
                    this.getClass().getName(),
                    true));
            referrerMap.put(GooglePlayReferrerProperties.APP_INSTALL_START, appInstallTime);
            dataRegistry.putDataRegistry(new DataUnit(
                    DataRegistryKey.LONG_APP_INSTALL_START,
                    appInstallTime,
                    5,
                    this.getClass().getName(),
                    true));

            int appInstallCompleted;
            try {
                PackageInfo packageInfo = context.getPackageManager().getPackageInfo(context.getPackageName(), 0);
                appInstallCompleted = (int) (packageInfo.firstInstallTime / 1000);
                referrerMap.put(GooglePlayReferrerProperties.APP_INSTALL_COMPLETED, appInstallCompleted);
            } catch (PackageManager.NameNotFoundException e) {
                // Remaster에서는 (int) ( -1 / 1000) 을 사용했다.
                appInstallCompleted = 0;
                referrerMap.put(GooglePlayReferrerProperties.APP_INSTALL_COMPLETED, 0);
            }
            dataRegistry.putDataRegistry(new DataUnit(
                    DataRegistryKey.INT_APP_INSTALL_COMPLETED,
                    appInstallCompleted,
                    5,
                    this.getClass().getName(),
                    true));

            referrerMap.put(GooglePlayReferrerProperties.REFERRER, referrerUrl);
            dataRegistry.putDataRegistry(new DataUnit(
                    DataRegistryKey.STRING_REFERRER,
                    referrerUrl,
                    5,
                    this.getClass().getName(),
                    true));
            int appFirstOpen = (int) (System.currentTimeMillis() / 1000);
            referrerMap.put(GooglePlayReferrerProperties.APP_FIRST_OPEN, appFirstOpen);
            dataRegistry.putDataRegistry(new DataUnit(
                    DataRegistryKey.INT_APP_FIRST_OPEN,
                    appFirstOpen,
                    5,
                    this.getClass().getName(),
                    true));
            referrerMap.put(GooglePlayReferrerProperties.CLICK_ADKEY, clickAdKey);
            dataRegistry.putDataRegistry(new DataUnit(
                    DataRegistryKey.STRING_CLICK_ADKEY,
                    clickAdKey,
                    5,
                    this.getClass().getName(),
                    true));
        } catch (RemoteException e) {
            AbxLog.e(e, true);
        }
    }

    private void onInstallReferrerResponseError(int responseCode) {
        switch (responseCode) {
            case -1:
                AbxLog.d("InstallReferrerResponse : SERVICE_DISCONNECTED", true);
                break;
            case 0:
                AbxLog.d("InstallReferrerResponse : OK", true);
                break;
            case 1:
                AbxLog.d("InstallReferrerResponse : SERVICE_UNAVAILABLE", true);
                break;
            case 2:
                AbxLog.d("InstallReferrerResponse : FEATURE_NOT_SUPPORTED", true);
                break;
            case 3:
                AbxLog.d("InstallReferrerResponse : DEVELOPER_ERROR", true);
                break;
            default:
                break;
        }
    }

    public interface Callback{
        void getReferrerMap(HashMap<String, Object> referrerMap, GooglePlayReferrerProperties googlePlayReferrerProperties);
    }
}
