package io.adbrix.sdk.ui.inappmessage;

import static android.view.View.GONE;
import static android.view.View.VISIBLE;

import android.app.Activity;
import android.content.ActivityNotFoundException;
import android.content.Intent;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.graphics.BlendMode;
import android.graphics.BlendModeColorFilter;
import android.graphics.Color;
import android.graphics.PorterDuff;
import android.graphics.drawable.Drawable;
import android.net.Uri;
import android.os.Build;
import android.view.LayoutInflater;
import android.view.MotionEvent;
import android.view.View;
import android.view.animation.Animation;
import android.widget.FrameLayout;
import android.widget.ImageButton;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.TextView;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

import io.adbrix.sdk.R;
import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.IObserver;
import io.adbrix.sdk.component.InAppMessageClickPostingObservable;
import io.adbrix.sdk.component.TryOptional;
import io.adbrix.sdk.configuration.IABXContextController;
import io.adbrix.sdk.domain.ABXConstants;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.DfnGrowthEventParam;
import io.adbrix.sdk.domain.model.DfnInAppMessage;
import io.adbrix.sdk.domain.model.IAMEnums;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.utils.AnimationUtils;
import io.adbrix.sdk.utils.CommonUtils;


public class StickyBannerMessageViewFactory extends InAppMessageViewFactory {
    public static final long ANIMATION_DURATION_MS = 500;
    public static final String ALIGN_TOP = "top";
    private List<IObserver<LogEventParameter>> observers = new ArrayList<>();
    private InAppMessageViewWrapper dialog;
    private AtomicBoolean isAnimating;

    public StickyBannerMessageViewFactory(InAppMessageDAO inAppMessageDAO, IABXContextController abxContextController) {
        super(inAppMessageDAO, abxContextController);
        isAnimating = new AtomicBoolean(false);
    }

    @Override
    public void createInAppMessage(Activity activity, DfnInAppMessage inAppMessage, boolean isCarried) {
        super.createInAppMessage(activity, inAppMessage, isCarried);
        LayoutInflater inflater = activity.getLayoutInflater();

        InAppMessageView view = (InAppMessageView) inflater.inflate(R.layout.layout_slideup_message, null);

        activity.runOnUiThread(() -> {

            try {
                dialog = new InAppMessageViewWrapper(
                        inAppMessage,
                        new InAppMessageViewListener() {
                            @Override
                            public void onClose(Activity attachedActivity, IAMEnums.CloseType action) {
                                AbxLog.i("inAppMessage onClose: "+action, true);
                                dialog.close(attachedActivity);
                            }
                        });

                boolean isPortrait = activity.getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT;

                setImage(view, isPortrait, inAppMessage, dialog, activity);

                Runnable onClickRunnable = () -> AbxLog.d("StickyBanner : unknown button action", true);

                if (IAMEnums.ActionType.CLOSE.getValue().equals(inAppMessage.getStickBannerClickActionActionType())) {
                    onClickRunnable = () -> {
                        InAppMessageManager.getInstance().close(IAMEnums.CloseType.ON_CLICK_FROM_CLOSE_BUTTON);
                        Map<String, Object> eventParam = TryOptional.of(() -> inAppMessage.growthEventParam)
                                .mapCatching(DfnGrowthEventParam::getJson)
                                .map(CommonUtils::getMapFromJSONObject)
                                .orElseGet(HashMap::new);

                        if (!eventParam.containsKey(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_ID)) {
                            eventParam.put(CoreConstants.IN_APP_MESSAGE_CAMPAIGN_ID_KEY, inAppMessage.getCampaignId());
                        }
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_EVENT_TYPE_ABX_KEY, IAMEnums.EventType.CLICK.getValue());
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_ACTION_ID_ABX_KEY, inAppMessage.getStickBannerClickActionActionID());

                        abxContextController.logEvent(new LogEventParameter(
                                CoreConstants.GROUP_ABX,
                                CoreConstants.EVENT_IN_APP_MESSAGE,
                                CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                                0,
                                0
                        ));

                        InAppMessageClickPostingObservable.getInstance().postInAppMessageClickToMainThread(
                                inAppMessage.getStickBannerClickActionActionID(),
                                inAppMessage.getStickBannerClickActionActionType(),
                                inAppMessage.getStickBannerClickOptionActionArg(),
                                true
                        );
                    };
                } else if (
                        IAMEnums.ActionType.DEEPLINK_AND_CLOSE.getValue().equals(inAppMessage.getStickBannerClickActionActionType())
                                || IAMEnums.ActionType.WEBLINK.getValue().equals(inAppMessage.getStickBannerClickActionActionType())
                                || IAMEnums.ActionType.WEBLINK_AND_CLOSE.getValue().equals(inAppMessage.getStickBannerClickActionActionType())) {
                    onClickRunnable = () -> {
                        String url = inAppMessage.getStickBannerClickOptionActionArg();
                        Intent intent = new Intent(Intent.ACTION_VIEW, Uri.parse(url));

                        Map<String, Object> eventParam = TryOptional.of(() -> inAppMessage.growthEventParam)
                                .mapCatching(DfnGrowthEventParam::getJson)
                                .map(CommonUtils::getMapFromJSONObject)
                                .orElseGet(HashMap::new);

                        if (!eventParam.containsKey(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_ID)) {
                            eventParam.put(CoreConstants.IN_APP_MESSAGE_CAMPAIGN_ID_KEY, inAppMessage.getCampaignId());
                        }
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_EVENT_TYPE_ABX_KEY, IAMEnums.EventType.CLICK.getValue());
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_ACTION_ID_ABX_KEY, inAppMessage.getStickBannerClickActionActionID());

                        abxContextController.logEvent(new LogEventParameter(
                                CoreConstants.GROUP_ABX,
                                CoreConstants.EVENT_IN_APP_MESSAGE,
                                CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                                0,
                                0
                        ));
                        InAppMessageManager.getInstance().close(IAMEnums.CloseType.OPEN_DEEPLINK);

                        try {
                            view.getContext().startActivity(intent);
                        } catch (ActivityNotFoundException exception) {
                            AbxLog.d("ModalMessageViewFactory: ActivityNotFoundException, button action url was " + url, true);
                        }

                        InAppMessageClickPostingObservable.getInstance().postInAppMessageClickToMainThread(
                                inAppMessage.getStickBannerClickActionActionID(),
                                inAppMessage.getStickBannerClickActionActionType(),
                                inAppMessage.getStickBannerClickOptionActionArg(),
                                true
                        );
                    };
                } else {
                    AbxLog.d("StickyBannerMessageViewFactory: unknown button action '" + inAppMessage.getStickBannerClickActionActionType() + "'", true);
                }
                setMessage(view, inAppMessage);
                setCloseButton(view, inAppMessage, activity);

                if (ALIGN_TOP.equals(inAppMessage.getStickyBannerAlign()))
                    setOnTop(activity, view, dialog, isCarried, onClickRunnable);
                else
                    setOnBottom(activity, view, dialog, isCarried, onClickRunnable);
                dialog.setContentView(view);
                dialog.setInAppMessageViewListener();
                dialog.open(activity);

                Map<String, Object> eventParam = TryOptional.of(() -> inAppMessage.growthEventParam)
                        .mapCatching(DfnGrowthEventParam::getJson)
                        .map(CommonUtils::getMapFromJSONObject)
                        .orElseGet(HashMap::new);

                if (!eventParam.containsKey(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_ID)) {
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_CAMPAIGN_ID_KEY, inAppMessage.getCampaignId());
                }
                eventParam.put(CoreConstants.IN_APP_MESSAGE_EVENT_TYPE_ABX_KEY, IAMEnums.EventType.IMPRESSION.getValue());

                if (isPortrait) {
                    if (inAppMessage.getPortraitImage() != null) {
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_URL_ABX_KEY, inAppMessage.getPortraitImageURL());
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_IS_IMAGE_DOWNLOADED_ABX_KEY, inAppMessage.getsIsPortraitImageDownloaded());
                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                            eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getPortraitImage().getAllocationByteCount());
                        } else {
                            eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getPortraitImage().getByteCount());
                        }
                    }
                } else {
                    if (inAppMessage.getLandScapeImage() != null) {
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_URL_ABX_KEY, inAppMessage.getLandscapeImageURL());
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_IS_IMAGE_DOWNLOADED_ABX_KEY, inAppMessage.getsIsLandscapeImageDownloaded());
                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                            eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getLandScapeImage().getAllocationByteCount());
                        } else {
                            eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getLandScapeImage().getByteCount());
                        }
                    }
                }

                abxContextController.logEvent(new LogEventParameter(
                        CoreConstants.GROUP_ABX,
                        CoreConstants.EVENT_IN_APP_MESSAGE,
                        CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                        0,
                        0
                ));
            } catch (NullPointerException nullPointerException) {
                AbxLog.d("CreateInAppMessage : nullPointerException\n " + nullPointerException, true);
            }
        });
    }

    private void setImage(View view, boolean isPortrait, DfnInAppMessage inAppMessage, InAppMessageViewWrapper dialog, Activity activity) {
        Bitmap bitmapImage = isPortrait ? inAppMessage.getPortraitImage() : inAppMessage.getLandScapeImage();

        ImageView imageView = view.findViewById(R.id.com_adbrix_inappmessage_slideup_imageview);
        FrameLayout progressBarLayout = view.findViewById(R.id.com_adbirx_inappmessage_slideup_progressbar_layout);
        FrameLayout defaultImageViewLayout = view.findViewById(R.id.com_adbrix_inappmessage_slideup_default_imageview_layout);

        progressBarLayout.setVisibility(GONE);
        imageView.setVisibility(VISIBLE);
        defaultImageViewLayout.setVisibility(GONE);

        imageView.setImageBitmap(bitmapImage);
    }

    private void setOnTop(Activity activity, InAppMessageView view, InAppMessageViewWrapper dialog, boolean isCarried, Runnable onClickRunnable) {
        if(!isCarried){
            view.clearAnimation();
            Animation openingAnimation = AnimationUtils.createVerticalAnimation(-1f, 0f, ANIMATION_DURATION_MS, false);
            openingAnimation.setAnimationListener(createAnimationListener());
            view.setAnimation(openingAnimation);
        }
        view.setOnTouchListener(new SwipeListener(activity) {
            @Override
            public void onSwipeLeft(MotionEvent motionEvent) {
                if(isAnimating.get()){
                    AbxLog.w("inAppMessage is Animating..", true);
                    return;
                }
                InAppMessageManager.getInstance().close(IAMEnums.CloseType.CLOSE_BY_SWIPE_LEFT);
            }

            @Override
            public void onSwipeRight(MotionEvent motionEvent) {
                if(isAnimating.get()){
                    AbxLog.w("inAppMessage is Animating..", true);
                    return;
                }
                InAppMessageManager.getInstance().close(IAMEnums.CloseType.CLOSE_BY_SWIPE_RIGHT);
            }

            @Override
            public void onClick() {
                super.onClick();
                if(isAnimating.get()){
                    AbxLog.w("inAppMessage is Animating..", true);
                    return;
                }
                InAppMessageManager.getInstance().close(IAMEnums.CloseType.ON_CLICK_FROM_FRAME);
                if (onClickRunnable != null)
                    onClickRunnable.run();
            }
        });

//        WindowManager.LayoutParams wlp = window.getAttributes();
//
//        wlp.gravity = Gravity.TOP;
//        wlp.flags &= ~WindowManager.LayoutParams.FLAG_DIM_BEHIND;
//
//        window.setFlags(WindowManager.LayoutParams.FLAG_NOT_TOUCH_MODAL,
//                WindowManager.LayoutParams.FLAG_NOT_TOUCH_MODAL);
//        window.setLayout(WindowManager.LayoutParams.MATCH_PARENT, WindowManager.LayoutParams.WRAP_CONTENT);

    }

    private void setOnBottom(Activity activity, InAppMessageView view, InAppMessageViewWrapper dialog, boolean isCarried, Runnable onClickRunnable) {
        if(!isCarried){
            view.clearAnimation();
            Animation openingAnimation = AnimationUtils.createVerticalAnimation(1f, 0f, ANIMATION_DURATION_MS, false);
            openingAnimation.setAnimationListener(createAnimationListener());
            view.setAnimation(openingAnimation);
        }
        view.setOnTouchListener(new SwipeListener(activity) {
            @Override
            public void onSwipeLeft(MotionEvent motionEvent) {
                if(isAnimating.get()){
                    AbxLog.w("inAppMessage is Animating..", true);
                    return;
                }
                InAppMessageManager.getInstance().close(IAMEnums.CloseType.CLOSE_BY_SWIPE_LEFT);
            }

            @Override
            public void onSwipeRight(MotionEvent motionEvent) {
                if(isAnimating.get()){
                    AbxLog.w("inAppMessage is Animating..", true);
                    return;
                }
                InAppMessageManager.getInstance().close(IAMEnums.CloseType.CLOSE_BY_SWIPE_RIGHT);
            }

            @Override
            public void onClick() {
                super.onClick();
                if(isAnimating.get()){
                    AbxLog.w("inAppMessage is Animating..", true);
                    return;
                }
                InAppMessageManager.getInstance().close(IAMEnums.CloseType.ON_CLICK_FROM_FRAME);
                if (onClickRunnable != null)
                    onClickRunnable.run();
            }
        });

//        WindowManager.LayoutParams wlp = window.getAttributes();
//
//        wlp.gravity = Gravity.BOTTOM;
//        wlp.flags &= ~WindowManager.LayoutParams.FLAG_DIM_BEHIND;
//        window.setFlags(WindowManager.LayoutParams.FLAG_NOT_TOUCH_MODAL,
//                WindowManager.LayoutParams.FLAG_NOT_TOUCH_MODAL);
//        window.setLayout(WindowManager.LayoutParams.MATCH_PARENT, WindowManager.LayoutParams.WRAP_CONTENT);

    }

    private void setMessage(View view, DfnInAppMessage inAppMessage) {
        TextView textView = view.findViewById(R.id.com_adbrix_inappmessage_slideup_textview);
        textView.setText(inAppMessage.getBodyText());
        textView.setTextColor(Color.parseColor(inAppMessage.getBodyTextColor()));

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            if (IAMEnums.TextAlign.CENTER.getValue().equals(inAppMessage.getTextAlign())) {
                textView.setTextAlignment(View.TEXT_ALIGNMENT_CENTER);
            } else if (IAMEnums.TextAlign.RIGHT.getValue().equals(inAppMessage.getTextAlign())) {
                textView.setTextAlignment(View.TEXT_ALIGNMENT_VIEW_END);
            } else if (IAMEnums.TextAlign.LEFT.getValue().equals(inAppMessage.getTextAlign())) {
                textView.setTextAlignment(View.TEXT_ALIGNMENT_VIEW_START);
            }
        }


        RelativeLayout frame = view.findViewById(R.id.com_adbrix_inappmessage_slideup_innerframe);
        frame.setBackgroundColor(Color.parseColor(inAppMessage.getBgColor()));
    }

    /**
     * 2023.01.13 bobos
     * 인앱메시지 DB closeType 값이 없으면 표시 X 있으면 O
     * @param view
     */
    private void setCloseButton(View view, DfnInAppMessage inAppMessage, Activity activity){
        RelativeLayout cancelButtonLayout = view.findViewById(R.id.com_adbrix_inappmessage_slideup_close_button_layout);
        ImageButton cancelButton = view.findViewById(R.id.com_adbrix_inappmessage_slideup_close_button);
        if(!inAppMessage.hasCloseButton()){
            cancelButton.setVisibility(GONE);
            cancelButtonLayout.setVisibility(GONE);
            RelativeLayout innerFrameLayout = view.findViewById(R.id.com_adbrix_inappmessage_slideup_innerframe);
            int layoutPadding = (int)activity.getResources().getDimension(R.dimen.InAppMessage_SlideUp_InnerFrame_Paddings);
            innerFrameLayout.setPadding(0,0,layoutPadding,0);
            return;
        }
        int closedButtonColor = Color.parseColor(inAppMessage.getDefaultCloseButtonColor());
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.Q) {
            cancelButton.setColorFilter(closedButtonColor, PorterDuff.Mode.SRC_IN);
        } else {
            cancelButton.setColorFilter(new BlendModeColorFilter(closedButtonColor, BlendMode.SRC_IN));
        }

        int closedButtonBgColor = Color.parseColor(inAppMessage.getDefaultCloseButtonBgColor());
        Drawable cancelButtonDrawable = view.getResources().getDrawable(R.drawable.ic_com_adbrix_inappmessage_cancel_button_border);

        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.Q) {
            cancelButtonDrawable.setColorFilter(closedButtonBgColor, PorterDuff.Mode.SRC_IN);
        } else {
            cancelButtonDrawable.setColorFilter(new BlendModeColorFilter(closedButtonBgColor, BlendMode.SRC_IN));
        }

        cancelButton.setBackgroundDrawable(cancelButtonDrawable);
        cancelButton.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                if(isAnimating.get()){
                    AbxLog.w("inAppMessage is Animating..", true);
                    return;
                }
                InAppMessageManager.getInstance().close(IAMEnums.CloseType.ON_CLICK_FROM_CLOSE_BUTTON);
            }
        });
        cancelButtonLayout.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View view) {
                if(isAnimating.get()){
                    AbxLog.w("inAppMessage is Animating..", true);
                    return;
                }
                InAppMessageManager.getInstance().close(IAMEnums.CloseType.ON_CLICK_FROM_CLOSE_BUTTON);
            }
        });
    }

    private Animation.AnimationListener createAnimationListener(){
        return new Animation.AnimationListener() {
            @Override
            public void onAnimationStart(Animation animation) {
                isAnimating.set(true);
            }

            @Override
            public void onAnimationEnd(Animation animation) {
                if(CommonUtils.isNull(dialog)){
                    AbxLog.d("onAnimationEnd() dialog is null", true);
                    return;
                }
//                AbxLog.d("onAnimationEnd() clearAnimation", true);
                dialog.clearInAppMessageViewAnimation();
                isAnimating.set(false);
            }

            @Override
            public void onAnimationRepeat(Animation animation) {

            }
        };
    }

    @Override
    public void dismissInAppMessage(Activity activity, IAMEnums.CloseType closeType) {
        super.dismissInAppMessage(activity, closeType);
        if (dialog != null){
            dialog.getInAppMessageViewListener().onClose(activity, closeType);
        }
    }
}
