package io.adbrix.sdk.ui.inappmessage;

import static android.view.View.GONE;
import static android.view.View.VISIBLE;

import android.app.Activity;
import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.graphics.BlendMode;
import android.graphics.BlendModeColorFilter;
import android.graphics.Color;
import android.graphics.Point;
import android.graphics.PorterDuff;
import android.graphics.drawable.Drawable;
import android.graphics.drawable.GradientDrawable;
import android.os.Build;
import android.view.Display;
import android.view.Gravity;
import android.view.MotionEvent;
import android.view.View;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.FrameLayout;
import android.widget.ImageButton;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.ProgressBar;
import android.widget.RelativeLayout;
import android.widget.TextView;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import io.adbrix.sdk.R;
import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.IObserver;
import io.adbrix.sdk.component.TryOptional;
import io.adbrix.sdk.configuration.AbxFacade;
import io.adbrix.sdk.configuration.IABXContextController;
import io.adbrix.sdk.domain.ABXConstants;
import io.adbrix.sdk.domain.CompatConstants;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.DfnGrowthEventParam;
import io.adbrix.sdk.domain.model.DfnInAppMessage;
import io.adbrix.sdk.domain.model.IAMEnums;
import io.adbrix.sdk.domain.model.LogEventParameter;
import io.adbrix.sdk.utils.CommonUtils;

public class ModalMessageViewFactory extends InAppMessageViewFactory {
    private List<IObserver<LogEventParameter>> observers = new ArrayList<>();
    private InAppMessageViewWrapper dialog;

    public ModalMessageViewFactory(InAppMessageDAO inAppMessageDAO, IABXContextController abxContextController) {
        super(inAppMessageDAO, abxContextController);
    }

    public static int getTextViewHeight(TextView textView) {
        WindowManager wm =
                (WindowManager) textView.getContext().getSystemService(Context.WINDOW_SERVICE);
        Display display = wm.getDefaultDisplay();

        int deviceWidth;

        Point size = new Point();
        display.getSize(size);
        deviceWidth = (int) (size.x * 0.85);

        int widthMeasureSpec = View.MeasureSpec.makeMeasureSpec(deviceWidth, View.MeasureSpec.AT_MOST);
        int heightMeasureSpec = View.MeasureSpec.makeMeasureSpec(0, View.MeasureSpec.UNSPECIFIED);
        textView.measure(widthMeasureSpec, heightMeasureSpec);
        return textView.getMeasuredHeight();
    }

    @Override
    public void createInAppMessage(Activity activity, DfnInAppMessage inAppMessage, boolean isCarried) {
        super.createInAppMessage(activity, inAppMessage, isCarried);
        activity.runOnUiThread(() -> {

            try {
                InAppMessageView view;
                dialog = new InAppMessageViewWrapper(
                        inAppMessage,
                        new InAppMessageViewListener() {
                            @Override
                            public void onClose(Activity attachedActivity, IAMEnums.CloseType action) {
                                AbxLog.i("inAppMessage onClose: "+action, true);
                                dialog.close(attachedActivity);
                            }
                        });

                boolean isPortrait = activity.getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT;

                if (IAMEnums.LayoutType.IMAGE_AND_TEXT.getValue().equals(inAppMessage.getLayout())) {
                    view = (InAppMessageView) activity.getLayoutInflater().inflate(R.layout.layout_modal_message_image_and_text, null);

                    setImage(view, isPortrait, inAppMessage, dialog, activity);

                    setText(view, inAppMessage);
                } else if (IAMEnums.LayoutType.IMAGE.getValue().equals(inAppMessage.getLayout())) {
                    view = (InAppMessageView) activity.getLayoutInflater().inflate(R.layout.layout_modal_message_image, null);

                    setImage(view, isPortrait, inAppMessage, dialog, activity);
                } else if (IAMEnums.LayoutType.TEXT.getValue().equals(inAppMessage.getLayout())) {
                    view = (InAppMessageView) activity.getLayoutInflater().inflate(R.layout.layout_modal_message_text, null);

                    setText(view, inAppMessage);
                } else {
                    AbxLog.d("createInAppMessage: unknown modal layout type", true);
                    return;
                }
                setOverlayColor(view, inAppMessage); // common


                //common
                switch (inAppMessage.getNumberOfButtons()) {
                    case 0:
                        setNoButton(view);
                        break;
                    case 1:
                        setSingleButton(view, dialog, inAppMessage, activity);
                        break;
                    case 2:
                        setDualButton(view, dialog, inAppMessage, activity);
                        break;
                    default:
                        setNoButton(view);
                        AbxLog.e("CreateInAppMessage : Invalid button number!", true);
                }

                //common
                setCancelButton(view, activity, inAppMessage);

                setModalSize(view, isPortrait, activity, inAppMessage);
                dialog.setContentView(view);
                dialog.setInAppMessageViewListener();

                //common
                open(activity, dialog, inAppMessage, isPortrait);
            } catch (Exception e) {
                AbxLog.e("CreateInAppMessage : ",e, true);
            }
        });
    }

    public void setModalSize(View view, boolean isPortrait, Activity activity, DfnInAppMessage inAppMessage) {
        RelativeLayout modal = view.findViewById(R.id.com_adbrix_inappmessage_modal);

        if (isPortrait) {
            int modalWidth = (int) (activity.getResources().getDisplayMetrics().widthPixels * 0.85);
            modal.getLayoutParams().width = modalWidth;

            if (IAMEnums.LayoutType.IMAGE_AND_TEXT.getValue().equals(inAppMessage.getLayout())
                    || IAMEnums.LayoutType.TEXT.getValue().equals(inAppMessage.getLayout())) {
                modal.getLayoutParams().height = modalWidth;

                TextView titleText = view.findViewById(R.id.com_adbrix_inappmessage_modal_header_text);
                TextView bodyText = view.findViewById(R.id.com_adbrix_inappmessage_modal_message_text);

                int titleHeight = 0;
                int bodyHegith = 0;

                try {
                    titleHeight = getTextViewHeight(titleText);
                    bodyHegith = getTextViewHeight(bodyText);
                } catch (NullPointerException nullPointerException) {
                    AbxLog.d("InAppMessage setText : " + nullPointerException, true);
                }

                if (titleHeight + bodyHegith > modalWidth * 0.7
                        && IAMEnums.LayoutType.TEXT.getValue().equals(inAppMessage.getLayout()))
                    modal.getLayoutParams().height = (int) (modalWidth * 1.5);
                else if (titleHeight + bodyHegith > modalWidth * 0.6
                        && IAMEnums.LayoutType.IMAGE_AND_TEXT.getValue().equals(inAppMessage.getLayout()))
                    modal.getLayoutParams().height = (int) (modalWidth * 1.5);
            }
        } else {
            int modalHeight = (int) (activity.getResources().getDisplayMetrics().heightPixels * 0.85);

            modal.getLayoutParams().height = modalHeight;

            if (IAMEnums.LayoutType.IMAGE_AND_TEXT.getValue().equals(inAppMessage.getLayout())
                    || IAMEnums.LayoutType.TEXT.getValue().equals(inAppMessage.getLayout())) {
                modal.getLayoutParams().width = modalHeight;
            } else if (inAppMessage.getLandScapeImage() != null) {
                modal.getLayoutParams().width = (int) (((float) inAppMessage.getLandScapeImage().getWidth()) / ((float) inAppMessage.getLandScapeImage().getHeight()) * (float) modalHeight);
            } else {
                modal.getLayoutParams().width = modalHeight;
            }
        }
    }

    public void setOverlayColor(View view, DfnInAppMessage inAppMessage) {
        RelativeLayout modalContainer = view.findViewById(R.id.com_adbrix_inappmessage_modal_container);
        modalContainer.setBackgroundColor(Color.parseColor(inAppMessage.getOverlayColor()));
        modalContainer.setOnTouchListener(new View.OnTouchListener() {
            @Override
            public boolean onTouch(View view, MotionEvent motionEvent) {
                return true;
            }
        });
    }

    public void setImage(View view, boolean isPortrait, DfnInAppMessage inAppMessage, InAppMessageViewWrapper dialog, Activity activity) {
        Bitmap bitmapImage = isPortrait ? inAppMessage.getPortraitImage() : inAppMessage.getLandScapeImage();
        ImageView imageView = view.findViewById(R.id.com_adbrix_inappmessage_modal_imageview);
        ProgressBar progressBar = view.findViewById(R.id.com_adbirx_inappmessage_modal_progressbar);
        ImageView defaultImageView = view.findViewById(R.id.com_adbrix_inappmessage_modal_default_imageview);
        LinearLayout defaultImageViewLayout = view.findViewById(R.id.com_adbrix_inappmessage_modal_default_imageview_layout);
        FrameLayout progressBarLayout = view.findViewById(R.id.com_adbirx_inappmessage_modal_progressbar_layout);

        imageView.setVisibility(VISIBLE);
        progressBar.setVisibility(GONE);
        progressBarLayout.setVisibility(GONE);
        defaultImageViewLayout.setVisibility(GONE);

        imageView.setScaleType(ImageView.ScaleType.CENTER_CROP);
        imageView.setImageBitmap(bitmapImage);

        if (inAppMessage.getImageClickActionId() != null) {
            imageView.setOnClickListener(new IInAppMessageViewFactory.InAppMessageOnClickListener(
                    inAppMessage.getCampaignId(),
                    inAppMessage.getImageClickActionId(),
                    inAppMessage.getImageClickActionType(),
                    inAppMessage.getImageClickActionArg(),
                    dialog.getInAppMessageViewListener(),
                    inAppMessageDAO,
                    () -> {
                        Map<String, Object> eventParam = TryOptional.of(() -> inAppMessage.growthEventParam)
                                .mapCatching(DfnGrowthEventParam::getJson)
                                .map(CommonUtils::getMapFromJSONObject)
                                .orElseGet(HashMap::new);

                        if (!eventParam.containsKey(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_ID)) {
                            eventParam.put(CoreConstants.IN_APP_MESSAGE_CAMPAIGN_ID_KEY, inAppMessage.getCampaignId());
                        }

                        eventParam.put(CoreConstants.IN_APP_MESSAGE_ACTION_ID_ABX_KEY, inAppMessage.getImageClickActionId());
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_EVENT_TYPE_ABX_KEY, IAMEnums.EventType.CLICK.getValue());

                        abxContextController.logEvent(new LogEventParameter(
                                CoreConstants.GROUP_ABX,
                                CoreConstants.EVENT_IN_APP_MESSAGE,
                                CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                                0,
                                0
                        ));
                        if (
                                IAMEnums.ActionType.CLOSE.getValue().equals(inAppMessage.getImageClickActionType())
                                        || IAMEnums.ActionType.WEBLINK_AND_CLOSE.getValue().equals(inAppMessage.getImageClickActionType())
                                        || IAMEnums.ActionType.DEEPLINK_AND_CLOSE.getValue().equals(inAppMessage.getImageClickActionType())
                                        || IAMEnums.ActionType.DONT_SHOW_ME_TODAY_AND_CLOSE.getValue().equals(inAppMessage.getImageClickActionType())
                        ) {

                        }
                    }
            ));
        }
    }

    public void setText(View view, DfnInAppMessage inAppMessage) {
        RelativeLayout textAndButtonLinearLayout = view.findViewById(R.id.com_adbirx_inappmessage_modal_text_and_button_layout);
        TextView titleText = view.findViewById(R.id.com_adbrix_inappmessage_modal_header_text);
        TextView bodyText = view.findViewById(R.id.com_adbrix_inappmessage_modal_message_text);

        titleText.setText(inAppMessage.getTitleText());
        titleText.setVisibility(VISIBLE);
        titleText.setTextColor(Color.parseColor(inAppMessage.getTitleTextColor()));

        bodyText.setText(inAppMessage.getBodyText());
        bodyText.setVisibility(VISIBLE);
        bodyText.setTextColor(Color.parseColor(inAppMessage.getBodyTextColor()));

        if (IAMEnums.TextAlign.CENTER.getValue().equals(inAppMessage.getTextAlign())) {
            titleText.setGravity(Gravity.CENTER);
            bodyText.setGravity(Gravity.CENTER);
        } else if (IAMEnums.TextAlign.RIGHT.getValue().equals(inAppMessage.getTextAlign())) {
            titleText.setGravity(Gravity.RIGHT);
            bodyText.setGravity(Gravity.RIGHT);
        } else if (IAMEnums.TextAlign.LEFT.getValue().equals(inAppMessage.getTextAlign())) {
            titleText.setGravity(Gravity.LEFT);
            bodyText.setGravity(Gravity.LEFT);
        }

        textAndButtonLinearLayout.setBackgroundColor(Color.parseColor(inAppMessage.getBgColor()));
    }

    private void setNoButton(View view) {
        LinearLayout singleButtonLayout = view.findViewById(R.id.com_adbrix_inappmessage_modal_button_layout_single);
        LinearLayout dualButtonLayout = view.findViewById(R.id.com_adbrix_inappmessage_modal_button_layout_dual);

        singleButtonLayout.setVisibility(GONE);
        dualButtonLayout.setVisibility(GONE);
    }

    public void setSingleButton(View view, InAppMessageViewWrapper dialog, DfnInAppMessage inAppMessage, Activity activity) {
        LinearLayout singleButtonLayout = view.findViewById(R.id.com_adbrix_inappmessage_modal_button_layout_single);
        LinearLayout dualButtonLayout = view.findViewById(R.id.com_adbrix_inappmessage_modal_button_layout_dual);

        singleButtonLayout.setVisibility(VISIBLE);
        dualButtonLayout.setVisibility(GONE);

        Button singleButton = view.findViewById(R.id.com_adbrix_inappmessage_modal_button_single);
        setButton(dialog, singleButton, inAppMessage, 0, activity);
    }

    public void setDualButton(View view, InAppMessageViewWrapper dialog, DfnInAppMessage inAppMessage, Activity activity) {
        LinearLayout singleButtonLayout = view.findViewById(R.id.com_adbrix_inappmessage_modal_button_layout_single);
        LinearLayout dualButtonLayout = view.findViewById(R.id.com_adbrix_inappmessage_modal_button_layout_dual);

        singleButtonLayout.setVisibility(GONE);
        dualButtonLayout.setVisibility(VISIBLE);

        Button buttonFirst = view.findViewById(R.id.com_adbrix_inappmessage_modal_button_dual_first);
        Button buttonSecond = view.findViewById(R.id.com_adbrix_inappmessage_modal_button_dual_second);

        setButton(dialog, buttonFirst, inAppMessage, 0, activity);
        setButton(dialog, buttonSecond, inAppMessage, 1, activity);
    }

    public void setButton(InAppMessageViewWrapper dialog, Button button, DfnInAppMessage inAppMessage, int index, Activity activity) {
        //text
        button.setText(inAppMessage.getButtonText(index));
        button.setTextColor(Color.parseColor(inAppMessage.getButtonTextColor(index)));

        //background and border
        GradientDrawable buttonBackground = new GradientDrawable();
        buttonBackground.setShape(GradientDrawable.RECTANGLE);
        buttonBackground.setStroke(5, Color.parseColor(inAppMessage.getButtonBoarderColor(index)));
        buttonBackground.setColor(Color.parseColor(inAppMessage.getButtonBackgroundColor(index)));

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
            button.setBackground(buttonBackground);
        } else {
            button.setBackgroundDrawable(buttonBackground);
        }

        //action
        button.setOnClickListener(new IInAppMessageViewFactory.InAppMessageOnClickListener(
                inAppMessage.getCampaignId(),
                inAppMessage.getButtonActionID(index),
                inAppMessage.getButtonActionType(index),
                inAppMessage.getButtonActionArgument(index),
                dialog.getInAppMessageViewListener(),
                inAppMessageDAO,
                () -> {
                    Map<String, Object> eventParam = TryOptional.of(() -> inAppMessage.growthEventParam)
                            .mapCatching(DfnGrowthEventParam::getJson)
                            .map(CommonUtils::getMapFromJSONObject)
                            .orElseGet(HashMap::new);

                    if (!eventParam.containsKey(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_ID)) {
                        eventParam.put(CoreConstants.IN_APP_MESSAGE_CAMPAIGN_ID_KEY, inAppMessage.getCampaignId());
                    }
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_EVENT_TYPE_ABX_KEY, IAMEnums.EventType.CLICK.getValue());
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_ACTION_ID_ABX_KEY, inAppMessage.getButtonActionID(index));

                    abxContextController.logEvent(new LogEventParameter(
                            CoreConstants.GROUP_ABX,
                            CoreConstants.EVENT_IN_APP_MESSAGE,
                            CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                            0,
                            0
                    ));
                    if (
                            IAMEnums.ActionType.CLOSE.getValue().equals(inAppMessage.getButtonActionType(index))
                                    || IAMEnums.ActionType.WEBLINK_AND_CLOSE.getValue().equals(inAppMessage.getButtonActionType(index))
                                    || IAMEnums.ActionType.DEEPLINK_AND_CLOSE.getValue().equals(inAppMessage.getButtonActionType(index))
                                    || IAMEnums.ActionType.DONT_SHOW_ME_TODAY_AND_CLOSE.getValue().equals(inAppMessage.getButtonActionType(index))
                    ) {

                    }
                }
        ));
    }

    public void setCancelButton(View view, Activity activity, DfnInAppMessage inAppMessage) {
        ImageButton cancelButton = view.findViewById(R.id.com_adbrix_inappmessage_modal_close_button);

        cancelButton.setOnClickListener(v -> {
            InAppMessageManager.getInstance().close(IAMEnums.CloseType.ON_CLICK_FROM_CLOSE_BUTTON);
        });

        int closedButtonColor = Color.parseColor(inAppMessage.getDefaultCloseButtonColor());
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.Q) {
            cancelButton.setColorFilter(closedButtonColor, PorterDuff.Mode.SRC_IN);
        } else {
            cancelButton.setColorFilter(new BlendModeColorFilter(closedButtonColor, BlendMode.SRC_IN));
        }

        int closedButtonBgColor = Color.parseColor(inAppMessage.getDefaultCloseButtonBgColor());
        Drawable cancelButtonDrawable = view.getResources().getDrawable(R.drawable.ic_com_adbrix_inappmessage_cancel_button_border);

        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.Q) {
            cancelButtonDrawable.setColorFilter(closedButtonBgColor, PorterDuff.Mode.SRC_IN);
        } else {
            cancelButtonDrawable.setColorFilter(new BlendModeColorFilter(closedButtonBgColor, BlendMode.SRC_IN));
        }

        cancelButton.setBackgroundDrawable(cancelButtonDrawable);
    }

    public void open(Activity activity, InAppMessageViewWrapper dialog, DfnInAppMessage inAppMessage, boolean isPortrait) {
        dialog.open(activity);

        Map<String, Object> eventParam = TryOptional.of(() -> inAppMessage.growthEventParam)
                .mapCatching(DfnGrowthEventParam::getJson)
                .map(CommonUtils::getMapFromJSONObject)
                .orElseGet(HashMap::new);

        if (!eventParam.containsKey(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_ID)) {
            eventParam.put(CoreConstants.IN_APP_MESSAGE_CAMPAIGN_ID_KEY, inAppMessage.getCampaignId());
        }
        eventParam.put(CoreConstants.IN_APP_MESSAGE_EVENT_TYPE_ABX_KEY, IAMEnums.EventType.IMPRESSION.getValue());

        if (!inAppMessage.getLayout().equals(IAMEnums.LayoutType.TEXT.getValue())) {
            if (isPortrait && inAppMessage.getPortraitImage() != null) {
                eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_URL_ABX_KEY, inAppMessage.getPortraitImageURL());
                eventParam.put(CoreConstants.IN_APP_MESSAGE_IS_IMAGE_DOWNLOADED_ABX_KEY, inAppMessage.getsIsPortraitImageDownloaded());
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getPortraitImage().getAllocationByteCount());
                } else {
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getPortraitImage().getByteCount());
                }
            } else if (!isPortrait && inAppMessage.getLandScapeImage() != null) {
                eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_URL_ABX_KEY, inAppMessage.getLandscapeImageURL());
                eventParam.put(CoreConstants.IN_APP_MESSAGE_IS_IMAGE_DOWNLOADED_ABX_KEY, inAppMessage.getsIsLandscapeImageDownloaded());
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getLandScapeImage().getAllocationByteCount());
                } else {
                    eventParam.put(CoreConstants.IN_APP_MESSAGE_IMAGE_SIZE_ABX_KEY, inAppMessage.getLandScapeImage().getByteCount());
                }
            }
        }

        abxContextController.logEvent(new LogEventParameter(
                CoreConstants.GROUP_ABX,
                CoreConstants.EVENT_IN_APP_MESSAGE,
                CommonUtils.parseValueWithDataType(eventParam, CommonUtils.FixType.PREFIX),
                0,
                0
        ));
    }

    @Override
    public void dismissInAppMessage(Activity activity, IAMEnums.CloseType closeType) {
        super.dismissInAppMessage(activity, closeType);
        if (dialog != null){
            dialog.getInAppMessageViewListener().onClose(activity, closeType);
        }
    }
}
