package io.adbrix.sdk.component.migration_v1;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.Arrays;
import java.util.Iterator;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.EventPackageContainer;
import io.adbrix.sdk.component.UserPropertyManager;
import io.adbrix.sdk.data.RemoteConfigProvider;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.model.EventPackage;
import io.adbrix.sdk.domain.model.UserPropertyCommand;
import io.adbrix.sdk.domain.model.UserPropertyModel;
import io.adbrix.sdk.utils.CommonUtils;

public class V1MigrationManager {
    private static final String ADID = "abx_generated_device_id";
    private static final String UUID = "uuid";
    private static final String LAST_FIRSTOPEN_ID = "last_firstopen_id";
    private static final String LAST_DEEPLINK_ID = "last_deeplink_id";
    private static final String LAST_OPEN_ID = "last_open_id";
    private static final String DAILY_FIRST_OPEN_PREV_DATE = "daily_first_open_prev_date";
    private static final String REGISTRATION_ID = "registration_id";
    private static final String PUSH_ENABLE = "push_enable";
    private static final String PUSH_ENABLE_OS = "push_enable_os";
    private static final String USER_PROFILE_SNAPSHOT_ID = "user_snapshot_id";
    private static final String USER_PROFILE = "abx:userProfile";
    private static final String USER_ID = "user_id";
    private UserPropertyManager userPropertyManager;
    public IV1DatabaseOpenHelper v1DatabaseOpenHelper;
    private EventPackageContainer eventPackageContainer;
    private DataRegistry dataRegistry;
    private RemoteConfigProvider remoteConfigHandler;

    public V1MigrationManager(
            DataRegistry dataRegistry,
            UserPropertyManager userPropertyManager,
            IV1DatabaseOpenHelper v1DatabaseOpenHelper,
            EventPackageContainer eventPackageContainer,
            RemoteConfigProvider remoteConfigHandler
    ) {
        this.dataRegistry = dataRegistry;
        this.userPropertyManager = userPropertyManager;
        this.v1DatabaseOpenHelper = v1DatabaseOpenHelper;
        this.eventPackageContainer = eventPackageContainer;
        this.remoteConfigHandler = remoteConfigHandler;
    }

    private boolean isMigrated() {
        return dataRegistry.safeGetBoolean(DataRegistryKey.BOOLEAN_IS_MIGRATED, false);
    }

    public boolean isTableDetected() {
        return v1DatabaseOpenHelper.isTableDetected();
    }

    public boolean isMigrationCompleted() {
        if (!isTableDetected())
            return false;

        if (!isMigrated()) {
            AbxLog.d("Migration V1 to V2 is available. Start migrating!", true);

            String adId = v1DatabaseOpenHelper.getValue(ADID);
            String uuid = v1DatabaseOpenHelper.getValue(UUID);
            String lastFirstOpenId = v1DatabaseOpenHelper.getValue(LAST_FIRSTOPEN_ID);
            String lastDeeplinkId = v1DatabaseOpenHelper.getValue(LAST_DEEPLINK_ID);
            String lastOpenId = v1DatabaseOpenHelper.getValue(LAST_OPEN_ID);
            String dailyFirstOpenPrevDate = v1DatabaseOpenHelper.getValue(DAILY_FIRST_OPEN_PREV_DATE);
            String registrationId = v1DatabaseOpenHelper.getValue(REGISTRATION_ID);
            boolean isPushEnable = Boolean.parseBoolean(v1DatabaseOpenHelper.getValue(PUSH_ENABLE));
            boolean isPushEnableOs = Boolean.parseBoolean(v1DatabaseOpenHelper.getValue(PUSH_ENABLE_OS));
            String userProperties = v1DatabaseOpenHelper.getValue(USER_PROFILE);

            String prevUUID = dataRegistry.safeGetString(DataRegistryKey.STRING_UUID, null);
            String prevLastFirstOpenId = dataRegistry.safeGetString(DataRegistryKey.STRING_LAST_FIRSTOPEN_ID, null);
            String prevLastDeeplinkId = dataRegistry.safeGetString(DataRegistryKey.STRING_LAST_DEEPLINK_ID, null);
            String prevLastOpenId = dataRegistry.safeGetString(DataRegistryKey.STRING_LAST_OPEN_ID, null);
            String prevDailyFirstOpenPrevDate = dataRegistry.safeGetString(DataRegistryKey.STRING_DAILY_FIRST_OPEN_PREV_DATE, null);
            String prevRegistrationId = dataRegistry.safeGetString(DataRegistryKey.STRING_REGISTRATION_ID, null);
            Boolean prevIsPushEnable = dataRegistry.getBooleanOrNull(DataRegistryKey.BOOLEAN_IS_PUSH_ENABLE);
            Boolean prevIsPushEnableOs = dataRegistry.getBooleanOrNull(DataRegistryKey.BOOLEAN_IS_PUSH_ENABLE_OS);

            //v1 -> v2 후 SDK 초기화 시 DeviceRealtimeDataProvider에서 ADID를 먼저 생성해버렸으므로 여기서 v1 값을 덮어씌운다.
            if(!CommonUtils.isNullOrEmpty(adId)){
                this.dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.STRING_ADID,
                                adId,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("Migrated adId = " + adId, true);
            }
            //v1 -> v2 후 SDK 초기화 시 DeviceStaticDataProvider에서 UUID를 먼저 생성해버렸으므로 여기서 v1 값을 덮어씌운다.
            dataRegistry.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.STRING_UUID,
                            uuid,
                            5,
                            this.getClass().getName(),
                            true
                    )
            );
            AbxLog.d("Migrated uuid = " + uuid, true);

            if (prevLastFirstOpenId == null) {
                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.STRING_LAST_FIRSTOPEN_ID,
                                lastFirstOpenId,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("Migrated last_firstopen_id = " + lastFirstOpenId, true);
            }

            if (prevLastDeeplinkId == null) {
                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.STRING_LAST_DEEPLINK_ID,
                                lastDeeplinkId,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("Migrated last_deeplink_id = " + lastDeeplinkId, true);
            }

            if (prevLastOpenId == null) {
                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.STRING_LAST_OPEN_ID,
                                lastOpenId,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("Migrated last_open_id = " + lastOpenId, true);
            }

            if (prevDailyFirstOpenPrevDate == null) {
                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.STRING_DAILY_FIRST_OPEN_PREV_DATE,
                                dailyFirstOpenPrevDate,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("Migrated daily_first_open_prev_date = " + dailyFirstOpenPrevDate, true);
            }

            if (prevRegistrationId == null) {
                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.STRING_REGISTRATION_ID,
                                registrationId,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("Migrated registration_id = " + registrationId, true);
            }

            if (prevIsPushEnable == null) {
                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.BOOLEAN_IS_PUSH_ENABLE,
                                isPushEnable,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("Migrated is_push_enable = " + isPushEnable, true);
            }

            if (prevIsPushEnableOs == null) {
                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.BOOLEAN_IS_PUSH_ENABLE_OS,
                                isPushEnableOs,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                AbxLog.d("Migrated is_push_enable_os = " + isPushEnableOs, true);
            }

            if (userProperties != null && !userProperties.equals("")) {
                saveUserProperties(userProperties);

                UserPropertyModel finalUserPropertyModel = userPropertyManager.getCurrentUserPropertyModel();

                try {
                    AbxLog.d("UserProperties after migration = " +
                            finalUserPropertyModel.getJson().toString(4), true);
                } catch (JSONException e) {
                    AbxLog.e(e, true);
                }
            }

            dataRegistry.putDataRegistry(
                    new DataUnit(
                            DataRegistryKey.BOOLEAN_IS_MIGRATED,
                            true,
                            5,
                            this.getClass().getName(),
                            true
                    )
            );
        }

        AbxLog.d("V1 event migration is available. Start migrating!", true);
        saveV1UnsentEvents();

        dataRegistry.saveRegistry();

        v1DatabaseOpenHelper.deleteDB();
        return true;
    }

    public void saveUserProperties(String userProperties) {
        try {
            UserPropertyModel userPropertyModel = userPropertyManager.getCurrentUserPropertyModel();
            JSONObject userPropertiesJSON = new JSONObject(userProperties);

            if (userPropertyModel.properties.isEmpty()) {
                updateEveryUserProperties(userPropertiesJSON);

                String userSnapshotId = v1DatabaseOpenHelper.getValue(USER_PROFILE_SNAPSHOT_ID);
                dataRegistry.putDataRegistry(
                        new DataUnit(
                                DataRegistryKey.STRING_USER_SNAPSHOT_ID,
                                userSnapshotId,
                                5,
                                this.getClass().getName(),
                                true
                        )
                );
                return;
            }

            String currentUserId = null;
            String prevUserId = null;

            if (userPropertyModel.properties.containsKey(USER_ID))
                currentUserId = (String) userPropertyModel.properties.get(USER_ID);

            if (currentUserId != null)
                currentUserId = currentUserId.replace("string:", "");

            if (userPropertiesJSON.has(USER_ID))
                prevUserId = userPropertiesJSON.getString(USER_ID);

            if (currentUserId != null && currentUserId.equals(prevUserId))
                updateCurrentUserNotContainedData(userPropertiesJSON);

        } catch (JSONException e) {
            AbxLog.e(e, true);
        }
    }

    private void updateCurrentUserNotContainedData(JSONObject userPropertiesJSON) {
        try {
            UserPropertyCommand command = new UserPropertyCommand();
            UserPropertyModel userPropertyModel = userPropertyManager.getCurrentUserPropertyModel();

            JSONObject truncatedUserPropertiesJson = CommonUtils.truncate(userPropertiesJSON, remoteConfigHandler.getConfigPropertyMaxSize());
            JSONObject parsedUserPropertiesJson = CommonUtils.parseValueWithDataType(truncatedUserPropertiesJson, CommonUtils.FixType.PREFIX);

            Iterator<?> keys = parsedUserPropertiesJson.keys();
            int currentSizeOfUserPropertyModel = userPropertyModel.properties.size();

            while (keys.hasNext()) {
                String key = (String) keys.next();

                if (currentSizeOfUserPropertyModel < remoteConfigHandler.getConfigPropertyMaxSize()) {

                    if (!userPropertyModel.properties.containsKey(key)) {
                        command.set(key, parsedUserPropertiesJson.get(key));
                        currentSizeOfUserPropertyModel++;
                    }

                } else {
                    AbxLog.d("UserProperties reaches MAX_PROPERTY_KEYS: " + remoteConfigHandler.getConfigPropertyMaxSize(), true);
                    break;
                }
            }

            userPropertyManager.merge(command);
        } catch (JSONException e) {
            AbxLog.e(e, true);
        }
    }

    private void updateEveryUserProperties(JSONObject userPropertiesJSON) {
        try {
            UserPropertyCommand command = new UserPropertyCommand();
            UserPropertyModel userPropertyModel = userPropertyManager.getCurrentUserPropertyModel();

            JSONObject truncatedUserPropertiesJson = CommonUtils.truncate(userPropertiesJSON,remoteConfigHandler.getConfigPropertyMaxSize());
            JSONObject parsedUserPropertiesJson = CommonUtils.parseValueWithDataType(truncatedUserPropertiesJson, CommonUtils.FixType.PREFIX);

            Iterator<?> keys = parsedUserPropertiesJson.keys();
            int currentSizeOfUserPropertyModel = userPropertyModel.properties.size();

            while (keys.hasNext()) {
                String key = (String) keys.next();

                if (currentSizeOfUserPropertyModel < remoteConfigHandler.getConfigPropertyMaxSize()) {
                    command.set(key, parsedUserPropertiesJson.get(key));
                    currentSizeOfUserPropertyModel++;
                } else {
                    AbxLog.d("UserProperties reaches MAX_PROPERTY_KEYS: " + remoteConfigHandler.getConfigPropertyMaxSize(), true);
                    break;
                }
            }

            userPropertyManager.merge(command);
        } catch (JSONException e) {
            AbxLog.e(e, true);
        }
    }

    private void saveV1UnsentEvents() {
        EventPackage v1UnsentEventPackage = v1DatabaseOpenHelper.getV1UnsentEventPackage();

        eventPackageContainer.mergeV1EventsAndV2Events(v1UnsentEventPackage);
        eventPackageContainer.updateUnsentEventPackagesInDataRegistry();
    }
}
