package io.adbrix.sdk.component;

import com.igaworks.v2.core.AdBrixRm;

import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;

import io.adbrix.sdk.configuration.AbxFacade;
import io.adbrix.sdk.data.dataprovider.INoWaitEventNameProvider;
import io.adbrix.sdk.data.modelprovider.CommonModelProvider;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.function.Completion;
import io.adbrix.sdk.domain.model.CommonModel;
import io.adbrix.sdk.domain.model.Empty;
import io.adbrix.sdk.domain.model.EventData;
import io.adbrix.sdk.domain.model.EventModel;
import io.adbrix.sdk.domain.model.EventPackage;
import io.adbrix.sdk.domain.model.Response;
import io.adbrix.sdk.domain.model.Result;
import io.adbrix.sdk.domain.model.Error;
import io.adbrix.sdk.domain.model.Success;
import io.adbrix.sdk.utils.CommonUtils;


public class EventBuffer implements IObserver<Void> {

    private int bufferCapacity = CoreConstants.EVENT_BUFFER_COUNT;
    private Queue<EventModel> currentBuffer;
    private EventUploadIntervalManager eventUploadIntervalManager;
    private EventPackageContainer eventPackageContainer;
    private INoWaitEventNameProvider noWaitEventNameProvider;
    private AbxFacade abxFacade;

    public EventBuffer(
            EventUploadIntervalManager eventUploadIntervalManager,
            EventPackageContainer eventPackageContainer,
            INoWaitEventNameProvider noWaitEventNameProvider,
            AbxFacade abxFacade
    ) {
        this.eventPackageContainer = eventPackageContainer;
        this.noWaitEventNameProvider = noWaitEventNameProvider;
        this.currentBuffer = new ConcurrentLinkedQueue<>();
        this.eventUploadIntervalManager = eventUploadIntervalManager;
        this.eventUploadIntervalManager.add(this);
        this.abxFacade = abxFacade;
    }

    public void addEventModel(EventModel model) {
        addEventModel(model, null);
    }

    public synchronized void addEventModel(EventModel model, Completion<Result<Response>> completion) {
        Queue<EventModel> buffer = getCurrentBuffer();
        buffer.add(model);

        if (noWaitEventNameProvider.isNowaitEvent(model.eventName)){
            if(CommonUtils.isNull(completion)){
                clearBufferAndFlushContainer(emptyResult -> {});
                sendEventToEventListener(model);
                return;
            }
            clearBufferAndFlushContainer(completion);
        }
        else{
            addEventPackageToContainerIfBufferFull();
        }
        sendEventToEventListener(model);
    }

    public void flushAllNow(Completion<Result<Empty>> completion) {
        clearBufferAndFlushContainer(new Completion<Result<Response>>() {
            @Override
            public void handle(Result<Response> responseResult) {
                completion.handle(Success.empty());
            }
        });
    }

    public void flushContainerAtIntervals() {
        this.eventPackageContainer.flushAtIntervals();
    }

    @Override
    public void update(Void v) {
        flushAllNow(emptyResult -> {});
    }

    public void saveUnsentEvents() {
        if (this.currentBuffer == null) {
            this.currentBuffer = new ConcurrentLinkedQueue<>();
        }

        if (currentBuffer.size() > 0) {
            eventPackageContainer.addEventPackage(new EventPackage(currentBuffer));
            eventPackageContainer.updateUnsentEventPackagesInDataRegistry();
            this.currentBuffer = new ConcurrentLinkedQueue<>();
        } else {
            eventPackageContainer.updateUnsentEventPackagesInDataRegistry();
        }
    }

    private Queue<EventModel> getCurrentBuffer() {
        if (this.currentBuffer == null) {
            this.currentBuffer = new ConcurrentLinkedQueue<>();
            return currentBuffer;
        }

        addEventPackageToContainerIfBufferFull();

        return this.currentBuffer;
    }

    private synchronized void clearBufferAndFlushContainer(Completion<Result<Response>> completion) {
        if (currentBuffer == null) {
            completion.handle(Error.of("eventBuffer is null") );
            return;
        }

        synchronized (currentBuffer) {
            //onNoWaitEvent
            if (currentBuffer.size() > 0) {
                eventPackageContainer.addEventPackage(new EventPackage(currentBuffer));
                eventPackageContainer.flushEventsNotSent(completion);
                this.currentBuffer = new ConcurrentLinkedQueue<>();
            } else {
                eventPackageContainer.flushEventsNotSent(completion);
            }
        }
    }

    private void addEventPackageToContainerIfBufferFull() {
        if (currentBuffer == null)
            return;

        synchronized (currentBuffer) {
            //onBufferFull
            if (currentBuffer.size() >= bufferCapacity) {
                eventPackageContainer.addEventPackage(new EventPackage(currentBuffer));
                this.currentBuffer = new ConcurrentLinkedQueue<>();
            }
        }
    }

    public static class BufferCapacityLimitsException extends Exception {
        BufferCapacityLimitsException(String message) {
            super(message);
        }
    }

    private void sendEventToEventListener(EventModel eventModel){
        AdBrixRm.EventListener eventListener = AdBrixRm.getEventListener();
        if(CommonUtils.isNull(eventListener)){
            return;
        }
        if(isBlockedEventForEventListener(eventModel)){
            return;
        }
        CommonModel commonModel = eventPackageContainer.getEventSender().getCommonModel();
        EventData eventData = new EventData(eventModel, commonModel);
        eventListener.onEvent(eventData);
    }
    private boolean isBlockedEventForEventListener(EventModel eventModel){
        boolean result = false;
        if(CommonUtils.isNull(eventModel)){
            return result;
        }
        String eventName = eventModel.eventName;
        if(CommonUtils.isNullOrEmpty(eventName)){
            return result;
        }
        eventName = eventName.replace("abx:","");
        if(CoreConstants.EVENT_SET_PUSH.equals(eventName)){
            result = true;
            return result;
        }
        if(CoreConstants.EVENT_ADID_CHANGED.equals(eventName)){
            result = true;
            return result;
        }
        return result;
    }
}
