package io.adbrix.sdk.push.style;

import android.app.Notification;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.content.res.Resources;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.graphics.Color;
import android.media.RingtoneManager;
import android.net.Uri;
import android.os.Build;
import android.util.Log;

import androidx.annotation.RequiresApi;

import com.igaworks.v2.core.AdBrixRm;

import org.json.JSONException;
import org.json.JSONObject;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.domain.ABXConstants;
import io.adbrix.sdk.domain.CompatConstants;
import io.adbrix.sdk.push.components.AbxPushCommonDAO;
import io.adbrix.sdk.push.models.AbxRemotePushModel;
import io.adbrix.sdk.push.models.BigPicturePushProperties;
import io.adbrix.sdk.push.models.BigTextPushProperties;
import io.adbrix.sdk.push.utils.PushUtils;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.HttpConnectionUtils;

public class BasePushStyle implements IPushStyle{
    Context context;
    Intent intent;
    AbxPushCommonDAO abxPushCommonDAO;
    int priority;
    int visibility;
    String title;
    String body;
    String summaryText;
    String bigTextTitle;
    String bigTextBody;
    Bitmap img;
    int notificationId;
    int color = Color.TRANSPARENT;
    int smallIcon;
    Bitmap largeIcon;
    boolean genVibe = true;
    boolean genSound = true;
    String notificationChannelId;
    Notification.Builder notificationBuilder;

    public BasePushStyle(Context context, Intent intent, AbxPushCommonDAO abxPushCommonDAO) {
        this.context = context;
        this.intent = intent;
        this.abxPushCommonDAO = abxPushCommonDAO;
        init();
    }

    protected void init(){
        this.priority = abxPushCommonDAO.getPriority();
        this.visibility = abxPushCommonDAO.getVisibility();
        this.color = abxPushCommonDAO.getARGB();
        this.smallIcon = getSmallIcon(abxPushCommonDAO);
        this.largeIcon = getLargeIcon(abxPushCommonDAO);

        String action = intent.getAction();
        if(CompatConstants.PUSH_REMOTE_MESSAGE_RECEIVE.equals(action) ||
            CompatConstants.PUSH_REMOTE_MESSAGE_RECEIVE_FROM_FIREBASE_MESSAGING_SERVICE.equals(action)){
            initRemotePush();
        } else if(CompatConstants.PUSH_LOCAL_MESSAGE_RECEIVE.equals(action)){
            String style = intent.getExtras().getString(CompatConstants.PUSH_PROP_STYLE);
            if(CommonUtils.isNullOrEmpty(style)){
                AbxLog.d("style is empty", false);
                return;
            }
            if(CompatConstants.PUSH_STYLE_BIG_TEXT.equals(style)){
                initLocalBigTextPush();
            } else if(CompatConstants.PUSH_STYLE_BIG_PICTURE.equals(style)){
                initLocalBigPicturePush();
            } else{
                AbxLog.d("Doesn't match with local notification style", false);
                return;
            }
        } else{
            AbxLog.d("Doesn't match with notification style", false);
            return;
        }
        this.notificationChannelId = ABXConstants.PUSH_NOTIFICATION_CHANNEL_ID;
        initNotificationBuilder(this.notificationChannelId);
        setContentView();
    }
    private void initLocalBigTextPush(){
        String properties = intent.getExtras().getString(CompatConstants.PUSH_TEXT_PROPERTIES);
        if(CommonUtils.isNullOrEmpty(properties)){
            AbxLog.e("ERROR :: notification properties is empty", false);
            return;
        }
        BigTextPushProperties bigTextPushProperties = new BigTextPushProperties();
        try {
            JSONObject jsonObject = new JSONObject(properties);
            bigTextPushProperties = bigTextPushProperties.setFromJSONObject(jsonObject);
        } catch (JSONException e) {
            AbxLog.e("ERROR :: bigTextPushProperties parsing is failed", false);
        }
        title = bigTextPushProperties.getTitle();
        if(CommonUtils.isNullOrEmpty(title)){
            title = PushUtils.getApplicationName(context);
        }
        body = bigTextPushProperties.getContentText();
        summaryText = bigTextPushProperties.getSummaryText();
        bigTextTitle = bigTextPushProperties.getBigContentTitle();
        bigTextBody = bigTextPushProperties.getBigText();
        notificationId = bigTextPushProperties.getEventId();
        genSound = true;
        genVibe = true;
    }
    private void initLocalBigPicturePush(){
        String properties = intent.getExtras().getString(CompatConstants.PUSH_PICTURE_PROPERTIES);
        if(CommonUtils.isNullOrEmpty(properties)){
            AbxLog.e("ERROR :: notification properties is empty", false);
            return;
        }
        BigPicturePushProperties bigPicturePushProperties = new BigPicturePushProperties();
        try {
            JSONObject jsonObject = new JSONObject(properties);
            bigPicturePushProperties = bigPicturePushProperties.setFromJSONObject(jsonObject);
        } catch (JSONException e) {
            AbxLog.e("ERROR :: bigPicturePushProperties parsing is failed", false);
        }
        title = bigPicturePushProperties.getTitle();
        if(CommonUtils.isNullOrEmpty(title)){
            title = PushUtils.getApplicationName(context);
        }
        body = bigPicturePushProperties.getContentText();
        summaryText = bigPicturePushProperties.getSummaryText();
        bigTextTitle = bigPicturePushProperties.getBigContentTitle();
        notificationId = bigPicturePushProperties.getEventId();
        genSound = true;
        genVibe = true;
        int resId = bigPicturePushProperties.getResourceId();
        if(resId != 0){
            img = PushUtils.getImageResource(context, resId);
            return;
        }
        String imgUrl = bigPicturePushProperties.getBigPictureUrl();
        if(CommonUtils.isNullOrEmpty(imgUrl)){
            AbxLog.e("ERROR :: imgUrl is empty", false);
            return;
        }
        img = HttpConnectionUtils.getBitmapFromURL(imgUrl, 0);
    }

    private void initRemotePush(){
        AbxRemotePushModel abxRemotePushModel = new AbxRemotePushModel(context, intent.getExtras());
        notificationId = abxRemotePushModel.notificationId;
        genVibe = abxRemotePushModel.genVibe;
        genSound = abxRemotePushModel.genSound;
        title = abxRemotePushModel.title;
        body = abxRemotePushModel.body;
        bigTextTitle = abxRemotePushModel.bigTextTitle;
        bigTextBody = abxRemotePushModel.bigTextBody;
        summaryText = abxRemotePushModel.summaryText;
        String imageUrl = abxRemotePushModel.imageUrl;
        if(!CommonUtils.isNullOrEmpty(imageUrl)){
            img = HttpConnectionUtils.getBitmapFromURL(imageUrl,0);
        }
        String customLargeIconUrl = abxRemotePushModel.largeIconUrl;
        if(!CommonUtils.isNullOrEmpty(customLargeIconUrl)){
            largeIcon = HttpConnectionUtils.getBitmapFromURL(customLargeIconUrl,0);
        }
    }

    protected void setContentView(){
        notificationBuilder.setContentTitle(PushUtils.fromHtml(title));
        notificationBuilder.setContentText(PushUtils.fromHtml(body));
        notificationBuilder.setSmallIcon(smallIcon);
        if(largeIcon != null){
            notificationBuilder.setLargeIcon(largeIcon);
        }
    }

    private void initNotificationBuilder(String notificationChannelId){
        notificationBuilder = new Notification.Builder(context);
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
            if(!isNotificationChannelExist(notificationChannelId)){
                AbxLog.e("notificationChannel is not exist. channel name: "+notificationChannelId, false);
                return;
            }
            notificationBuilder = new Notification.Builder(context, notificationChannelId);
        }
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN) {
            notificationBuilder.setPriority(priority);
        }
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            notificationBuilder.setVisibility(visibility);
        }
        if(isAvailableColor(color)){
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                notificationBuilder.setColor(color);
                AbxLog.d("Use custom brand color. Color : " + color, false);
            }
        }
        notificationBuilder.setAutoCancel(true);
        if(genSound){
            notificationBuilder.setSound(getSound());
        }
        if(genVibe){
            notificationBuilder.setDefaults(Notification.DEFAULT_VIBRATE);
        }
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
            notificationBuilder.setShowWhen(true);
        }
        notificationBuilder.setContentIntent(getContentIntent());
    }

    @RequiresApi(api = Build.VERSION_CODES.O)
    private boolean isNotificationChannelExist(String notificationChannelId){
        NotificationManager notificationManager = (NotificationManager) context.getSystemService(Context.NOTIFICATION_SERVICE);
        NotificationChannel notificationChannel = notificationManager.getNotificationChannel(notificationChannelId);
        if(notificationChannel == null){
            return false;
        } else{
            return true;
        }
    }

    private Uri getSound(){
        return RingtoneManager.getDefaultUri(RingtoneManager.TYPE_NOTIFICATION);
    }

    private PendingIntent getContentIntent(){
        Intent contentIntent = context.getPackageManager().getLaunchIntentForPackage(context.getPackageName());
        contentIntent.putExtras(intent.getExtras());
        contentIntent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK);
        String action = intent.getAction();
        if(CompatConstants.PUSH_LOCAL_MESSAGE_RECEIVE.equals(action)){
            contentIntent.setAction(CompatConstants.LOCAL_PUSH_OPEN_CLCKED);
        } else if(CompatConstants.PUSH_REMOTE_MESSAGE_RECEIVE.equals(action) ||
            CompatConstants.PUSH_REMOTE_MESSAGE_RECEIVE_FROM_FIREBASE_MESSAGING_SERVICE.equals(action)){
            contentIntent.setAction(CompatConstants.REMOTE_PUSH_OPEN_CLICKED);
        }
        return PendingIntent.getActivity(context, notificationId, contentIntent, PushUtils.getPendingIntentFlags(context));
    }

    private int getSmallIcon(AbxPushCommonDAO pushCommonDAO){
        int iconId = getDefaultSmallIcon();
        String smallIconName = pushCommonDAO.getSmallIconName();
        if(CommonUtils.isNullOrEmpty(smallIconName)){
            return iconId;
        }
        iconId = context.getResources().getIdentifier(smallIconName, "drawable", context.getPackageName());
        if(iconId <= 0){
            iconId = getDefaultSmallIcon();
            return iconId;
        }
        if(Build.VERSION.SDK_INT >= 21) {
            return iconId;
        }
        Bitmap bitmap = BitmapFactory.decodeResource(context.getResources(), iconId);
        if(bitmap == null){
            iconId = getDefaultSmallIcon();
            AbxLog.d("device does not support vector drawable", false);
            return iconId;
        }
        return iconId;
    }
    private int getDefaultSmallIcon(){
        int iconId = -1;
        Intent launchIntent = context.getPackageManager().getLaunchIntentForPackage(context.getPackageName());
        ComponentName componentName = launchIntent.getComponent();
        String packageName=componentName.getPackageName();
        final PackageManager pm = context.getPackageManager();
        final ApplicationInfo applicationInfo;
        try {
            applicationInfo = pm.getApplicationInfo(packageName, PackageManager.GET_META_DATA);
            final Resources resources = pm.getResourcesForApplication(applicationInfo);
            iconId = applicationInfo.icon;
            AbxLog.d("A default small icon is set", false);
        } catch (PackageManager.NameNotFoundException e) {
            AbxLog.e(e, true);
        }
        return iconId;
    }

    private Bitmap getLargeIcon(AbxPushCommonDAO pushCommonDAO){
        String largeIconName = pushCommonDAO.getLargeiconName();
        if(CommonUtils.isNullOrEmpty(largeIconName)){
            return null;
        }
        Bitmap largeIcon = PushUtils.getImageResource(context, pushCommonDAO.getLargeiconName());
        AbxLog.d("large icon name : " + largeIconName, false);
        return largeIcon;
    }

    private boolean isAvailableColor(int color){
        boolean result = false;
        if(color == Color.TRANSPARENT){
            return result;
        }
        result = true;
        return result;
    }

    @Override
    public int getNotificationId() {
        return notificationId;
    }

    @Override
    public Notification getNotification() {
        return notificationBuilder.getNotification();
    }
}
