package io.adbrix.sdk.data.push;

import android.content.Context;

import org.json.JSONArray;
import org.json.JSONObject;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import javax.annotation.Nullable;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.Optional;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.ABXConstants;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.ActionContent;
import io.adbrix.sdk.domain.model.ActionHistory;
import io.adbrix.sdk.domain.model.ActionHistoryError;
import io.adbrix.sdk.domain.model.Empty;
import io.adbrix.sdk.domain.model.Result;
import io.adbrix.sdk.domain.model.Success;
import io.adbrix.sdk.utils.CommonUtils;

public class ActionHistoryDAO {
    private Context context;
    private DataRegistry dataRegistry;
    private ActionHistorySqliteStore actionHistorySqliteStore;

    public ActionHistoryDAO(Context context, DataRegistry dataRegistry) {
        this.context = context;
        this.dataRegistry = dataRegistry;
        this.actionHistorySqliteStore = new ActionHistorySqliteStore(context);
    }

    public Result<Empty> insertActionHistory(@Nullable JSONArray historyJsonArray) {
        Result<Empty> result = Success.empty();

        if (historyJsonArray == null)
            return ActionHistoryError.NO_JSON_VALUE_ERROR.getError();

        for (int i = 0; i < historyJsonArray.length(); i++) {
            try {
                JSONObject historyJSONObject = historyJsonArray.getJSONObject(i);

                actionHistorySqliteStore.insertActionHistory(
                        historyJSONObject.optString(CoreConstants.ACTION_HISTORY_RESPONSE_HISTORY_ID),
                        historyJSONObject.optString(CoreConstants.ACTION_HISTORY_RESPONSE_ACTION_TYPE),
                        Optional.ofNullable(historyJSONObject.optJSONObject(CoreConstants.ACTION_HISTORY_RESPONSE_CONTENTS))
                                .map(JSONObject::toString)
                                .orElseGet(() -> ""),
                        historyJSONObject.optLong(CoreConstants.ACTION_HISTORY_RESPONSE_TIMESTAMP),
                        historyJSONObject.optString(CoreConstants.ACTION_HISTORY_RESPONSE_CAMPAIGN_ID),
                        historyJSONObject.optInt(CoreConstants.ACTION_HISTORY_RESPONSE_CAMPAIGN_REVISION_NO),
                        historyJSONObject.optString(CoreConstants.ACTION_HISTORY_RESPONSE_STEP_ID),
                        historyJSONObject.optString(CoreConstants.ACTION_HISTORY_RESPONSE_CYCLE_TIME),
                        historyJSONObject.optString(CoreConstants.ACTION_HISTORY_RESPONSE_GROUP),
                        true,
                        historyJSONObject.optBoolean(CoreConstants.ACTION_HISTORY_RESPONSE_IS_READ)
                );
            } catch (Exception e) {
                result = ActionHistoryError.SQLITE_QUERY_ERROR.getError();
            }
        }

        return result;
    }

    public Result<List<ActionHistory>> getActionHistory(int skip, int limit, List<String> actionType) {
        try {
            return Success.of(actionHistorySqliteStore.getActionHistory(skip, limit, actionType));
        } catch (Exception e) {
            return ActionHistoryError.SQLITE_QUERY_ERROR.getError();
        }
    }

    public Result<List<ActionHistory>> getAllActionHistory(List<String> actionType) {
        try {
            return Success.of(actionHistorySqliteStore.getAllActionHistory(actionType));
        } catch (Exception e) {
            return ActionHistoryError.SQLITE_QUERY_ERROR.getError();
        }
    }

    public Result<Empty> deleteActionHistory(String historyId, long timestamp) {
        try {
            actionHistorySqliteStore.deleteActionHistory(historyId, timestamp);
            return Success.empty();
        } catch (Exception e) {
            return ActionHistoryError.SQLITE_QUERY_ERROR.getError();
        }
    }

    public Result<Empty> clearSyncedActionHistoryInLocalDB() {
        try {
            actionHistorySqliteStore.clearSyncedActionHistoryInLocalDB();
            return Success.empty();
        } catch (Exception e) {
            return ActionHistoryError.SQLITE_QUERY_ERROR.getError();
        }
    }

    public Result<Empty> clearAllActionHistoryInLocalDB() {
        try {
            actionHistorySqliteStore.clearAllActionHistoryInLocalDB();
            return Success.empty();
        } catch (Exception e) {
            return ActionHistoryError.SQLITE_QUERY_ERROR.getError();
        }
    }

    public Result<Long> getLastServerTimeStamp() {
        try {
            return Success.of(actionHistorySqliteStore.getLastServerTimeStamp());
        } catch (Exception e) {
            return ActionHistoryError.SQLITE_QUERY_ERROR.getError();
        }
    }

    public Result<Empty> deleteOutdatedPushData(long lastTimestamp) {
        try {
            actionHistorySqliteStore.deleteOutdatedPushData(lastTimestamp);
            return Success.empty();
        } catch (Exception e) {
            return ActionHistoryError.SQLITE_QUERY_ERROR.getError();
        }
    }

    public void insertPushData(String pushDataString) {
        try {
            JSONObject pushJson = new JSONObject(pushDataString);
            JSONObject customAlertJSON = pushJson.getJSONObject(ABXConstants.PUSH_REMOTE_KEY_CUSTOM_ALERT);
            JSONObject languageJson;
            String language = CommonUtils.getLanguage(context);

            if (customAlertJSON.has(language)) {
                languageJson = customAlertJSON.getJSONObject(language);
            } else if (customAlertJSON.has(language.substring(0, 2))) {
                languageJson = customAlertJSON.getJSONObject(language.substring(0, 2));
            } else if (pushJson.has(ABXConstants.PUSH_REMOTE_KEY_ALERT)) {
                languageJson = pushJson.getJSONObject(ABXConstants.PUSH_REMOTE_KEY_ALERT);
            } else {
                languageJson = new JSONObject();
            }

            String title = languageJson.optString(ABXConstants.PUSH_REMOTE_KEY_TITLE);
            String body = languageJson.optString(ABXConstants.PUSH_REMOTE_KEY_BODY);
            String imageUrl = languageJson.optString(ABXConstants.PUSH_REMOTE_KEY_PUSH_IMAGE);
            String largeIconUrl = languageJson.optString(ABXConstants.PUSH_REMOTE_KEY_PUSH_LARGE_ICON);
            String bigTextTitle = languageJson.optString(ABXConstants.PUSH_REMOTE_KEY_BIG_TITLE);
            String bigTextBody = languageJson.optString(ABXConstants.PUSH_REMOTE_KEY_BIG_BODY);
            String summaryText = languageJson.optString(ABXConstants.PUSH_REMOTE_KEY_SUMMARY_TEXT);

            String campaignId = pushJson.optString(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_ID);
            String stepId = pushJson.optString(ABXConstants.GROWTH_EVENT_KEY_STEP_ID);
            String cycleTime = pushJson.optString(ABXConstants.GROWTH_EVENT_KEY_CYCLE_TIME);
            long notificationId = pushJson.optLong(ABXConstants.PUSH_REMOTE_KEY_NOTIFICATION_ID);
            String historyId = CoreConstants.ACTION_HISTORY_ACTION_TYPE_PUSH_ANDROID + "::" + notificationId;
            int revisionNo = pushJson.optInt(ABXConstants.GROWTH_EVENT_KEY_CAMPAIGN_REVISION_NO);
            JSONObject deeplinkJson = pushJson.optJSONObject(ABXConstants.PUSH_REMOTE_KEY_DEEPLINKJSON);
            String deeplinkUrl = pushJson.optString(ABXConstants.PUSH_REMOTE_KEY_DEEPLINKURL);

            long timestamp = System.currentTimeMillis();

            ActionContent actionContent = new ActionContent(
                    notificationId,
                    title,
                    null,
                    body,
                    bigTextTitle,
                    bigTextBody,
                    summaryText,
                    largeIconUrl,
                    imageUrl,
                    deeplinkJson,
                    deeplinkUrl,
                    false,
                    null,
                    new ArrayList<>()
            );

            actionHistorySqliteStore.insertActionHistory(
                    historyId,
                    CoreConstants.ACTION_HISTORY_ACTION_TYPE_PUSH_ANDROID,
                    actionContent.getJson().toString(),
                    timestamp,
                    campaignId,
                    revisionNo,
                    stepId,
                    cycleTime,
                    "",
                    false,
                    false
            );

        } catch (Exception e) {
            AbxLog.e(e, true);
        }
    }

    public void deleteDB() {
        try {
            actionHistorySqliteStore.deleteDB();
        } catch (Exception e) {
            AbxLog.e(e, true);
        }
    }

    public void restartDB() {
        try {
            actionHistorySqliteStore.restartDB();
        } catch (Exception e) {
            AbxLog.e(e, true);
        }
    }
}
