package io.adbrix.sdk.data.net;

import com.igaworks.v2.core.AdBrixRm;

import java.util.Arrays;
import java.util.Timer;
import java.util.TimerTask;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.data.DFNSessionState;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.model.Error;
import io.adbrix.sdk.domain.model.Success;
import io.adbrix.sdk.utils.CommonUtils;

public class ApiConnectionManager<Param> {
    private static final int MAX_RETRY = 3;
    private final Result<Param> result;
    private Param param;
    private int attempt = 0;
    private Timer retryTimer;
    private TimerTask retryTimerTask;

    public ApiConnectionManager(Result<Param> result, Param param) {
        this.param = param;
        if (result == null)
            this.result = new NullResult<>();
        else this.result = result;
    }

    public ApiConnectionManager() {
        this.result = new NullResult<>();
    }

    public void executeWithRetry(IApiConnection apiConnection) {

        String response = null;
        try {
            response = apiConnection.request();
        } catch (Exception e) {
            AbxLog.e(e, true);
            return;
        }

        if (apiConnection.isHttpOK()) {
            if (response != null && response.equals("{\"deeplink\":null}")) {
                retry(apiConnection);
            } else
                result.connectSuccess(response, apiConnection.getResponseCode(), param);
        } else if (apiConnection.isWrongAppkey()) {
            // HTTP 500: wrong appKey
            //DFNSessionState.getInstance().adbrixError.getAndSet(true);
            AbxLog.w("Warning!! :: wrong appkey : " + response, true);

            retry(apiConnection);
        } else if (apiConnection.isInvalidAppkey()) {
            //AppKey ERR
            //DFNSessionState.getInstance().adbrixError.getAndSet(true);
            AbxLog.w("Warning!! :: Appkey is not valid. " + response, true);
            // not retry
            AbxLog.d("Broken connection to AdBrixRm. Skip retry uploading events. " + response, true);

            AdBrixRm.disableSDK("AdbrixAllStop");
            result.connectFail(apiConnection.getResponseCode(), param);
        } else {
            // not retry
            AbxLog.d("Broken connection to AdBrixRm. Skip retry uploading events. " + response, true);
            result.connectFail(apiConnection.getResponseCode(), param);
        }
    }

    public void execute(IApiConnection apiConnection) {

        String response = null;
        try {
            response = apiConnection.request();
        } catch (Exception e) {
            AbxLog.e(e, true);
            return;
        }

        if (apiConnection.isHttpOK()) {
            result.connectSuccess(response, apiConnection.getResponseCode(), param);
        } else if (apiConnection.isWrongAppkey()) {
            // HTTP 500: wrong appKey
            //DFNSessionState.getInstance().adbrixError.getAndSet(true);
            AbxLog.w("Warning!! :: wrong appkey : " + response, true);
            result.connectFail(apiConnection.getResponseCode(), param);
        } else if (apiConnection.isInvalidAppkey()) {
            //AppKey ERR
            //DFNSessionState.getInstance().adbrixError.getAndSet(true);
            AbxLog.w("Warning!! :: Appkey is not valid. " + response, true);
            // not retry
            AbxLog.w("Broken connection to AdBrixRm. Skip retry uploading events. " + response, true);

            AdBrixRm.disableSDK("AdbrixAllStop");
            result.connectFail(apiConnection.getResponseCode(), param);
        } else {
            // not retry
            AbxLog.d("Broken connection to AdBrixRm. Skip retry uploading events. " + response, true);
            result.connectFail(apiConnection.getResponseCode(), param);
        }
    }

    public io.adbrix.sdk.domain.model.Result<Response> executeWithResult(IApiConnection apiConnection) {

        String response = null;
        try {
            response = apiConnection.request();
        } catch (Exception e) {
            AbxLog.e(e, true);
            return Error.of(e);
        }

        if (apiConnection.isHttpOK()) {
            return Success.of(new Response(apiConnection.getResponseCode(), response));
        } else if (apiConnection.isWrongAppkey()) {
            // HTTP 500: wrong appKey
            //DFNSessionState.getInstance().adbrixError.getAndSet(true);
            AbxLog.w("Warning!! :: wrong appkey : " + response, true);
            return Error.of(response);
        } else if (apiConnection.isInvalidAppkey()) {
            //AppKey ERR
            //DFNSessionState.getInstance().adbrixError.getAndSet(true);
            AbxLog.w("Warning!! :: Appkey is not valid. " + response, true);
            // not retry
            AbxLog.w("Broken connection to AdBrixRm. Skip retry uploading events. " + response, true);

            AdBrixRm.disableSDK("AdbrixAllStop");
            return Error.of(response);
        } else {
            // not retry
            AbxLog.d("Broken connection to AdBrixRm. Skip retry uploading events. " + response, true);
            return Error.of(response);
        }
    }

    private void retry(IApiConnection apiConnection) {
        if (attempt >= MAX_RETRY) {
            AbxLog.d("ApiConnectionManager retry failed. Skip retry uploading events.", true);
            result.connectFail(apiConnection.getResponseCode(), param);
            attempt = 0;
            return;
        }
        retryTimerTask = new TimerTask() {
            @Override
            public void run() {
                AbxLog.d("ApiConnectionManager retry : (" + (attempt + 1) + "/3)", true);
                attempt++;
                executeWithRetry(apiConnection);
            }
        };
        retryTimer = new Timer();
        int delay = (attempt + 1) * CoreConstants.NETWORK_BACKOFF_MULTIPLIER;
        retryTimer.schedule(retryTimerTask, delay);
    }

    public static class Response {
        public final int code;
        public final String message;

        public Response(int code, String message) {
            this.code = code;
            this.message = message;
        }
    }

    public interface Result<Param> {
        void connectSuccess(String responseString, int responseCode, Param param);

        void connectFail(int responseCode, Param param);
    }

    private static class NullResult<Param> implements Result<Param> {
        @Override
        public void connectSuccess(String responseString, int responseCode, Param param) {
            //do nothing
        }

        @Override
        public void connectFail(int responseCode, Param param) {
            //do nothing
        }
    }
    public void setTimerToNull() {
        CommonUtils.cancelTimerTask(retryTimerTask);
        retryTimerTask = null;
        CommonUtils.cancelTimer(retryTimer);
        retryTimer = null;
    }
}
