package io.adbrix.sdk.data.inAppMessage;

import android.content.ContentValues;
import android.content.Context;
import android.content.ContextWrapper;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.text.TextUtils;
import android.util.Pair;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Queue;
import java.util.TimeZone;
import java.util.UUID;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

import io.adbrix.sdk.component.AbxLog;
import io.adbrix.sdk.component.TryOptional;
import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.IAMConstants;
import io.adbrix.sdk.domain.model.DfnIAMTrigger;
import io.adbrix.sdk.domain.model.DfnInAppMessage;
import io.adbrix.sdk.domain.model.Error;
import io.adbrix.sdk.domain.model.IAMEnums;
import io.adbrix.sdk.domain.model.Result;
import io.adbrix.sdk.domain.model.Success;
import io.adbrix.sdk.utils.CommonUtils;
import io.adbrix.sdk.utils.HttpConnectionUtils;
import io.adbrix.sdk.utils.CoreUtils;

public class InAppMessageDAO {
    InAppMessageSqliteStore dbHelper;
    DataRegistry dataRegistry;
    Context context;
    ExecutorService imageDownloadExecutorService;

    public InAppMessageDAO(Context context, DataRegistry dataRegistry) {
        dbHelper = new InAppMessageSqliteStore(context);
        this.dataRegistry = dataRegistry;
        this.context = context;
        this.imageDownloadExecutorService = Executors.newSingleThreadExecutor();
    }

    public InAppMessageDAO(Context context, DataRegistry dataRegistry, InAppMessageSqliteStore dbHelper) {
        this.dbHelper = dbHelper;
        this.dataRegistry = dataRegistry;
        this.context = context;
        this.imageDownloadExecutorService = Executors.newSingleThreadExecutor();
    }

    public void insertApiResponse2Table(String response) throws JSONException {
        ArrayList<String> responseCampaignIdArrayList = new ArrayList<>();

        JSONObject responseJsonObject = new JSONObject(response);

        JSONObject data = responseJsonObject.getJSONObject(IAMConstants.RESPONSE_DATA);

        String status = data.optString(IAMConstants.RESPONSE_STATUS);
        String checksum = data.optString(IAMConstants.RESPONSE_CHECKSUM);
        int minutesToExpiry = data.optInt(IAMConstants.RESPONSE_MINUTES_TO_EXPIRY);

        if (IAMConstants.CHECKSUM_UNCHANGED.equals(status) || IAMConstants.CHECKSUM_NO_ACTIVE_CAMPAIGN.equals(checksum)) {
            if (IAMConstants.CHECKSUM_UNCHANGED.equals(status))
                AbxLog.d("insertApiResponse2Table: inAppMessage Campaign unchanged", true);
            else
                AbxLog.d("insertApiResponse2Table: No Active Campaign", true);

            dataRegistry.putDataRegistry(new DataUnit(
                    DataRegistryKey.STRING_IN_APP_MESSAGE_CHECKSUM,
                    checksum,
                    5,
                    this.getClass().getName(),
                    true
            ));

            dataRegistry.putDataRegistry(new DataUnit(
                    DataRegistryKey.INT_IN_APP_MESSAGE_MINUTES_TO_EXPIRY,
                    minutesToExpiry,
                    5,
                    this.getClass().getName(),
                    true
            ));

            dataRegistry.putDataRegistry(new DataUnit(
                    DataRegistryKey.LONG_IN_APP_MESSAGE_LAST_RESPONSE_TIME,
                    System.currentTimeMillis(),
                    5,
                    this.getClass().getName(),
                    true
            ));

            return;
        }

        JSONArray inAppMessages = TryOptional.optGet(() -> data.getJSONArray(IAMConstants.RESPONSE_IAM), JSONArray::new);

        deleteBeforeInsertApiResponse();

        ConcurrentLinkedQueue<Runnable> imageDownloadRunnableQueue = new ConcurrentLinkedQueue<>();

        for (int inAppMessagesIdx = 0; inAppMessagesIdx < inAppMessages.length(); inAppMessagesIdx++) {
            JSONObject inAppMessage = inAppMessages.optJSONObject(inAppMessagesIdx);
            JSONObject contents = TryOptional.optGet(() -> inAppMessage.getJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS), JSONObject::new);
            JSONObject frequencyCap = TryOptional.optGet(() -> inAppMessage.getJSONObject(IAMConstants.RESPONSE_IAM_FREQUENCY_CAP), JSONObject::new);
            JSONArray triggers = TryOptional.optGet(() -> inAppMessage.getJSONArray(IAMConstants.RESPONSE_IAM_TRIGGERS), JSONArray::new);
            JSONArray availableTimes = TryOptional.optGet(() -> inAppMessage.getJSONArray(IAMConstants.RESPONSE_IAM_AVAILABLE_TIME), JSONArray::new);
            JSONObject extConfig = inAppMessage.optJSONObject(IAMConstants.RESPONSE_IAM_EXT_CONFIG);

            long startDateTime = CoreUtils.getStartDateTimeLong(inAppMessage.optString(IAMConstants.RESPONSE_IAM_START_DATETIME));
            long endDateTime = CoreUtils.getEndDateTimeLong(inAppMessage.optString(IAMConstants.RESPONSE_IAM_END_DATETIME));
            long lastUpdatedDateTime = getLastUpdatedDateTimeLong(inAppMessage.optString(IAMConstants.RESPONSE_IAM_LAST_UPDATED_DATETIME));

            responseCampaignIdArrayList.add(inAppMessage.optString(IAMConstants.RESPONSE_IAM_CAMPAIGN_ID));

            if (startDateTime == 0 || endDateTime == -1 || lastUpdatedDateTime == 0) {
                AbxLog.d("insertApiResponse2Table: DateTime Parse Error", true);
                return;
            }

            String campaignId = inAppMessage.optString(IAMConstants.RESPONSE_IAM_CAMPAIGN_ID);

            if (campaignId.equals(""))
                continue;

            boolean portraitAvailable = TryOptional.of(() -> inAppMessage.getJSONArray(IAMConstants.RESPONSE_IAM_PAGE_ORIENTATION))
                    .map(JSONArray::toString)
                    .map(pageOrientation -> pageOrientation.contains(IAMConstants.PORTRAIT))
                    .orElse(false);

            boolean landscapeAvailable = TryOptional.of(() -> inAppMessage.getJSONArray(IAMConstants.RESPONSE_IAM_PAGE_ORIENTATION))
                    .map(JSONArray::toString)
                    .map(pageOrientation -> pageOrientation.contains(IAMConstants.LANDSCAPE))
                    .orElse(false);

            insertInAppMessageTable(
                    campaignId,
                    portraitAvailable ? 1 : 0,
                    landscapeAvailable ? 1 : 0,
                    inAppMessage.optJSONObject(IAMConstants.RESPONSE_IAM_EXT_ATTR) == null ? null : inAppMessage.getJSONObject(IAMConstants.RESPONSE_IAM_EXT_ATTR).toString(),
                    inAppMessage.optInt(IAMConstants.RESPONSE_IAM_TIMEZONE_OFFSET),
                    inAppMessage.optString(IAMConstants.RESPONSE_IAM_TIMEZONE_TYPE),
                    inAppMessage.optString(IAMConstants.RESPONSE_IAM_TYPE),
                    inAppMessage.optString(IAMConstants.RESPONSE_IAM_LAYOUT),
                    contents.optJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_BG_STYLE) == null ? null : contents.getJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_BG_STYLE).getString(IAMConstants.RESPONSE_IAM_CONTENTS_BG_STYLE_BG_COLOR),
                    contents.optJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_BG_STYLE) == null ? null : contents.getJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_BG_STYLE).getString(IAMConstants.RESPONSE_IAM_CONTENTS_BG_STYLE_OVERLAY_COLOR),
                    contents.optJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_TEXT) == null ? null : contents.getJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_TEXT).toString(),
                    contents.optJSONArray(IAMConstants.RESPONSE_IAM_CONTENTS_BUTTONS) == null ? null : contents.getJSONArray(IAMConstants.RESPONSE_IAM_CONTENTS_BUTTONS).toString(),
                    contents.optJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_DEFAULT_CLOSE_BUTTON) == null ? null : contents.getJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_DEFAULT_CLOSE_BUTTON).optString(IAMConstants.RESPONSE_IAM_CONTENTS_DEFAULT_CLOSE_BUTTON_COLOR),
                    contents.optJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_DEFAULT_CLOSE_BUTTON) == null ? null : contents.getJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_DEFAULT_CLOSE_BUTTON).optString(IAMConstants.RESPONSE_IAM_CONTENTS_DEFAULT_CLOSE_BUTTON_BG_COLOR),
                    contents.optJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_STICKY_BANNER_OPTION) == null ? null : contents.getJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_STICKY_BANNER_OPTION).toString(),
                    contents.optJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_SCROLLABLE_IMAGE_OPTION) == null ? null : contents.getJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_SCROLLABLE_IMAGE_OPTION).toString(),
                    frequencyCap.optInt(IAMConstants.RESPONSE_IAM_FREQUENCY_CAP_PER_SESSION),
                    frequencyCap.optInt(IAMConstants.RESPONSE_IAM_FREQUENCY_CAP_PER_USER),
                    frequencyCap.optJSONObject(IAMConstants.RESPONSE_IAM_FREQUENCY_CAP_PER_PERIOD) == null ? 0 : frequencyCap.getJSONObject(IAMConstants.RESPONSE_IAM_FREQUENCY_CAP_PER_PERIOD).getInt(IAMConstants.RESPONSE_IAM_FREQUENCY_CAP_PER_PERIOD_PERIOD_IN_MINUTES),
                    frequencyCap.optJSONObject(IAMConstants.RESPONSE_IAM_FREQUENCY_CAP_PER_PERIOD) == null ? 0 : frequencyCap.getJSONObject(IAMConstants.RESPONSE_IAM_FREQUENCY_CAP_PER_PERIOD).getInt(IAMConstants.RESPONSE_IAM_FREQUENCY_CAP_PER_PERIOD_PER_PERIOD),
                    startDateTime,
                    endDateTime,
                    lastUpdatedDateTime,
                    extConfig == null ? null : extConfig.toString()
            );

            for (int triggersIdx = 0; triggersIdx < triggers.length(); triggersIdx++) {
                JSONObject trigger = triggers.getJSONObject(triggersIdx);
                String eventName = trigger.optString(IAMConstants.RESPONSE_IAM_TRIGGERS_EVT);

                insertTriggersTable(
                        campaignId,
                        trigger.optString(IAMConstants.RESPONSE_IAM_TRIGGERS_TYPE),
                        eventName,
                        trigger.optInt(IAMConstants.RESPONSE_IAM_TRIGGERS_PRIORITY)
                );
            }

            if (availableTimes.length() == 0) {
                insertAvailableTimeTable(
                        campaignId,
                        IAMConstants.AVAILABLE_TIME_ALL,
                        0,
                        0
                );
            } else {
                for (int availableTimeIdx = 0; availableTimeIdx < availableTimes.length(); availableTimeIdx++) {
                    JSONObject availableTime = availableTimes.getJSONObject(availableTimeIdx);
                    JSONArray days = availableTime.getJSONArray(IAMConstants.RESPONSE_IAM_AVAILABLE_TIME_DAYS);

                    for (int dayIdx = 0; dayIdx < days.length(); dayIdx++) {
                        insertAvailableTimeTable(
                                campaignId,
                                days.getString(dayIdx),
                                availableTime.getInt(IAMConstants.RESPONSE_IAM_AVAILABLE_TIME_START_MIN),
                                availableTime.getInt(IAMConstants.RESPONSE_IAM_AVAILABLE_TIME_END_MIN)
                        );
                    }
                }
            }

            if (!isCurrentFrequencyExist(inAppMessage.getString(IAMConstants.RESPONSE_IAM_CAMPAIGN_ID))) {
                insertCurrentFrequencyCapTable(
                        campaignId,
                        0,
                        0,
                        0,
                        0
                );
            }

            if (portraitAvailable && contents.optJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_IMAGE) != null) {
                String portraitUrlString = contents.getJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_IMAGE).getJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_IMAGE_URL).getString(IAMConstants.RESPONSE_IAM_CONTENTS_IMAGE_URL_PORTRAIT);

                if (!getImageUrlFromImageTable(inAppMessage.getString(IAMConstants.RESPONSE_IAM_CAMPAIGN_ID), IAMConstants.PORTRAIT).equals(portraitUrlString)) {
                    deleteImageTableWithOrientation(inAppMessage.getString(IAMConstants.RESPONSE_IAM_CAMPAIGN_ID), IAMConstants.PORTRAIT);

                    insertImageTable(
                            campaignId,
                            IAMConstants.PORTRAIT,
                            portraitUrlString,
                            IAMConstants.NO_IMAGE,
                            contents.getJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_IMAGE).optJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_IMAGE_CLICK_ACTION) == null ? null : contents.getJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_IMAGE).getJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_IMAGE_CLICK_ACTION).toString(),
                            false
                    );

                    imageDownloadRunnableQueue.add(() -> getAndSaveImageInfo(campaignId, IAMConstants.PORTRAIT, false));
                }
            }

            if (landscapeAvailable && contents.optJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_IMAGE) != null) {
                String landscapeUrlString = contents.getJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_IMAGE).getJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_IMAGE_URL).getString(IAMConstants.RESPONSE_IAM_CONTENTS_IMAGE_URL_LANDSCAPE);

                if (!getImageUrlFromImageTable(inAppMessage.getString(IAMConstants.RESPONSE_IAM_CAMPAIGN_ID), IAMConstants.LANDSCAPE).equals(landscapeUrlString)) {
                    deleteImageTableWithOrientation(inAppMessage.getString(IAMConstants.RESPONSE_IAM_CAMPAIGN_ID), IAMConstants.LANDSCAPE);

                    insertImageTable(
                            campaignId,
                            IAMConstants.LANDSCAPE,
                            landscapeUrlString,
                            IAMConstants.NO_IMAGE,
                            contents.getJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_IMAGE).optJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_IMAGE_CLICK_ACTION) == null ? null : contents.getJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_IMAGE).getJSONObject(IAMConstants.RESPONSE_IAM_CONTENTS_IMAGE_CLICK_ACTION).toString(),
                            false
                    );

                    imageDownloadRunnableQueue.add(() -> getAndSaveImageInfo(campaignId, IAMConstants.LANDSCAPE, false));
                }
            }
        }

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_IN_APP_MESSAGE_CHECKSUM,
                checksum,
                5,
                this.getClass().getName(),
                true
        ));

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.INT_IN_APP_MESSAGE_MINUTES_TO_EXPIRY,
                minutesToExpiry,
                5,
                this.getClass().getName(),
                true
        ));

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.LONG_IN_APP_MESSAGE_LAST_RESPONSE_TIME,
                System.currentTimeMillis(),
                5,
                this.getClass().getName(),
                true
        ));

        deleteStoppedOrDeletedInAppMessage(responseCampaignIdArrayList);

        while (!imageDownloadRunnableQueue.isEmpty()) {
            Runnable imageDownloadRunnable = imageDownloadRunnableQueue.poll();
            if(imageDownloadRunnable != null){
                imageDownloadExecutorService.submit(imageDownloadRunnable);
            }
        }
    }

    public long getLastUpdatedDateTimeLong(String lastUpdatedDateString) {
        SimpleDateFormat lastUpdatedDateFormat = new SimpleDateFormat(CoreConstants.IAM_LAST_UPDATE_DATE_FORMAT, Locale.US);
        Date lastUpdatedDate;
        try {
            lastUpdatedDate = lastUpdatedDateFormat.parse(lastUpdatedDateString);
        } catch (ParseException e) {
            AbxLog.w("getLastUpdatedDateTimeLong: parse Error", true);
            return 0;
        }

        if (lastUpdatedDate != null) {
            return lastUpdatedDate.getTime();
        }

        return 0;
    }


    public void insertInAppMessageTable(
            String campaignId,
            int portraitAvailable,
            int landscapeAvailable,
            String externalAttr,
            Integer timezoneOffset,
            String timezoneType,
            String type,
            String layout,
            String bgColor,
            String overlayColor,
            String contentsText,
            String contentsButtons,
            String defaultCloseButtonColor,
            String defaultCloseButtonBgColor,
            String stickyBannerOption,
            String scrollableImageOption,
            int frequencyCapPerSession,
            int frequencyCapPerUser,
            int frequencyCapForPeriodInMinutes,
            int frequencyCapForPeriodPerPeriod,
            long startDateTime,
            long endDateTime,
            long lastUpdatedDateTime,
            String extConfig
    ) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        ContentValues contentValues = new ContentValues();
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CAMPAIGN_ID, campaignId);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_PORTRAIT_AVAILABLE, portraitAvailable);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_LANDSCAPE_AVAILABLE, landscapeAvailable);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_EXTERNAL_ATTR, externalAttr);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_TIMEZONE_OFFSET, timezoneOffset);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_TIMEZONE_TYPE, timezoneType);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_TYPE, type);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_LAYOUT, layout);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_BG_STYLE_BG_COLOR, bgColor);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_BG_STYLE_OVERLAY_COLOR, overlayColor);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CONTENTS_TEXT, contentsText);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CONTENTS_BUTTONS, contentsButtons);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_DEFAULT_CLOSE_BUTTON_COLOR, defaultCloseButtonColor);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_DEFAULT_CLOSE_BUTTON_BG_COLOR, defaultCloseButtonBgColor);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_STICKY_BANNER_OPTION, stickyBannerOption);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_SCROLLABLE_IMAGE_OPTION, scrollableImageOption);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_PER_SESSION, frequencyCapPerSession);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_PER_USER, frequencyCapPerUser);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_FOR_PERIOD_IN_MINUTES, frequencyCapForPeriodInMinutes);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_FOR_PERIOD_PER_PERIOD, frequencyCapForPeriodPerPeriod);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_START_DATETIME, startDateTime);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_END_DATETIME, endDateTime);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_LAST_UPDATED_DATETIME, lastUpdatedDateTime);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_UNAVAILABLE_DAY, "");
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_UNAVAILABLE_TIME, 0);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_EXT_CONFIG, extConfig);

        long newRowId = db.insertWithOnConflict(InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME, null, contentValues, SQLiteDatabase.CONFLICT_REPLACE);

        if (newRowId == -1)
            AbxLog.d("Insert to InAppMessageTable fail", true);
        else
            AbxLog.d("Insert Success, newRowId in InAppMessageTable : " + newRowId, true);
    }

    public void updateOneDayCloseInAppMessageTable(
            String campaignId,
            String unavailableDay,
            long unavailableTime
    ) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        ContentValues contentValues = new ContentValues();
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_UNAVAILABLE_DAY, unavailableDay);
        contentValues.put(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_UNAVAILABLE_TIME, unavailableTime);

        db.update(InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME, contentValues,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CAMPAIGN_ID + "= ? ", new String[]{campaignId});
    }

    public void insertTriggersTable(
            String campaignId,
            String type,
            String eventName,
            int priority
    ) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        ContentValues contentValues = new ContentValues();
        contentValues.put(InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_CAMPAIGN_ID, campaignId);
        contentValues.put(InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_TYPE, type);
        contentValues.put(InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_EVENT_NAME, eventName);
        contentValues.put(InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_PRIORITY, priority);

        long newRowId = db.insertWithOnConflict(InAppMessageSqliteStore.TriggersTable.TABLE_NAME, null, contentValues, SQLiteDatabase.CONFLICT_REPLACE);

        if (newRowId == -1)
            AbxLog.d("Insert to TriggersTable fail", true);
        else
            AbxLog.d("Insert Success, newRowId in TriggersTable : " + newRowId, true);
    }

    public void insertAvailableTimeTable(
            String campaignId,
            String day,
            int startMin,
            int endMin
    ) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        ContentValues contentValues = new ContentValues();
        contentValues.put(InAppMessageSqliteStore.AvailableTimeTable.COLUMN_NAME_CAMPAIGN_ID, campaignId);
        contentValues.put(InAppMessageSqliteStore.AvailableTimeTable.COLUMN_NAME_DAY, day);
        contentValues.put(InAppMessageSqliteStore.AvailableTimeTable.COLUMN_NAME_START_MIN, startMin);
        contentValues.put(InAppMessageSqliteStore.AvailableTimeTable.COLUMN_NAME_END_MIN, endMin);

        long newRowId = db.insertWithOnConflict(InAppMessageSqliteStore.AvailableTimeTable.TABLE_NAME, null, contentValues, SQLiteDatabase.CONFLICT_REPLACE);

        if (newRowId == -1)
            AbxLog.d("insert to AvailableTimeTable fail", true);
        else
            AbxLog.d("Insert Success, newRowId in AvailableTimeTable : " + newRowId, true);
    }

    public void insertCurrentFrequencyCapTable(
            String campaignId,
            int capPerSession,
            int capPerUser,
            long lastGroupStartTime,
            int lastGroupCount
    ) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        ContentValues contentValues = new ContentValues();
        contentValues.put(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CAMPAIGN_ID, campaignId);
        contentValues.put(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CURRENT_FREQUENCY_CAP_PER_SESSION, capPerSession);
        contentValues.put(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CURRENT_FREQUENCY_CAP_PER_USER, capPerUser);
        contentValues.put(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_FREQUENCY_LAST_GROUP_START_TIME, lastGroupStartTime);
        contentValues.put(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_FREQUENCY_LAST_GROUP_COUNT, lastGroupCount);

        long newRowId = db.insertWithOnConflict(InAppMessageSqliteStore.CurrentFrequencyCapTable.TABLE_NAME, null, contentValues, SQLiteDatabase.CONFLICT_REPLACE);

        if (newRowId == -1)
            AbxLog.d("insert to CurrentFrequencyCapTable fail", true);
        else
            AbxLog.d("Insert Success, newRowId in CurrentFrequencyCapTable : " + newRowId, true);
    }

    public synchronized void insertImageTable(
            String campaignId,
            String orientation,
            String urlString,
            String imagePathString,
            String clickAction,
            boolean isFirstShow
    ) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        ContentValues contentValues = new ContentValues();
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CAMPAIGN_ID, campaignId);
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_ORIENTATION, orientation);
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_URL, urlString);
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IMAGE_PATH_STRING, imagePathString);
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CLICK_ACTION, clickAction);
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IS_FIRST_SHOW, isFirstShow ? 1 : 0);

        long newRowId = db.insertWithOnConflict(InAppMessageSqliteStore.ImageTable.TABLE_NAME, null, contentValues, SQLiteDatabase.CONFLICT_REPLACE);

        if (newRowId == -1)
            AbxLog.d("insert to ImageTable fail", true);
        else
            AbxLog.d("Insert Success, newRowId in ImageTable : " + newRowId, true);
    }

    public synchronized void updateImageTable(
            String campaignId,
            String orientation,
            String urlString,
            String imageString,
            String clickAction,
            boolean isFirstShow
    ) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        ContentValues contentValues = new ContentValues();
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CAMPAIGN_ID, campaignId);
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_ORIENTATION, orientation);
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_URL, urlString);
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IMAGE_PATH_STRING, imageString);
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CLICK_ACTION, clickAction);
        contentValues.put(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IS_FIRST_SHOW, isFirstShow ? 1 : 0);

        long numberOfRowsAffected = db.update(InAppMessageSqliteStore.ImageTable.TABLE_NAME, contentValues,
                InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CAMPAIGN_ID + " = ? AND " +
                        InAppMessageSqliteStore.ImageTable.COLUMN_NAME_ORIENTATION + " = ?",
                new String[]{campaignId, orientation});

        if (numberOfRowsAffected == 0) {
            AbxLog.d("updateImageTable Fail", true);
        }
    }

    public void deleteInAppMessageTable(String campaignId) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        int deletedRowCount = db.delete(InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME, InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CAMPAIGN_ID + "= '" + campaignId + "'", null);

        AbxLog.d("deleteInAppMessageTable: Deleted row count = " + deletedRowCount, true);
    }

    public void deleteAllInAppMessageTable() {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        int deletedRowCount = db.delete(InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME, null, null);

        AbxLog.d("deleteAllInAppMessageTable: Deleted row count = " + deletedRowCount, true);
    }


    public void deleteAvailableTimeTable(String campaignId) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        int deletedRowCount = db.delete(InAppMessageSqliteStore.AvailableTimeTable.TABLE_NAME, InAppMessageSqliteStore.AvailableTimeTable.COLUMN_NAME_CAMPAIGN_ID + "= '" + campaignId + "'", null);


        AbxLog.d("deleteAvailableTimeTable: Deleted row count = " + deletedRowCount, true);
    }

    public void deleteAllAvailableTimeTable() {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        int deletedRowCount = db.delete(InAppMessageSqliteStore.AvailableTimeTable.TABLE_NAME, null, null);


        AbxLog.d("deleteAllAvailableTimeTable: Deleted row count = " + deletedRowCount, true);
    }


    public void deleteTriggersTable(String campaignId) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        int deletedRowCount = db.delete(InAppMessageSqliteStore.TriggersTable.TABLE_NAME, InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_CAMPAIGN_ID + "= '" + campaignId + "'", null);


        AbxLog.d("deleteTriggersTable: Deleted row count = " + deletedRowCount, true);
    }

    public void deleteAllTriggersTable() {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        int deletedRowCount = db.delete(InAppMessageSqliteStore.TriggersTable.TABLE_NAME, null, null);


        AbxLog.d("deleteAllTriggersTable: Deleted row count = " + deletedRowCount, true);
    }

    public void deleteCurrentFrequencyCapTable(String campaignId) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        int deletedRowCount = db.delete(InAppMessageSqliteStore.CurrentFrequencyCapTable.TABLE_NAME, InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CAMPAIGN_ID + "= '" + campaignId + "'", null);


        AbxLog.d("deleteCurrentFrequencyCapTable: Deleted row count = " + deletedRowCount, true);
    }

    public void deleteAllCurrentFrequencyCapTable() {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        int deletedRowCount = db.delete(InAppMessageSqliteStore.CurrentFrequencyCapTable.TABLE_NAME, null, null);

        AbxLog.d("deleteAllCurrentFrequencyCapTable: Deleted row count = " + deletedRowCount, true);
    }

    public void deleteImageTable(String campaignId) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        String portraitImagePathQueue = getImagePathFromImageTable(campaignId, IAMConstants.PORTRAIT);
        String landscapeImagePathQueue = getImagePathFromImageTable(campaignId, IAMConstants.LANDSCAPE);

        deleteImageFromInternalStorage(context, portraitImagePathQueue);
        deleteImageFromInternalStorage(context, landscapeImagePathQueue);

        int deletedRowCount = db.delete(InAppMessageSqliteStore.ImageTable.TABLE_NAME, InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CAMPAIGN_ID + "= '" + campaignId + "'", null);

        AbxLog.d("deleteImageTable: Deleted row count = " + deletedRowCount, true);
    }

    public void deleteAllImageTable() {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        int deletedRowCount = db.delete(InAppMessageSqliteStore.ImageTable.TABLE_NAME, null, null);

        AbxLog.d("deleteAllImageTable: Deleted row count = " + deletedRowCount, true);
    }

    public void deleteAllImageStorage() throws Exception {
        Queue<String> imagePathQueue;

        try {
            imagePathQueue = getAllImagePathFromImageTable();
        } catch (Exception e) {
            throw new Exception("InAppMessage DB is already deleted");
        }

        if (imagePathQueue != null) {
            while (!imagePathQueue.isEmpty()) {
                String imagePath = imagePathQueue.poll();
                if(!TextUtils.isEmpty(imagePath)){
                    deleteImageFromInternalStorage(context, imagePath);
                }
            }
        }
    }

    public String getImageUrlFromImageTable(String campaignId, String orientation) {
        SQLiteDatabase db = dbHelper.getReadableDatabase();

        String[] projection = {
                InAppMessageSqliteStore.ImageTable.COLUMN_NAME_URL,
        };

        String selection;

        selection = InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CAMPAIGN_ID + " = ? AND "
                + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_ORIENTATION + " = ? ";

        String[] selectionArgs = {campaignId, orientation};

        Cursor cursor = db.query(
                InAppMessageSqliteStore.ImageTable.TABLE_NAME,
                projection,
                selection,
                selectionArgs,
                null,
                null,
                null
        );

        if (cursor.moveToNext()) {
            String imageUrl = cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_URL));
            if (imageUrl != null) {
                return imageUrl;
            }
        }

        cursor.close();

        return "";
    }

    public String getImagePathFromImageTable(String campaignId, String orientation) {
        SQLiteDatabase db = dbHelper.getReadableDatabase();

        String[] projection = {
                InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IMAGE_PATH_STRING,
        };

        String selection;

        selection = InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CAMPAIGN_ID + " = ? AND "
                + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_ORIENTATION + " = ? ";

        String[] selectionArgs = {campaignId, orientation};

        Cursor cursor = db.query(
                InAppMessageSqliteStore.ImageTable.TABLE_NAME,
                projection,
                selection,
                selectionArgs,
                null,
                null,
                null
        );

        if (cursor.moveToNext()) {
            String imagePath = cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IMAGE_PATH_STRING));
            return imagePath;
        }

        cursor.close();

        return null;
    }

    public Queue<String> getAllImagePathFromImageTable() {
        SQLiteDatabase db = dbHelper.getReadableDatabase();

        String[] projection = {
                InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IMAGE_PATH_STRING,
        };

        Cursor cursor = db.query(
                InAppMessageSqliteStore.ImageTable.TABLE_NAME,
                projection,
                null,
                null,
                null,
                null,
                null
        );

        Queue<String> imagePaths = new ConcurrentLinkedQueue<>();

        while (cursor.moveToNext()) {
            String imagePath = cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IMAGE_PATH_STRING));
            imagePaths.add(imagePath);
        }

        cursor.close();

        return imagePaths;
    }

    public void deleteImageTableWithOrientation(String campaignId, String orientation) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        String imagePath = getImagePathFromImageTable(campaignId, orientation);

        if (imagePath != null) {
            boolean isInternalStorageDeleted = deleteImageFromInternalStorage(context, imagePath);
            AbxLog.d("deleteImageTableWithOrientation: isInternalStorageDeleted" + isInternalStorageDeleted, true);
        }

        int deletedRowCount = db.delete(InAppMessageSqliteStore.ImageTable.TABLE_NAME,
                InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CAMPAIGN_ID + "= '" + campaignId + "' "
                        + " AND " + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_ORIENTATION + "= '" + orientation + "' ", null);

        AbxLog.d("deleteImageTableWithOrientation: Deleted row count = " + deletedRowCount, true);
    }


    public void deleteCampaign(String campaignId) {
        deleteInAppMessageTable(campaignId);
        deleteAvailableTimeTable(campaignId);
        deleteTriggersTable(campaignId);
        deleteCurrentFrequencyCapTable(campaignId);
        deleteImageTable(campaignId);
    }

    public void deletePerUserFrequencyExpiredCampaign(String campaignId) {
        deleteInAppMessageTable(campaignId);
        deleteAvailableTimeTable(campaignId);
        deleteTriggersTable(campaignId);
        deleteImageTable(campaignId);
    }

    public void deleteBeforeInsertApiResponse() {
        deleteAllInAppMessageTable();
        deleteAllTriggersTable();
        deleteAllAvailableTimeTable();
    }

    public void deleteAllInAppMessageDBContents() {
        try {
            deleteAllImageStorage();
            deleteAllImageTable();
            deleteAllInAppMessageTable();
            deleteAllTriggersTable();
            deleteAllAvailableTimeTable();
            deleteAllCurrentFrequencyCapTable();

            dataRegistry.putDataRegistry(new DataUnit(
                    DataRegistryKey.INT_IN_APP_MESSAGE_MINUTES_TO_EXPIRY,
                    0,
                    5,
                    this.getClass().getName(),
                    true
            ));

            dataRegistry.putDataRegistry(new DataUnit(
                    DataRegistryKey.STRING_IN_APP_MESSAGE_CHECKSUM,
                    null,
                    5,
                    this.getClass().getName(),
                    true
            ));
        } catch (Exception e) {
            AbxLog.e(e, true);
        }
    }

    public void deleteStoppedOrDeletedInAppMessage(List<String> responseCampaignIdArrayList) {
        SQLiteDatabase db = dbHelper.getReadableDatabase();

        String getCampaignIdFromCurrentFrequencyCapTableQuery = "SELECT " + InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CAMPAIGN_ID
                + " FROM " + InAppMessageSqliteStore.CurrentFrequencyCapTable.TABLE_NAME;

        Cursor campaignIdCursor = db.rawQuery(getCampaignIdFromCurrentFrequencyCapTableQuery, null);

        while (campaignIdCursor.moveToNext()) {
            String campaignId = campaignIdCursor.getString(campaignIdCursor.getColumnIndex(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CAMPAIGN_ID));

            if (!responseCampaignIdArrayList.contains(campaignId)) {
                deleteCampaign(campaignId);
            }
        }

        campaignIdCursor.close();
    }

    public void deleteDB() {
        imageDownloadExecutorService.shutdownNow();

        try {
            deleteAllImageStorage();
        } catch (Exception e) {
            AbxLog.d("InAppMessage DB is already deleted", true);
            return;
        }

        dbHelper.deleteDB();
    }

    public void restartInAppMessage() {
        imageDownloadExecutorService = Executors.newSingleThreadExecutor();
        dbHelper.restartDB();
    }

    public boolean isCampaignIdTimeAvailable(String campaignId) {
        SQLiteDatabase db = dbHelper.getReadableDatabase();

        String getTimeInfoQuery = "SELECT " + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_TIMEZONE_OFFSET + ", "
                + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_TIMEZONE_TYPE + ", "
                + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_START_DATETIME + ", "
                + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_END_DATETIME + ", "
                + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_UNAVAILABLE_DAY + ", "
                + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_UNAVAILABLE_TIME
                + " FROM " + InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME
                + " WHERE " + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CAMPAIGN_ID + " = '" + campaignId + "'";

        Cursor getTimeCursor = db.rawQuery(getTimeInfoQuery, null);

        long startDateTime;
        long endDateTime;
        String timezoneType;
        int timezoneOffset;
        String currentDay;
        String currentMin;
        long currentTimeMillis;
        String unavailableDay;
        long unavailableTime;

        if (getTimeCursor.moveToNext()) {
            startDateTime = getTimeCursor.getLong(getTimeCursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_START_DATETIME));
            endDateTime = getTimeCursor.getLong(getTimeCursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_END_DATETIME));
            timezoneType = getTimeCursor.getString(getTimeCursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_TIMEZONE_TYPE));
            timezoneOffset = getTimeCursor.getInt(getTimeCursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_TIMEZONE_OFFSET));
            unavailableDay = getTimeCursor.getString(getTimeCursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_UNAVAILABLE_DAY));
            unavailableTime = getTimeCursor.getLong(getTimeCursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_UNAVAILABLE_TIME));

            if (timezoneType.equals(IAMConstants.TIMEZONE_GLOBAL)) {
                Calendar utcCalendar = Calendar.getInstance(TimeZone.getTimeZone("UTC"));
                SimpleDateFormat utcSimpleDateFormat = new SimpleDateFormat("EE", Locale.ENGLISH);
                utcSimpleDateFormat.setTimeZone(TimeZone.getTimeZone("UTC"));

                currentTimeMillis = System.currentTimeMillis();
                currentDay = utcSimpleDateFormat.format(System.currentTimeMillis() + timezoneOffset * 3600 * 1000);
                currentMin = Integer.toString((utcCalendar.get(Calendar.MINUTE) + (utcCalendar.get(Calendar.HOUR_OF_DAY) + timezoneOffset) * 60) % 1440);
            } else if (timezoneType.equals(IAMConstants.TIMEZONE_LOCAL)) {
                Calendar localCalendar = Calendar.getInstance();
                SimpleDateFormat localSimpleDateFormat = new SimpleDateFormat("EE", Locale.ENGLISH);

                currentTimeMillis = System.currentTimeMillis();
                currentDay = localSimpleDateFormat.format(System.currentTimeMillis());
                currentMin = Integer.toString(localCalendar.get(Calendar.MINUTE) + localCalendar.get(Calendar.HOUR_OF_DAY) * 60);
            } else {
                AbxLog.d("isCampaignIdTimeAvailable: invalid timezoneOffset: " + timezoneOffset, true);
                return false;
            }

            //one day close check
            if (currentDay.equals(unavailableDay) && unavailableTime + 2 * 24 * 3600 * 1000 > currentTimeMillis) {
                AbxLog.d("isCampaignIdTimeAvailable: one day close ", true);
                return false;
            } else {
                updateOneDayCloseInAppMessageTable(campaignId, "", 0);
            }


            //start, end time available check
            if (currentTimeMillis < startDateTime) {
                AbxLog.d("isCampaignIdTimeAvailable: currentTimeMillis is earlier than startDateTime", true);
                return false;
            } else if (endDateTime < currentTimeMillis && endDateTime != 0) {
                AbxLog.d("isCampaignIdTimeAvailable: currentTimeMillis over endDateTime", true);
                deleteCampaign(campaignId);
                return false;
            }
        } else {
            AbxLog.d("isCampaignIdTimeAvailable: campaignId dateTime inAvailable", true);
            return false;
        }

        if (currentDay.equals("Thu")) {
            currentDay = "Thur";
        }

        AbxLog.d("currentDay: " + currentDay + ", currentMin: " + currentMin, true);

        String isCampaignIdValidQuery = "SELECT * FROM "
                + InAppMessageSqliteStore.AvailableTimeTable.TABLE_NAME
                + " WHERE " + InAppMessageSqliteStore.AvailableTimeTable.TABLE_NAME + "." + InAppMessageSqliteStore.AvailableTimeTable.COLUMN_NAME_CAMPAIGN_ID + " = '" + campaignId + "' "
                + " AND (" + InAppMessageSqliteStore.AvailableTimeTable.TABLE_NAME + "." + InAppMessageSqliteStore.AvailableTimeTable.COLUMN_NAME_DAY + " = '" + currentDay + "'"
                + " AND " + InAppMessageSqliteStore.AvailableTimeTable.TABLE_NAME + "." + InAppMessageSqliteStore.AvailableTimeTable.COLUMN_NAME_START_MIN + " <= " + currentMin
                + " AND " + InAppMessageSqliteStore.AvailableTimeTable.TABLE_NAME + "." + InAppMessageSqliteStore.AvailableTimeTable.COLUMN_NAME_END_MIN + " >= " + currentMin
                + " OR " + InAppMessageSqliteStore.AvailableTimeTable.TABLE_NAME + "." + InAppMessageSqliteStore.AvailableTimeTable.COLUMN_NAME_DAY + " = '" + IAMConstants.AVAILABLE_TIME_ALL + "')";


        Cursor cursor = db.rawQuery(isCampaignIdValidQuery, null);

        boolean isCampaignIdValid = cursor.moveToNext();

        getTimeCursor.close();

        cursor.close();

        return isCampaignIdValid;
    }

    public boolean isCampaignIdFrequencyAvailable(String campaignId) {
        SQLiteDatabase db = dbHelper.getReadableDatabase();

        String isCampaignIdFrequencyAvailableQuery = "SELECT " + InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_FREQUENCY_LAST_GROUP_COUNT + ", "
                + InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_FREQUENCY_LAST_GROUP_START_TIME + ", "
                + InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CURRENT_FREQUENCY_CAP_PER_SESSION + ", "
                + InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CURRENT_FREQUENCY_CAP_PER_USER + ", "
                + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_FOR_PERIOD_IN_MINUTES + ", "
                + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_FOR_PERIOD_PER_PERIOD + ", "
                + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_PER_USER + ", "
                + InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CURRENT_FREQUENCY_CAP_PER_USER
                + " FROM " + InAppMessageSqliteStore.CurrentFrequencyCapTable.TABLE_NAME
                + " JOIN " + InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME
                + " ON " + InAppMessageSqliteStore.CurrentFrequencyCapTable.TABLE_NAME + "." + InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CAMPAIGN_ID
                + " = " + InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME + "." + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CAMPAIGN_ID
                + " WHERE " + InAppMessageSqliteStore.CurrentFrequencyCapTable.TABLE_NAME + "." + InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CAMPAIGN_ID + " = '" + campaignId + "' "
                + " AND (" + InAppMessageSqliteStore.CurrentFrequencyCapTable.TABLE_NAME + "." + InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CURRENT_FREQUENCY_CAP_PER_SESSION + " "
                + " < " + InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME + "." + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_PER_SESSION
                + " OR " + InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME + "." + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_PER_SESSION + " = 0" //0일 경우 무제한 노출
                + " )";

        AbxLog.d("isCampaignIdFrequencyAvailableQuery: " + isCampaignIdFrequencyAvailableQuery, true);
        Cursor cursor = db.rawQuery(isCampaignIdFrequencyAvailableQuery, null);


        if (cursor.moveToNext()) {
            long lastGroupStartTime = cursor.getLong(cursor.getColumnIndex(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_FREQUENCY_LAST_GROUP_START_TIME));
            int lastGroupCount = cursor.getInt(cursor.getColumnIndex(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_FREQUENCY_LAST_GROUP_COUNT));
            int currentSessionCount = cursor.getInt(cursor.getColumnIndex(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CURRENT_FREQUENCY_CAP_PER_SESSION));
            int currentUserCount = cursor.getInt(cursor.getColumnIndex(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CURRENT_FREQUENCY_CAP_PER_USER));
            int currentPerUser = cursor.getInt(cursor.getColumnIndex(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CURRENT_FREQUENCY_CAP_PER_USER));
            int periodInMinutes = cursor.getInt(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_FOR_PERIOD_IN_MINUTES));
            int periodPerPeriod = cursor.getInt(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_FOR_PERIOD_PER_PERIOD));
            int perUser = cursor.getInt(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_FREQUENCY_CAP_PER_USER));
            long currentTimeInMillis = System.currentTimeMillis();

            cursor.close();

            //perUser valid check
            if (perUser != 0 && currentPerUser >= perUser) {
                AbxLog.d("isCampaignIdFrequencyAvailable: frequency perUser is invalid", true);
                deletePerUserFrequencyExpiredCampaign(campaignId);
                return false;
            }

            //perperiod valid check
            if (periodPerPeriod == 0) { //0일 경우 무제한 노출
                insertCurrentFrequencyCapTable(campaignId, currentSessionCount + 1, currentUserCount + 1, currentTimeInMillis, lastGroupCount + 1);
                AbxLog.d("isCampaignIdFrequencyAvailable: frequency is valid", true);
                return true;
            } else if (lastGroupStartTime + periodInMinutes * 60 * 1000 < currentTimeInMillis) {
                insertCurrentFrequencyCapTable(campaignId, currentSessionCount + 1, currentUserCount + 1, currentTimeInMillis, 1);
                AbxLog.d("isCampaignIdFrequencyAvailable: frequency is valid", true);
                return true;
            } else if (lastGroupCount < periodPerPeriod) {
                insertCurrentFrequencyCapTable(campaignId, currentSessionCount + 1, currentUserCount + 1, lastGroupStartTime, lastGroupCount + 1);
                AbxLog.d("isCampaignIdFrequencyAvailable: frequency is valid", true);
                return true;
            } else {
                AbxLog.d("isCampaignIdFrequencyAvailable: frequency period is invalid", true);
                return false;
            }
        } else {
            cursor.close();
            AbxLog.d("isCampaignIdFrequencyAvailable: frequency perSession is invalid", true);
            return false;
        }
    }

    public boolean isCurrentFrequencyExist(String campaignId) {
        SQLiteDatabase db = dbHelper.getReadableDatabase();

        String isCurrentFrequencyExistQuery = "SELECT * FROM " + InAppMessageSqliteStore.CurrentFrequencyCapTable.TABLE_NAME
                + " WHERE " + InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CAMPAIGN_ID + " = '" + campaignId + "'";

        AbxLog.d("isCurrentFrequencyExistQuery: " + isCurrentFrequencyExistQuery, true);
        Cursor cursor = db.rawQuery(isCurrentFrequencyExistQuery, null);
        boolean isExist = cursor.moveToNext();
        cursor.close();
        return isExist;
    }

    public void resetFrequencyCapPerSession() {
        SQLiteDatabase db = dbHelper.getWritableDatabase();

        ContentValues contentValues = new ContentValues();
        contentValues.put(InAppMessageSqliteStore.CurrentFrequencyCapTable.COLUMN_NAME_CURRENT_FREQUENCY_CAP_PER_SESSION, 0);
        int updateResult = db.update(InAppMessageSqliteStore.CurrentFrequencyCapTable.TABLE_NAME, contentValues, null, null);

        if (updateResult == -1) {
            AbxLog.d("resetFrequencyCapPerSessionQuery: resetFrequencyCapPerSession Fail!", true);
        } else {
            AbxLog.d("resetFrequencyCapPerSessionQuery: resetFrequencyCapPerSession SUCCESS", true);
        }
    }

    public DfnInAppMessage getInAppMessageByEventName(String triggeredEventName, boolean isPortrait) {
        HashMap<String, Pair<Long, Long>> triggeredCampaignIdMap = getCampaignIdFromTriggerEvent(triggeredEventName);

        for (String campaignId : sortMapByValue(triggeredCampaignIdMap)) {
            if (!isCampaignIdTimeAvailable(campaignId))
                continue;

            DfnInAppMessage inAppMessage = getInAppMessageByCampaignId(campaignId, isPortrait);

            if (inAppMessage == null) {
                AbxLog.d("getInAppMessageByCampaignId Fail, campaignId = " + campaignId, true);
            } else {

                if (!isCampaignIdFrequencyAvailable(campaignId))
                    continue;

                AbxLog.d("getInAppMessageByCampaignId Success, campaignId = " + campaignId, true);

                return inAppMessage;
            }
        }

        AbxLog.d("getInAppMessageByCampaignId Fail, No Valid InAppMessage", true);

        return null;
    }

    public HashMap<String, Pair<Long, Long>> getCampaignIdFromTriggerEvent(String triggeredEventName) {
        HashMap<String, Pair<Long, Long>> campaignIdHashMap = new HashMap<>();

        SQLiteDatabase db = dbHelper.getReadableDatabase();

        String getCampaignIdWithPriority = "SELECT "
                + InAppMessageSqliteStore.TriggersTable.TABLE_NAME + "." + InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_CAMPAIGN_ID + ", "
                + InAppMessageSqliteStore.TriggersTable.TABLE_NAME + "." + InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_PRIORITY + ", "
                + InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME + "." + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_LAST_UPDATED_DATETIME
                + " FROM " + InAppMessageSqliteStore.TriggersTable.TABLE_NAME
                + " JOIN " + InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME
                + " ON " + InAppMessageSqliteStore.TriggersTable.TABLE_NAME + "." + InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_CAMPAIGN_ID
                + " = " + InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME + "." + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CAMPAIGN_ID
                + " WHERE " + InAppMessageSqliteStore.TriggersTable.TABLE_NAME + "." + InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_EVENT_NAME + " = '" + triggeredEventName + "'"
                + " AND " + InAppMessageSqliteStore.TriggersTable.TABLE_NAME + "." + InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_TYPE + " = 'inappevent'";

        AbxLog.d("getCampaignIdHashMapFromTriggerEvent query:" + getCampaignIdWithPriority, true);

        Cursor cursor = db.rawQuery(getCampaignIdWithPriority, null);

        while (cursor.moveToNext()) {
            campaignIdHashMap.put(
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_CAMPAIGN_ID)),
                    new Pair<>(
                            cursor.getLong(cursor.getColumnIndex(InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_PRIORITY)),
                            cursor.getLong(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_LAST_UPDATED_DATETIME))
                    )
            );
        }

        cursor.close();
        return campaignIdHashMap;
    }

    public Result<List<DfnInAppMessage>> getAllInAppMessage() {

        List<DfnInAppMessage> inAppMessages = new ArrayList<>();
        try {
            SQLiteDatabase db = dbHelper.getReadableDatabase();
            String[] projection = {
                    InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CAMPAIGN_ID,
                    InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_PORTRAIT_AVAILABLE,
                    InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_LANDSCAPE_AVAILABLE,
                    InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_TIMEZONE_OFFSET,
                    InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_TIMEZONE_TYPE,
                    InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_TYPE,
                    InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_LAYOUT,
                    InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_HTML,
                    InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_BG_STYLE_BG_COLOR,
                    InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_BG_STYLE_OVERLAY_COLOR,
                    InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CONTENTS_TEXT,
                    InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CONTENTS_BUTTONS,
                    InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_DEFAULT_CLOSE_BUTTON_COLOR,
                    InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_DEFAULT_CLOSE_BUTTON_BG_COLOR,
                    InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_STICKY_BANNER_OPTION,
                    InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_SCROLLABLE_IMAGE_OPTION,
                    InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_START_DATETIME,
                    InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_END_DATETIME,
                    InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_EXTERNAL_ATTR,
                    InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_EXT_CONFIG
            };

            Cursor cursor = db.query(
                    InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME,
                    projection,
                    null,
                    null,
                    null,
                    null,
                    null
            );

            while (cursor.moveToNext()) {
                String campaignId = cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CAMPAIGN_ID));

                HashMap<String, Object> imageInfoPortrait = getImageInfo(campaignId, IAMConstants.PORTRAIT, false);
                HashMap<String, Object> imageInfoLandscape = getImageInfo(campaignId, IAMConstants.LANDSCAPE, false);

                String imageAction = null;
                if (imageInfoPortrait != null) {
                    imageAction = (String) imageInfoPortrait.get(IAMEnums.ImageInfoHashmapValue.CLICK_ACTION.getValue());
                }

                DfnInAppMessage inAppMessage = new DfnInAppMessage(
                        campaignId,
                        cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_TYPE)),
                        cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_LAYOUT)),
                        imageInfoPortrait == null ? null : (Bitmap) imageInfoPortrait.get(IAMEnums.ImageInfoHashmapValue.IMAGE_BITMAP.getValue()),
                        imageInfoPortrait == null ? null : (String) imageInfoPortrait.get(IAMEnums.ImageInfoHashmapValue.IMAGE_URL.getValue()),
                        imageInfoPortrait == null ? null : (Boolean) imageInfoPortrait.get(IAMEnums.ImageInfoHashmapValue.IS_DOWNLOADED.getValue()),
                        imageInfoLandscape == null ? null : (Bitmap) imageInfoLandscape.get(IAMEnums.ImageInfoHashmapValue.IMAGE_BITMAP.getValue()),
                        imageInfoLandscape == null ? null : (String) imageInfoLandscape.get(IAMEnums.ImageInfoHashmapValue.IMAGE_URL.getValue()),
                        imageInfoLandscape == null ? null : (Boolean) imageInfoLandscape.get(IAMEnums.ImageInfoHashmapValue.IS_DOWNLOADED.getValue()),
                        imageAction,
                        cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_BG_STYLE_BG_COLOR)),
                        cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_BG_STYLE_OVERLAY_COLOR)),
                        cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CONTENTS_TEXT)),
                        cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CONTENTS_BUTTONS)),
                        cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_DEFAULT_CLOSE_BUTTON_COLOR)),
                        cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_DEFAULT_CLOSE_BUTTON_BG_COLOR)),
                        cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_STICKY_BANNER_OPTION)),
                        cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_SCROLLABLE_IMAGE_OPTION)),
                        cursor.getLong(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_START_DATETIME)),
                        cursor.getLong(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_END_DATETIME)),
                        CommonUtils.convertNullStringToNull(cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_EXTERNAL_ATTR))),
                        CommonUtils.convertNullStringToNull(cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_EXT_CONFIG))),
                        getTriggers(campaignId)
                );

                inAppMessages.add(inAppMessage);
            }

            cursor.close();

            return Success.of(inAppMessages);
        } catch (Exception e) {
            return Error.of(e);
        }
    }

    public DfnInAppMessage getInAppMessageByCampaignId(String campaignId, boolean isPortrait) {
        DfnInAppMessage inAppMessage = null;

        SQLiteDatabase db = dbHelper.getReadableDatabase();

        // Define a projection that specifies which columns from the database
        // you will actually use after this query.
        String[] projection = {
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_TYPE,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_LAYOUT,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_BG_STYLE_BG_COLOR,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_BG_STYLE_OVERLAY_COLOR,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CONTENTS_TEXT,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CONTENTS_BUTTONS,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_DEFAULT_CLOSE_BUTTON_COLOR,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_DEFAULT_CLOSE_BUTTON_BG_COLOR,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_STICKY_BANNER_OPTION,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_SCROLLABLE_IMAGE_OPTION,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_START_DATETIME,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_END_DATETIME,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_EXTERNAL_ATTR,
                InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_EXT_CONFIG
        };

        String selection;
        if (isPortrait) {
            selection = InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CAMPAIGN_ID + " = ? AND "
                    + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_PORTRAIT_AVAILABLE + " = ? ";
        } else {
            selection = InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CAMPAIGN_ID + " = ? AND "
                    + InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_LANDSCAPE_AVAILABLE + " = ? ";
        }

        String[] selectionArgs = {campaignId, Integer.toString(1)};

        Cursor cursor = db.query(
                InAppMessageSqliteStore.InAppMessageTable.TABLE_NAME,
                projection,
                selection,
                selectionArgs,
                null,
                null,
                null
        );

        if (cursor.moveToNext()) {
            HashMap<String, Object> imageInfoPortrait = null;
            HashMap<String, Object> imageInfoLandscape = null;
            String imageAction;
            if (isPortrait) {
                imageInfoPortrait = getImageInfo(campaignId, IAMConstants.PORTRAIT, true);
                imageAction = imageInfoPortrait == null ? null : (String) imageInfoPortrait.get(IAMEnums.ImageInfoHashmapValue.CLICK_ACTION.getValue());
            } else {
                imageInfoLandscape = getImageInfo(campaignId, IAMConstants.LANDSCAPE, true);
                imageAction = imageInfoLandscape == null ? null : (String) imageInfoLandscape.get(IAMEnums.ImageInfoHashmapValue.CLICK_ACTION.getValue());
            }
            inAppMessage = new DfnInAppMessage(
                    campaignId,
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_TYPE)),
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_LAYOUT)),
                    imageInfoPortrait == null ? null : (Bitmap) imageInfoPortrait.get(IAMEnums.ImageInfoHashmapValue.IMAGE_BITMAP.getValue()),
                    imageInfoPortrait == null ? null : (String) imageInfoPortrait.get(IAMEnums.ImageInfoHashmapValue.IMAGE_URL.getValue()),
                    imageInfoPortrait == null ? null : (Boolean) imageInfoPortrait.get(IAMEnums.ImageInfoHashmapValue.IS_DOWNLOADED.getValue()),
                    imageInfoLandscape == null ? null : (Bitmap) imageInfoLandscape.get(IAMEnums.ImageInfoHashmapValue.IMAGE_BITMAP.getValue()),
                    imageInfoLandscape == null ? null : (String) imageInfoLandscape.get(IAMEnums.ImageInfoHashmapValue.IMAGE_URL.getValue()),
                    imageInfoLandscape == null ? null : (Boolean) imageInfoLandscape.get(IAMEnums.ImageInfoHashmapValue.IS_DOWNLOADED.getValue()),
                    imageAction,
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_BG_STYLE_BG_COLOR)),
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_BG_STYLE_OVERLAY_COLOR)),
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CONTENTS_TEXT)),
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_CONTENTS_BUTTONS)),
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_DEFAULT_CLOSE_BUTTON_COLOR)),
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_DEFAULT_CLOSE_BUTTON_BG_COLOR)),
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_STICKY_BANNER_OPTION)),
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_SCROLLABLE_IMAGE_OPTION)),
                    cursor.getLong(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_START_DATETIME)),
                    cursor.getLong(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_END_DATETIME)),
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_EXTERNAL_ATTR)),
                    cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.InAppMessageTable.COLUMN_NAME_EXT_CONFIG)),
                    new ArrayList<>()
            );
        }

        cursor.close();

        return inAppMessage;
    }

    public synchronized HashMap<String, Object> getAndSaveImageInfo(
            String campaignId,
            String orientation,
            boolean isImageBeUsed
    ) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();
        HashMap<String, Object> imageHashMap = new HashMap<>();
        Bitmap bitmapImage = null;

        String getImageQuery = "SELECT "
                + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IMAGE_PATH_STRING + ", "
                + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CLICK_ACTION + ", "
                + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_URL + ", "
                + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IS_FIRST_SHOW
                + " FROM "
                + InAppMessageSqliteStore.ImageTable.TABLE_NAME
                + " WHERE " + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CAMPAIGN_ID + " = '" + campaignId + "'"
                + " AND " + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_ORIENTATION + " = '" + orientation + "'";

        Cursor cursor = db.rawQuery(getImageQuery, null);

        if (cursor.moveToNext()) {
            String imagePathString = cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IMAGE_PATH_STRING));
            String clickAction = cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CLICK_ACTION));
            String imageUrlString = cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_URL));
            boolean isDownloaded = cursor.getInt(cursor.getColumnIndex(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IS_FIRST_SHOW)) == 1;

            cursor.close();

            if (imagePathString.equals(IAMConstants.NO_IMAGE)) {
                bitmapImage = HttpConnectionUtils.getBitmapFromURL(imageUrlString, 0);
                if (bitmapImage != null){
                    imagePathString = saveToInternalStorage(bitmapImage, context);
                }

                if (bitmapImage == null) {
                    AbxLog.d("InAppMessageDAO getAndSaveImageInfo: bitmapImage is null", true);
                    return null;
                } else if (imagePathString == null) {
                    AbxLog.d("InAppMessageDAO getAndSaveImageInfo: bitmapImage is null", true);
                    return null;
                }

                updateImageTable(campaignId, orientation, imageUrlString, imagePathString, clickAction, !isImageBeUsed);

                isDownloaded = true;
            } else if (isDownloaded && isImageBeUsed) {
                updateImageTable(campaignId, orientation, imageUrlString, imagePathString, clickAction, false);
                bitmapImage = getImageBitmapFromInternalStorage(context, imagePathString);
            } else {
                bitmapImage = getImageBitmapFromInternalStorage(context, imagePathString);
            }

            if (bitmapImage == null)
                return null;

            imageHashMap.put(IAMEnums.ImageInfoHashmapValue.IMAGE_BITMAP.getValue(), bitmapImage);
            imageHashMap.put(IAMEnums.ImageInfoHashmapValue.IMAGE_URL.getValue(), imageUrlString);
            imageHashMap.put(IAMEnums.ImageInfoHashmapValue.IS_DOWNLOADED.getValue(), isDownloaded);
            imageHashMap.put(IAMEnums.ImageInfoHashmapValue.CLICK_ACTION.getValue(), clickAction);

            return imageHashMap;
        } else {
            AbxLog.d("InAppMessage No image with campaignId = " + campaignId + ", orientation = " + orientation, true);
            cursor.close();

            return null;
        }
    }

    public HashMap<String, Object> getImageInfo(
            String campaignId,
            String orientation,
            boolean isImageBeUsed
    ) {
        SQLiteDatabase db = dbHelper.getWritableDatabase();
        HashMap<String, Object> imageHashMap = new HashMap<>();

        String getImageQuery = "SELECT "
                + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IMAGE_PATH_STRING + ", "
                + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CLICK_ACTION + ", "
                + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_URL + ", "
                + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IS_FIRST_SHOW
                + " FROM "
                + InAppMessageSqliteStore.ImageTable.TABLE_NAME
                + " WHERE " + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CAMPAIGN_ID + " = '" + campaignId + "'"
                + " AND " + InAppMessageSqliteStore.ImageTable.COLUMN_NAME_ORIENTATION + " = '" + orientation + "'";

        Cursor cursor = db.rawQuery(getImageQuery, null);

        if (cursor.moveToNext()) {
            String imagePathString = cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IMAGE_PATH_STRING));
            String clickAction = cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_CLICK_ACTION));
            String imageUrlString = cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_URL));
            boolean isDownloaded = cursor.getInt(cursor.getColumnIndex(InAppMessageSqliteStore.ImageTable.COLUMN_NAME_IS_FIRST_SHOW)) == 1;

            cursor.close();

            if (isDownloaded && isImageBeUsed) {
                updateImageTable(campaignId, orientation, imageUrlString, imagePathString, clickAction, false);
            }

            Bitmap imageBitmap = getImageBitmapFromInternalStorage(context, imagePathString);

            if (imageBitmap == null) {
                AbxLog.d("InAppMessage getImageInfo : can not get image from internal storage", true);
            }

            imageHashMap.put(IAMEnums.ImageInfoHashmapValue.IMAGE_BITMAP.getValue(), imageBitmap);
            imageHashMap.put(IAMEnums.ImageInfoHashmapValue.IMAGE_URL.getValue(), imageUrlString);
            imageHashMap.put(IAMEnums.ImageInfoHashmapValue.IS_DOWNLOADED.getValue(), isDownloaded);
            imageHashMap.put(IAMEnums.ImageInfoHashmapValue.CLICK_ACTION.getValue(), clickAction);

            return imageHashMap;
        } else {
            AbxLog.d("InAppMessage No image with campaignId = " + campaignId + ", orientation = " + orientation, true);
            cursor.close();

            return null;
        }
    }

    public void closeDataBase() {
        dbHelper.getWritableDatabase().close();
    }

    public ArrayList<String> sortMapByValue(final HashMap<String, Pair<Long, Long>> map) {
        ArrayList<String> list = new ArrayList<>(map.keySet());
        Collections.sort(list, (key1, key2) -> {
            Pair<Long, Long> value1 = map.get(key1);
            Pair<Long, Long> value2 = map.get(key2);

            if (value1 == null || value2 == null) {
                AbxLog.e("sorMapByValue null value error", true);
                return 1;
            }

            if (value1.first.equals(value2.first)) {
                if (value1.second < value2.second)
                    return 1;
                else
                    return -1;
            } else {
                if (value1.first < value2.first)
                    return 1;
                else
                    return -1;
            }
        });

        return list;
    }

    public String saveToInternalStorage(Bitmap bitmapImage, Context applicationContext) {
        String fileName = UUID.randomUUID() + ".png";

        ContextWrapper contextWrapper = new ContextWrapper(applicationContext);
        // path to /data/data/yourapp/app_data/imageDir
        File imageDirectory = contextWrapper.getDir("imageDir", Context.MODE_PRIVATE);
        // Create imageDir
        File imagePath = new File(imageDirectory, fileName);

        try (FileOutputStream fileOutputStream = new FileOutputStream(imagePath)) {
            // Use the compress method on the BitMap object to write image to the OutputStream
            bitmapImage.compress(Bitmap.CompressFormat.PNG, 100, fileOutputStream);
        } catch (Exception e) {
            AbxLog.e(e, true);
            return null;
        }

        return fileName;
    }

    public Bitmap getImageBitmapFromInternalStorage(Context applicationContext, String imagePathString) {
        ContextWrapper contextWrapper = new ContextWrapper(applicationContext);

        try {
            File imageDirectory = contextWrapper.getDir("imageDir", Context.MODE_PRIVATE);
            File imagePath = new File(imageDirectory, imagePathString);
            Bitmap imageBitmap = BitmapFactory.decodeStream(new FileInputStream(imagePath));
            return imageBitmap;
        } catch (Exception e) {
            AbxLog.e(e, true);
            return null;
        }
    }

    public boolean deleteImageFromInternalStorage(Context applicationContext, String fileName) {
        ContextWrapper contextWrapper = new ContextWrapper(applicationContext);

        try {
            File imageDirectory = contextWrapper.getDir("imageDir", Context.MODE_PRIVATE);
            File imageFile = new File(imageDirectory, fileName);
            return imageFile.delete();
        } catch (NullPointerException nullPointerException) {
            return false;
        }
    }

    public List<DfnIAMTrigger> getTriggers(String campaignId) {
        List<DfnIAMTrigger> triggers = new ArrayList<>();

        SQLiteDatabase db = dbHelper.getReadableDatabase();

        String getTriggersQuery = "SELECT "
                + InAppMessageSqliteStore.TriggersTable.TABLE_NAME + "." + InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_TYPE + ", "
                + InAppMessageSqliteStore.TriggersTable.TABLE_NAME + "." + InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_EVENT_NAME + ", "
                + InAppMessageSqliteStore.TriggersTable.TABLE_NAME + "." + InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_PRIORITY
                + " FROM " + InAppMessageSqliteStore.TriggersTable.TABLE_NAME
                + " WHERE " + InAppMessageSqliteStore.TriggersTable.TABLE_NAME + "." + InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_CAMPAIGN_ID + " = '" + campaignId + "'";

        AbxLog.d("getTriggersQuery query:" + getTriggersQuery, true);

        try (Cursor cursor = db.rawQuery(getTriggersQuery, null)) {
            while (cursor.moveToNext()) {
                triggers.add(
                        new DfnIAMTrigger(
                                cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_TYPE)),
                                CommonUtils.convertNullStringToNull(cursor.getString(cursor.getColumnIndex(InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_EVENT_NAME))),
                                cursor.getInt(cursor.getColumnIndex(InAppMessageSqliteStore.TriggersTable.COLUMN_NAME_PRIORITY))
                        )
                );
            }

            return triggers;
        } catch (Exception e) {
            return triggers;
        }
    }
}
