package io.adbrix.sdk.component;

import org.json.JSONArray;
import org.json.JSONException;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;

import io.adbrix.sdk.data.entity.DataRegistryKey;
import io.adbrix.sdk.data.entity.DataUnit;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.CoreConstants;
import io.adbrix.sdk.domain.function.Completion;
import io.adbrix.sdk.domain.model.DRState;
import io.adbrix.sdk.domain.model.Empty;
import io.adbrix.sdk.domain.model.Error;
import io.adbrix.sdk.domain.model.EventModel;
import io.adbrix.sdk.domain.model.EventPackage;
import io.adbrix.sdk.domain.model.Result;
import io.adbrix.sdk.domain.model.Success;
import io.adbrix.sdk.utils.CoreUtils;

public class EventPackageContainer {
    private ConcurrentLinkedQueue<EventPackage> packageQueue;
    private HashSet<String> successPackageIdTable;
    private IEventSender eventSender;
    private DataRegistry dataRegistry;

    public EventPackageContainer(IEventSender eventSender, DataRegistry dataRegistry) {
        this.dataRegistry = dataRegistry;
        this.eventSender = eventSender;
        packageQueue = getPreviousPackageQueue();
        AbxLog.d("getPreviousPackageQueue : " + packageQueue.size(), true);
        successPackageIdTable = new HashSet<>();
    }

    public void mergeV1EventsAndV2Events(EventPackage unsentV1EventPackage) {
        try {
            if (unsentV1EventPackage == null)
                return;

            Queue<EventPackage> mergedQueue = new ConcurrentLinkedQueue<>();
            Queue<EventPackage> v2Queue = new ConcurrentLinkedQueue<>(packageQueue);

            mergedQueue.offer(unsentV1EventPackage);

            while (!v2Queue.isEmpty()) {
                EventPackage eventPackage = v2Queue.poll();
                if(eventPackage != null){
                    mergedQueue.offer(eventPackage);
                }
            }

            packageQueue = formatEventPackageQueue(mergedQueue);

            AbxLog.d("Merging v1 events and v2 events is finished. EventQueue Size : " + packageQueue.size(), true);
        } catch (Exception e) {
            AbxLog.e(e, true);
        }
    }

    public void updateUnsentEventPackagesInDataRegistry() {

        JSONArray jsonArray = new JSONArray();

        for (EventPackage eventPackage : packageQueue) {
            try {
                jsonArray.put(eventPackage.toJson());
            } catch (JSONException e) {
                AbxLog.e(e, true);
            }
        }

        dataRegistry.putDataRegistry(new DataUnit(
                DataRegistryKey.STRING_UNSENT_EVENTPACKAGES,
                jsonArray.toString(),
                5,
                this.getClass().getName(),
                true
        ));
    }

    private ConcurrentLinkedQueue<EventPackage> getPreviousPackageQueue() {
        String unsentEventPackagesString = dataRegistry.safeGetString(DataRegistryKey.STRING_UNSENT_EVENTPACKAGES, null);

        if (unsentEventPackagesString == null) return new ConcurrentLinkedQueue<>();

        JSONArray jsonArray = null;
        try {
            jsonArray = new JSONArray(unsentEventPackagesString);
        } catch (JSONException e) {
            AbxLog.d(Arrays.toString(e.getStackTrace()), true);
        }

        if (jsonArray == null) return new ConcurrentLinkedQueue<>();

        Queue<EventPackage> tempQueue = new ConcurrentLinkedQueue<>();
        try {
            for (int i = 0; i < jsonArray.length(); i++) {
                EventPackage eventPackage = EventPackage.fromJson(jsonArray.getJSONObject(i));
                if (eventPackage == null) {
                    continue;
                }
                tempQueue.offer(eventPackage);
            }
        } catch (Exception e) {
            AbxLog.e(e, true);
        }

        return formatEventPackageQueue(tempQueue);
    }

    private ConcurrentLinkedQueue<EventPackage> formatEventPackageQueue(Queue<EventPackage> unformattedQueue) {
        Queue<EventModel> flattedEventModelQueue = new ConcurrentLinkedQueue<>();

        for (EventPackage eventPackage : unformattedQueue) {
            flattedEventModelQueue.addAll(eventPackage.eventModels);
        }

        while (flattedEventModelQueue.size() > CoreConstants.MAX_EVENT_COUNT)
            flattedEventModelQueue.poll();

        ConcurrentLinkedQueue<EventPackage> formattedEventPackageQueue = partition(flattedEventModelQueue);

        return formattedEventPackageQueue;
    }

    private ConcurrentLinkedQueue<EventPackage> partition(Queue<EventModel> eventModels) {
        ConcurrentLinkedQueue<EventPackage> eventPackages = new ConcurrentLinkedQueue<>();
        ConcurrentLinkedQueue<EventModel> chunked = new ConcurrentLinkedQueue<>();

        for (EventModel eventModel : eventModels) {
            chunked.add(eventModel);

            if (chunked.size() == CoreConstants.EVENT_BUFFER_COUNT) {
                eventPackages.add(new EventPackage(chunked));
                chunked = new ConcurrentLinkedQueue<>();
            }
        }

        if (!chunked.isEmpty()) {
            eventPackages.add(new EventPackage(chunked));
        }

        return eventPackages;
    }

    public void addEventPackage(EventPackage sendWaitEventPackage) {
        if(sendWaitEventPackage == null){
            AbxLog.e("sendWaitEventPackage is null", true);
            return;
        }
        if (!successPackageIdTable.contains(sendWaitEventPackage.packageId)) {
            this.packageQueue.offer(sendWaitEventPackage);
        }
    }

    public void addEventPackageToFront(EventPackage sendWaitEventPackage) {
        if(sendWaitEventPackage == null){
            AbxLog.e("sendWaitEventPackage is null", true);
            return;
        }
        if (successPackageIdTable.contains(sendWaitEventPackage.packageId)){
            return;
        }
        if(packageQueue.isEmpty()){
            packageQueue.offer(sendWaitEventPackage);
            return;
        }
        final ConcurrentLinkedQueue<EventPackage> packageQueueCopy = new ConcurrentLinkedQueue<>(this.packageQueue);
        final ConcurrentLinkedQueue<EventPackage> tempQueue = new ConcurrentLinkedQueue<>();
        tempQueue.offer(sendWaitEventPackage);
        while (!packageQueueCopy.isEmpty()){
            EventPackage eventPackage = packageQueueCopy.poll();
            if(eventPackage != null){
                tempQueue.offer(eventPackage);
            }
        }
        this.packageQueue = tempQueue;
    }

    public void flushAtIntervals() {

        Runnable eventSendingTask = new Runnable() {
            @Override
            public void run() {
                flushEventsNotSent(emptyResult -> {});
            }
        };

        EventUploadIntervalManager.getInstance().manageEventUploadInterval(eventSendingTask, packageQueue);
    }

    public void flushSingleEventPackageImmediately(EventPackage eventPackage, Completion<Result<Empty>> completion) {
        eventSender.sendEvent(eventPackage, new IEventSender.IEventSendActionListener() {
            @Override
            public void onSuccess(EventPackage successfulEventPackage) {
                if(completion==null){
                    AbxLog.e("completion is null", true);
                    return;
                }
                if(successPackageIdTable == null){
                    AbxLog.e("successPackageIdTable is null", true);
                    return;
                }
                successPackageIdTable.add(successfulEventPackage.packageId);
                completion.handle(Success.empty());
            }

            @Override
            public void onFail(EventPackage failedEventPackage) {
                if(completion==null){
                    AbxLog.e("completion is null", true);
                    return;
                }
                if(failedEventPackage == null){
                    completion.handle(Error.of("onFail() failedEventPackage is null"));
                    return;
                }
                EventPackageContainer.this.addEventPackageToFront(failedEventPackage);
                completion.handle(Error.of("network connection error"));
            }

            @Override
            public void onComplete() {
                AbxLog.d("flushSingleEventPackageImmediately.IEventSendActionListener onComplete()", true);
                completion.handle(Success.empty());
            }
        });
    }

    public void flushEventsNotSent(Completion<Result<Empty>> completion) {
        if (CoreUtils.getDRState(dataRegistry) == DRState.INIT_RESTART_NOT_SYNCED) {
            String message = "Initializing restart not synced! Cannot send events.";
            AbxLog.d(message, true);
            completion.handle(Error.of(message));
            return;
        }

        //fail된 eventPackage는 다음에 보내기 위한 장치
        if(this.packageQueue.isEmpty()){
            completion.handle(Success.empty());
            return;
        }
        // eventBuffer에만 존재하던 이벤트가 유실 될 수 있기 때문에 eventSender로 보내기 전에 Preference와 동기화
        updateUnsentEventPackagesInDataRegistry();
        String prevEventPackageId = "";
        int packageQueueCount = packageQueue.size();
        synchronized (packageQueue){
            while (packageQueueCount > 0){
                final EventPackage eventPackage = packageQueue.peek();
                if(eventPackage == null){
                    packageQueue.poll();
                    continue;
                }
                if(prevEventPackageId.equals(eventPackage.packageId)){
                    break;
                }
                prevEventPackageId = eventPackage.packageId;
                flushSingleEventPackageImmediately(eventPackage, completion);
                packageQueue.poll();
                packageQueueCount--;
            }
        }
        // eventSender를 통해 데이터를 보낸 후 바로 Preference와 동기화
        updateUnsentEventPackagesInDataRegistry();
    }

    public void clearQueue() {
        packageQueue.clear();
    }
}
