package io.adbrix.sdk.component;

import android.content.Context;

import org.json.JSONException;

import java.util.Arrays;

import io.adbrix.sdk.data.dataprovider.DeviceRealtimeDataProvider;
import io.adbrix.sdk.data.modelprovider.CommonModelProvider;
import io.adbrix.sdk.data.modelprovider.EventBundleModelProvider;
import io.adbrix.sdk.data.net.ApiConnection;
import io.adbrix.sdk.data.net.ApiConnectionManager;
import io.adbrix.sdk.data.net.IApiConnection;
import io.adbrix.sdk.data.repository.DataRegistry;
import io.adbrix.sdk.domain.model.CommonModel;
import io.adbrix.sdk.domain.model.Error;
import io.adbrix.sdk.domain.model.EventBundleModel;
import io.adbrix.sdk.domain.model.EventPackage;
import io.adbrix.sdk.utils.CoreUtils;

public class DefaultEventSender implements IEventSender {
    private final DataRegistry dataRegistry;
    private final Context androidContext;
    private final DeviceRealtimeDataProvider deviceRealtimeDataProvider;
    private final IABXComponentsFactory componentsFactory;

    public DefaultEventSender(DataRegistry dataRegistry, Context androidContext, DeviceRealtimeDataProvider deviceRealtimeDataProvider,IABXComponentsFactory componentsFactory) {
        this.dataRegistry = dataRegistry;
        this.androidContext = androidContext;
        this.deviceRealtimeDataProvider = deviceRealtimeDataProvider;
        this.componentsFactory = componentsFactory;
    }

    @Override
    public void sendEvent(final EventPackage eventPackage, IEventSendActionListener sendActionListener) {
        if (eventPackage == null || eventPackage.eventModels == null) {
            AbxLog.d("Invalid eventPackage!", true);
            return;
        }

        if (!CoreUtils.isOnline(androidContext)) {
            AbxLog.e("Fail to sending event. You are not connected to the network.", true);
            sendActionListener.onFail(eventPackage);
            return;
        }

        if (CoreUtils.isGdprForgetMe(dataRegistry, () -> { })) {
            AbxLog.d("GdprForgetMe is true", true);
            sendActionListener.onFail(eventPackage);
            return;
        }

        CommonModel commonModel = new CommonModelProvider(dataRegistry, androidContext, deviceRealtimeDataProvider).provide();
        EventBundleModel eventBundleModel = new EventBundleModelProvider(commonModel, eventPackage).provide();

        IApiConnection apiConnection;
        ApiConnectionManager<EventPackage> manager = new ApiConnectionManager<>(new ApiConnectionManager.Result<EventPackage>() {
            @Override
            public void connectSuccess(String responseString, int responseCode, EventPackage eventPackage) {
                if(sendActionListener == null){
                    AbxLog.d("sendActionListener is null", true);
                    return;
                }
                sendActionListener.onSuccess(eventPackage);
                AbxLog.d("Sending event is complete. responseCode : " + responseCode + ", responseString : " + responseString, true);
            }

            @Override
            public void connectFail(int responseCode, EventPackage eventPackage) {
                if(sendActionListener == null){
                    AbxLog.d("sendActionListener is null", true);
                    return;
                }
                if(eventPackage == null){
                    AbxLog.d("eventPackage is null", true);
                    sendActionListener.onComplete();
                    return;
                }
                sendActionListener.onFail(eventPackage);
                AbxLog.d("Sending event is failed. responseCode : " + responseCode, true);
            }
        }, eventPackage);

        try {
            AbxLog.d(eventBundleModel.getJson().toString(4), true);
            apiConnection = this.componentsFactory.createOrGetAPIConnection().post(eventBundleModel);
            manager.execute(apiConnection);

        } catch (JSONException e) {
            AbxLog.e("updateServer >> JSONException: ",e, true);
        }
    }
}
