/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.storage.jdbc.history;

import io.debezium.annotation.ThreadSafe;
import io.debezium.common.annotation.Incubating;
import io.debezium.config.Configuration;
import io.debezium.document.DocumentReader;
import io.debezium.document.DocumentWriter;
import io.debezium.relational.history.AbstractSchemaHistory;
import io.debezium.relational.history.HistoryRecord;
import io.debezium.relational.history.HistoryRecordComparator;
import io.debezium.relational.history.SchemaHistoryException;
import io.debezium.relational.history.SchemaHistoryListener;
import io.debezium.storage.jdbc.history.JdbcSchemaHistoryConfig;
import io.debezium.util.FunctionalReadWriteLock;
import java.io.IOException;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
@Incubating
public final class JdbcSchemaHistory
extends AbstractSchemaHistory {
    private static final Logger LOG = LoggerFactory.getLogger(JdbcSchemaHistory.class);
    private final FunctionalReadWriteLock lock = FunctionalReadWriteLock.reentrant();
    private final DocumentWriter writer = DocumentWriter.defaultWriter();
    private final DocumentReader reader = DocumentReader.defaultReader();
    private final AtomicBoolean running = new AtomicBoolean();
    private final AtomicInteger recordInsertSeq = new AtomicInteger(0);
    private Connection conn;
    private JdbcSchemaHistoryConfig config;

    public void configure(Configuration config, HistoryRecordComparator comparator, SchemaHistoryListener listener, boolean useCatalogBeforeSchema) {
        this.config = new JdbcSchemaHistoryConfig(config);
        if (this.running.get()) {
            throw new IllegalStateException("Database history already initialized db: " + this.config.getJdbcUrl());
        }
        super.configure(config, comparator, listener, useCatalogBeforeSchema);
        try {
            this.conn = DriverManager.getConnection(this.config.getJdbcUrl(), this.config.getUser(), this.config.getPassword());
            this.conn.setAutoCommit(false);
        }
        catch (SQLException e) {
            throw new IllegalStateException("Failed to connect " + this.config.getJdbcUrl(), e);
        }
    }

    public void start() {
        super.start();
        this.lock.write(() -> {
            if (this.running.compareAndSet(false, true)) {
                if (this.conn == null) {
                    throw new IllegalStateException("Database connection must be set before it is started");
                }
                try {
                    if (!this.storageExists()) {
                        this.initializeStorage();
                    }
                }
                catch (Exception e) {
                    throw new SchemaHistoryException("Unable to create history table " + this.config.getJdbcUrl() + ": " + e.getMessage(), (Throwable)e);
                }
            }
        });
    }

    protected void storeRecord(HistoryRecord record) throws SchemaHistoryException {
        if (record == null) {
            return;
        }
        this.lock.write(() -> {
            if (!this.running.get()) {
                throw new IllegalStateException("The history has been stopped and will not accept more records");
            }
            try {
                String line = this.writer.write(record.document());
                Timestamp currentTs = new Timestamp(System.currentTimeMillis());
                List<String> substrings = JdbcSchemaHistory.split(line, 65000);
                int partSeq = 0;
                for (String dataPart : substrings) {
                    PreparedStatement sql = this.conn.prepareStatement(this.config.getTableInsert());
                    sql.setString(1, UUID.randomUUID().toString());
                    sql.setString(2, dataPart);
                    sql.setInt(3, partSeq);
                    sql.setTimestamp(4, currentTs);
                    sql.setInt(5, this.recordInsertSeq.incrementAndGet());
                    sql.executeUpdate();
                    ++partSeq;
                }
                this.conn.commit();
            }
            catch (IOException | SQLException e) {
                try {
                    this.conn.rollback();
                }
                catch (SQLException sQLException) {
                    // empty catch block
                }
                throw new SchemaHistoryException("Failed to store record: " + record, (Throwable)e);
            }
        });
    }

    private static List<String> split(String s, int chunkSize) {
        ArrayList<String> chunks = new ArrayList<String>();
        for (int i = 0; i < s.length(); i += chunkSize) {
            chunks.add(s.substring(i, Math.min(s.length(), i + chunkSize)));
        }
        return chunks;
    }

    public void stop() {
        this.running.set(false);
        super.stop();
        try {
            this.conn.close();
        }
        catch (SQLException e) {
            LOG.error("Exception during stop", (Throwable)e);
        }
    }

    protected synchronized void recoverRecords(Consumer<HistoryRecord> records) {
        this.lock.write(() -> {
            try {
                if (this.exists()) {
                    Statement stmt = this.conn.createStatement();
                    ResultSet rs = stmt.executeQuery(this.config.getTableSelect());
                    while (rs.next()) {
                        String historyData = rs.getString("history_data");
                        if (historyData.isEmpty()) continue;
                        records.accept(new HistoryRecord(this.reader.read(historyData)));
                    }
                } else {
                    LOG.error("Storage does not exist when recovering records");
                }
            }
            catch (IOException | SQLException e) {
                throw new SchemaHistoryException("Failed to recover records", (Throwable)e);
            }
        });
    }

    public boolean storageExists() {
        boolean sExists = false;
        try {
            DatabaseMetaData dbMeta = this.conn.getMetaData();
            String databaseName = this.config.getDatabaseName();
            ResultSet tableExists = dbMeta.getTables(databaseName, null, this.config.getTableName(), null);
            if (tableExists.next()) {
                sExists = true;
            }
        }
        catch (SQLException e) {
            throw new SchemaHistoryException("Failed to check database history storage", (Throwable)e);
        }
        return sExists;
    }

    public boolean exists() {
        if (!this.storageExists()) {
            return false;
        }
        boolean isExists = false;
        try {
            Statement stmt = this.conn.createStatement();
            ResultSet rs = stmt.executeQuery(this.config.getTableDataExistsSelect());
            while (rs.next()) {
                isExists = true;
            }
        }
        catch (SQLException e) {
            throw new SchemaHistoryException("Failed to recover records", (Throwable)e);
        }
        return isExists;
    }

    JdbcSchemaHistoryConfig getConfig() {
        return this.config;
    }

    public String toString() {
        return "Jdbc database: " + (this.config != null ? this.config.getJdbcUrl() : "(unstarted)");
    }

    public void initializeStorage() {
        try {
            DatabaseMetaData dbMeta = this.conn.getMetaData();
            ResultSet tableExists = dbMeta.getTables(null, null, this.config.getTableName(), null);
            if (tableExists.next()) {
                return;
            }
            LOG.info("Creating table {} to store database history", (Object)this.config.getTableName());
            this.conn.prepareStatement(this.config.getTableCreate()).execute();
            LOG.info("Created table in given database...");
        }
        catch (SQLException e) {
            throw new SchemaHistoryException("Error initializing Database history storage", (Throwable)e);
        }
    }
}

