/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.jdbc;

import io.debezium.annotation.ThreadSafe;
import io.debezium.config.Configuration;
import io.debezium.config.Field;
import io.debezium.jdbc.JdbcConfiguration;
import io.debezium.relational.Column;
import io.debezium.relational.ColumnEditor;
import io.debezium.relational.TableEditor;
import io.debezium.relational.TableId;
import io.debezium.relational.Tables;
import io.debezium.util.Collect;
import io.debezium.util.Strings;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JdbcConnection
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(JdbcConnection.class);
    private final Configuration config;
    private final ConnectionFactory factory;
    private final Operations initialOps;
    private volatile Connection conn;

    protected static ConnectionFactory patternBasedFactory(String urlPattern, Field ... variables) {
        return config -> {
            LOGGER.trace("Config: {}", (Object)config.asProperties());
            Properties props = config.asProperties();
            Field[] varsWithDefaults = JdbcConnection.combineVariables(variables, JdbcConfiguration.HOSTNAME, JdbcConfiguration.PORT, JdbcConfiguration.USER, JdbcConfiguration.PASSWORD, JdbcConfiguration.DATABASE);
            String url = JdbcConnection.findAndReplace(urlPattern, props, varsWithDefaults);
            LOGGER.trace("Props: {}", (Object)props);
            LOGGER.trace("URL: {}", (Object)url);
            Connection conn = DriverManager.getConnection(url, props);
            LOGGER.debug("Connected to {} with {}", (Object)url, (Object)props);
            return conn;
        };
    }

    private static Field[] combineVariables(Field[] overriddenVariables, Field ... defaultVariables) {
        HashMap<String, Field> fields = new HashMap<String, Field>();
        if (defaultVariables != null) {
            for (Field variable : defaultVariables) {
                fields.put(variable.name(), variable);
            }
        }
        if (overriddenVariables != null) {
            for (Field variable : overriddenVariables) {
                fields.put(variable.name(), variable);
            }
        }
        return fields.values().toArray(new Field[fields.size()]);
    }

    private static String findAndReplace(String url, Properties props, Field ... variables) {
        for (Field field : variables) {
            String value;
            String variable = field.name();
            if (variable == null || !url.contains("${" + variable + "}") || (value = props.getProperty(variable)) == null) continue;
            props.remove(variable);
            url = url.replaceAll("\\$\\{" + variable + "\\}", value);
        }
        return url;
    }

    public JdbcConnection(Configuration config, ConnectionFactory connectionFactory) {
        this(config, connectionFactory, null);
    }

    public JdbcConnection(Configuration config, ConnectionFactory connectionFactory, Operations initialOperations) {
        this(config, connectionFactory, initialOperations, null);
    }

    protected JdbcConnection(Configuration config, ConnectionFactory connectionFactory, Operations initialOperations, Consumer<Configuration.Builder> adapter) {
        this.config = adapter == null ? config : ((Configuration.Builder)config.edit().apply((Consumer)adapter)).build();
        this.factory = connectionFactory;
        this.initialOps = initialOperations;
        this.conn = null;
    }

    public JdbcConfiguration config() {
        return JdbcConfiguration.adapt(this.config);
    }

    public JdbcConnection connect() throws SQLException {
        this.connection();
        return this;
    }

    public JdbcConnection execute(String ... sqlStatements) throws SQLException {
        return this.execute((Statement statement) -> {
            for (String sqlStatement : sqlStatements) {
                if (sqlStatement == null) continue;
                statement.execute(sqlStatement);
            }
        });
    }

    public JdbcConnection execute(Operations operations) throws SQLException {
        Connection conn = this.connection();
        conn.setAutoCommit(false);
        try (Statement statement = conn.createStatement();){
            operations.apply(statement);
            conn.commit();
        }
        return this;
    }

    public JdbcConnection query(String query, Consumer<ResultSet> resultConsumer) throws SQLException {
        Connection conn = this.connection();
        conn.setAutoCommit(false);
        try (Statement statement = conn.createStatement();){
            ResultSet resultSet = statement.executeQuery(query);
            if (resultConsumer != null) {
                resultConsumer.accept(resultSet);
            }
        }
        return this;
    }

    public void print(ResultSet resultSet) {
        this.print(resultSet, System.out::println);
    }

    public void print(ResultSet resultSet, Consumer<String> lines) {
        try {
            int i;
            ResultSetMetaData rsmd = resultSet.getMetaData();
            int columnCount = rsmd.getColumnCount();
            int[] columnSizes = this.findMaxLength(resultSet);
            lines.accept(this.delimiter(columnCount, columnSizes));
            StringBuilder sb = new StringBuilder();
            for (i = 1; i <= columnCount; ++i) {
                if (i > 1) {
                    sb.append(" | ");
                }
                sb.append(Strings.setLength(rsmd.getColumnLabel(i), columnSizes[i], ' '));
            }
            lines.accept(sb.toString());
            sb.setLength(0);
            lines.accept(this.delimiter(columnCount, columnSizes));
            while (resultSet.next()) {
                sb.setLength(0);
                for (i = 1; i <= columnCount; ++i) {
                    if (i > 1) {
                        sb.append(" | ");
                    }
                    sb.append(Strings.setLength(resultSet.getString(i), columnSizes[i], ' '));
                }
                lines.accept(sb.toString());
                sb.setLength(0);
            }
            lines.accept(this.delimiter(columnCount, columnSizes));
        }
        catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

    private String delimiter(int columnCount, int[] columnSizes) {
        StringBuilder sb = new StringBuilder();
        for (int i = 1; i <= columnCount; ++i) {
            if (i > 1) {
                sb.append("---");
            }
            sb.append(Strings.createString('-', columnSizes[i]));
        }
        return sb.toString();
    }

    private int[] findMaxLength(ResultSet resultSet) throws SQLException {
        int i;
        ResultSetMetaData rsmd = resultSet.getMetaData();
        int columnCount = rsmd.getColumnCount();
        int[] columnSizes = new int[columnCount + 1];
        for (i = 1; i <= columnCount; ++i) {
            columnSizes[i] = Math.max(columnSizes[i], rsmd.getColumnLabel(i).length());
        }
        while (resultSet.next()) {
            for (i = 1; i <= columnCount; ++i) {
                String value = resultSet.getString(i);
                if (value == null) continue;
                columnSizes[i] = Math.max(columnSizes[i], value.length());
            }
        }
        resultSet.beforeFirst();
        return columnSizes;
    }

    public synchronized Connection connection() throws SQLException {
        if (this.conn == null) {
            this.conn = this.factory.connect(JdbcConfiguration.adapt(this.config));
            if (this.conn == null) {
                throw new SQLException("Unable to obtain a JDBC connection");
            }
            if (this.initialOps != null) {
                this.execute(this.initialOps);
            }
        }
        return this.conn;
    }

    @Override
    public synchronized void close() throws SQLException {
        if (this.conn != null) {
            try {
                this.conn.close();
            }
            finally {
                this.conn = null;
            }
        }
    }

    public void readSchema(Tables tables, String databaseCatalog, String schemaNamePattern, Tables.TableFilter tableFilter, Tables.ColumnFilter columnFilter) throws SQLException {
        DatabaseMetaData metadata = this.conn.getMetaData();
        ConcurrentHashMap<TableId, List> columnsByTable = new ConcurrentHashMap<TableId, List>();
        try (ResultSet rs = metadata.getColumns(databaseCatalog, schemaNamePattern, null, null);){
            while (rs.next()) {
                String catalogName = rs.getString(1);
                String schemaName = rs.getString(2);
                String tableName = rs.getString(3);
                if (tableFilter != null && !tableFilter.test(catalogName, schemaName, tableName)) continue;
                TableId tableId = new TableId(catalogName, schemaName, tableName);
                List cols = columnsByTable.computeIfAbsent(tableId, name -> new ArrayList());
                String columnName = rs.getString(4);
                if (columnFilter != null && !columnFilter.test(catalogName, schemaName, tableName, columnName)) continue;
                ColumnEditor column = Column.editor().name(columnName);
                column.jdbcType(rs.getInt(5));
                column.typeName(rs.getString(6));
                column.length(rs.getInt(7));
                column.scale(rs.getInt(9));
                column.optional(JdbcConnection.isNullable(rs.getInt(11)));
                column.position(rs.getInt(17));
                column.autoIncremented("YES".equalsIgnoreCase(rs.getString(23)));
                column.generated("YES".equalsIgnoreCase(rs.getString(24)));
                cols.add(column.create());
            }
        }
        for (TableId id : columnsByTable.keySet()) {
            ArrayList<String> pkColumnNames = null;
            try (ResultSet rs = metadata.getPrimaryKeys(id.catalog(), id.schema(), id.table());){
                while (rs.next()) {
                    if (pkColumnNames == null) {
                        pkColumnNames = new ArrayList<String>();
                    }
                    String columnName = rs.getString(4);
                    int columnIndex = rs.getInt(5);
                    Collect.set(pkColumnNames, columnIndex - 1, columnName, null);
                }
            }
            List columns = (List)columnsByTable.get(id);
            Collections.sort(columns);
            tables.overwriteTable(id, columns, pkColumnNames);
        }
    }

    public static void columnsFor(ResultSet resultSet, TableEditor editor) throws SQLException {
        ArrayList<Column> columns = new ArrayList<Column>();
        JdbcConnection.columnsFor(resultSet, columns::add);
        editor.setColumns(columns);
    }

    public static void columnsFor(ResultSet resultSet, Consumer<Column> consumer) throws SQLException {
        ResultSetMetaData metadata = resultSet.getMetaData();
        ColumnEditor column = Column.editor();
        for (int position = 1; position <= metadata.getColumnCount(); ++position) {
            String columnLabel = metadata.getColumnLabel(position);
            column.name(columnLabel != null ? columnLabel : metadata.getColumnName(position));
            column.typeName(metadata.getColumnTypeName(position));
            column.jdbcType(metadata.getColumnType(position));
            column.length(metadata.getPrecision(position));
            column.scale(metadata.getScale(position));
            column.optional(JdbcConnection.isNullable(metadata.isNullable(position)));
            column.autoIncremented(metadata.isAutoIncrement(position));
            column.generated(false);
            consumer.accept(column.create());
        }
    }

    private static boolean isNullable(int jdbcNullable) {
        return jdbcNullable == 1 || jdbcNullable == 2;
    }

    @FunctionalInterface
    public static interface Operations {
        public void apply(Statement var1) throws SQLException;
    }

    @FunctionalInterface
    @ThreadSafe
    public static interface ConnectionFactory {
        public Connection connect(JdbcConfiguration var1) throws SQLException;
    }
}

