/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.transforms;

import io.debezium.DebeziumException;
import io.debezium.Module;
import io.debezium.config.Configuration;
import io.debezium.config.Field;
import io.debezium.time.ZonedTime;
import io.debezium.time.ZonedTimestamp;
import io.debezium.transforms.SmtManager;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.OffsetDateTime;
import java.time.OffsetTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.TimeZone;
import java.util.regex.Pattern;
import org.apache.kafka.common.config.ConfigDef;
import org.apache.kafka.connect.components.Versioned;
import org.apache.kafka.connect.connector.ConnectRecord;
import org.apache.kafka.connect.data.Schema;
import org.apache.kafka.connect.data.Struct;
import org.apache.kafka.connect.errors.DataException;
import org.apache.kafka.connect.transforms.Transformation;
import org.apache.kafka.connect.transforms.util.Requirements;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TimezoneConverter<R extends ConnectRecord<R>>
implements Transformation<R>,
Versioned {
    private static final Logger LOGGER = LoggerFactory.getLogger(TimezoneConverter.class);
    private static final Field CONVERTED_TIMEZONE = Field.create("converted.timezone").withDisplayName("Converted Timezone").withType(ConfigDef.Type.STRING).withWidth(ConfigDef.Width.MEDIUM).withImportance(ConfigDef.Importance.HIGH).withValidation(Field::isRequired).withDescription("A string that represents the timezone to which the time-based fields should be converted.The format can be geographic (e.g. America/Los_Angeles), or it can be a UTC offset in the format of +/-hh:mm, (e.g. +08:00).");
    private static final Field INCLUDE_LIST = Field.create("include.list").withDisplayName("Include List").withType(ConfigDef.Type.LIST).withWidth(ConfigDef.Width.MEDIUM).withImportance(ConfigDef.Importance.HIGH).withValidation(Field::isListOfRegex).withDescription("A comma-separated list of rules that specify what events should be evaluated for timezone conversion, using one of the following formats: `source:<tablename>`:: Matches only Debezium change events with a source information block with the specified table name. All time-based fields will be converted. `source:<tablename>:<fieldname>`:: Matches only Debezium change events with a source information with the specified table name. Only the specified field name will be converted. `topic:<topicname>`:: Matches the specified topic name, converting all time-based fields. `topic:<topicname>:<fieldname>`:: Matches the specified topic name, converting only the specified field name. `<matchname>:<fieldname>`:: Uses a heuristic matching algorithm to matches the source information block table name if the source information block exists, otherwise matches against the topic name. The conversion is applied only to to the specified field name. ");
    private static final Field EXCLUDE_LIST = Field.create("exclude.list").withDisplayName("Exclude List").withType(ConfigDef.Type.LIST).withWidth(ConfigDef.Width.MEDIUM).withImportance(ConfigDef.Importance.HIGH).withValidation(Field::isListOfRegex).withDescription("A comma-separated list of rules that specify what events should be excluded from timezone conversion, using one of the following formats: `source:<tablename>`:: Matches only Debezium change events with a source information block with the specified table name. All time-based fields will be excluded. `source:<tablename>:<fieldnames>`:: Matches only Debezium change events with a source information with the specified table name. Only the specified field name will be excluded. `topic:<topicname>`:: Matches the specified topic name, excluding all time-based fields. `topic:<topicname>:<fieldnames>`:: Matches the specified topic name, excluding only the specified field name. `<matchname>:<fieldnames>`:: Uses a heuristic matching algorithm to matches the source information block table name if the source information block exists, otherwise matches against the topic name. The conversion is applied only to to the specified field name. ");
    private SmtManager<R> smtManager;
    private String convertedTimezone;
    private List<String> includeList;
    private List<String> excludeList;
    private static final String SOURCE = "source";
    private static final String TOPIC = "topic";
    private static final String FIELD_SOURCE_PREFIX = "source.";
    private static final String FIELD_BEFORE_PREFIX = "before.";
    private static final String FIELD_AFTER_PREFIX = "after.";
    private static final Pattern TIMEZONE_OFFSET_PATTERN = Pattern.compile("^[+-]\\d{2}:\\d{2}(:\\d{2})?$");
    private static final Pattern LIST_PATTERN = Pattern.compile("^\\[(source|topic|[\".\\w\\s_]+):([\".\\w\\s_]+(?::[\".\\w\\s_]+)?(?:,|]$))+$");
    private final Map<String, Set<String>> topicFieldsMap = new HashMap<String, Set<String>>();
    private final Map<String, Set<String>> tableFieldsMap = new HashMap<String, Set<String>>();
    private final Map<String, Set<String>> noPrefixFieldsMap = new HashMap<String, Set<String>>();
    private static final List<String> SUPPORTED_TIMESTAMP_LOGICAL_NAMES = List.of("io.debezium.time.MicroTimestamp", "io.debezium.time.NanoTimestamp", "io.debezium.time.Timestamp", "io.debezium.time.ZonedTimestamp", "io.debezium.time.ZonedTime", "org.apache.kafka.connect.data.Timestamp");
    private static final List<String> UNSUPPORTED_LOGICAL_NAMES = List.of("io.debezium.time.Date", "io.debezium.time.MicroTime", "io.debezium.time.NanoTime", "io.debezium.time.Time", "org.apache.kafka.connect.data.Date", "org.apache.kafka.connect.data.Time");

    public ConfigDef config() {
        ConfigDef config = new ConfigDef();
        Field.group(config, null, CONVERTED_TIMEZONE, INCLUDE_LIST, EXCLUDE_LIST);
        return config;
    }

    public R apply(R record) {
        if (record.value() == null || !this.smtManager.isValidEnvelope(record)) {
            return record;
        }
        Struct value = (Struct)record.value();
        String table = this.getTableFromSource(value);
        String topic = record.topic();
        if (this.includeList.isEmpty() && this.excludeList.isEmpty()) {
            this.handleAllRecords(value, table, topic);
        } else if (!this.includeList.isEmpty()) {
            this.handleInclude(value, table, topic);
        } else {
            this.handleExclude(value, table, topic);
        }
        return (R)record.newRecord(record.topic(), record.kafkaPartition(), record.keySchema(), record.key(), record.valueSchema(), record.value(), record.timestamp(), (Iterable)record.headers());
    }

    public void configure(Map<String, ?> configs) {
        Configuration config = Configuration.from(configs);
        this.smtManager = new SmtManager(config);
        this.smtManager.validate(config, Field.setOf(CONVERTED_TIMEZONE, INCLUDE_LIST, EXCLUDE_LIST));
        this.convertedTimezone = config.getString(CONVERTED_TIMEZONE);
        this.includeList = config.getList(INCLUDE_LIST);
        this.excludeList = config.getList(EXCLUDE_LIST);
        this.validateConfiguration();
        if (!this.excludeList.isEmpty()) {
            this.collectTablesAndTopics(this.excludeList);
        } else if (!this.includeList.isEmpty()) {
            this.collectTablesAndTopics(this.includeList);
        }
    }

    private void collectTablesAndTopics(List<String> list) {
        String commonPrefix = null;
        for (String item : list) {
            FieldItem parseItem = this.parseItem(item);
            String prefix = parseItem.prefix;
            String matchName = parseItem.getMatchName();
            String field = parseItem.getFieldName();
            if (prefix != null) {
                commonPrefix = prefix;
            }
            if (Objects.equals(commonPrefix, TOPIC)) {
                if (!this.topicFieldsMap.containsKey(matchName)) {
                    this.topicFieldsMap.put(matchName, new HashSet());
                }
                if (field == null) continue;
                this.topicFieldsMap.get(matchName).add(field);
                continue;
            }
            if (Objects.equals(commonPrefix, SOURCE)) {
                if (!this.tableFieldsMap.containsKey(matchName)) {
                    this.tableFieldsMap.put(matchName, new HashSet());
                }
                if (field == null) continue;
                this.tableFieldsMap.get(matchName).add(field);
                continue;
            }
            if (!this.noPrefixFieldsMap.containsKey(matchName)) {
                this.noPrefixFieldsMap.put(matchName, new HashSet());
            }
            if (field == null) continue;
            this.noPrefixFieldsMap.get(matchName).add(field);
        }
    }

    private void validateConfiguration() {
        if (!this.includeList.isEmpty()) {
            if (!LIST_PATTERN.matcher(this.includeList.toString()).matches()) {
                throw new DebeziumException("Invalid include list format. Please specify a list of rules in the format of \"source:<tablename>:<fieldnames>\", \"topic:<topicname>:<fieldnames>\", \"<matchname>:<fieldnames>\"");
            }
        } else if (!this.excludeList.isEmpty() && !LIST_PATTERN.matcher(this.excludeList.toString()).matches()) {
            throw new DebeziumException("Invalid exclude list format. Please specify a list of rules in the format of \"source:<tablename>:<fieldnames>\", \"topic:<topicname>:<fieldnames>\", \"<matchname>:<fieldnames>\"");
        }
        if (!this.validateTimezoneString()) {
            throw new DebeziumException("Invalid timezone format. Please specify either a geographic timezone (e.g. America/Los_Angeles) or a UTC offset in the format of +/-hh:mm, (e.g. +08:00).");
        }
        if (!this.includeList.isEmpty() && !this.excludeList.isEmpty()) {
            throw new DebeziumException("Both include and exclude lists are specified. Please specify only one.");
        }
    }

    private boolean validateTimezoneString() {
        if (TIMEZONE_OFFSET_PATTERN.matcher(this.convertedTimezone).matches()) {
            return true;
        }
        if (ZoneId.getAvailableZoneIds().contains(this.convertedTimezone)) {
            return true;
        }
        return Arrays.asList(TimeZone.getAvailableIDs()).contains(this.convertedTimezone);
    }

    public void close() {
    }

    public String version() {
        return Module.version();
    }

    private Object getTimestampWithTimezone(String schemaName, Object fieldValue) {
        Object updatedFieldValue = fieldValue;
        ZoneId zoneId = ZoneId.of(this.convertedTimezone);
        ZoneOffset zoneOffset = zoneId.getRules().getOffset(Instant.now());
        switch (schemaName) {
            case "io.debezium.time.ZonedTimestamp": {
                OffsetDateTime offsetDateTime = OffsetDateTime.parse((String)fieldValue, DateTimeFormatter.ISO_OFFSET_DATE_TIME);
                zoneOffset = zoneId.getRules().getOffset(offsetDateTime.toLocalDateTime());
                OffsetDateTime offsetDateTimeWithZone = offsetDateTime.withOffsetSameInstant(zoneOffset);
                updatedFieldValue = ZonedTimestamp.toIsoString(offsetDateTimeWithZone, null);
                break;
            }
            case "io.debezium.time.ZonedTime": {
                OffsetTime offsetTime = OffsetTime.parse((String)fieldValue, DateTimeFormatter.ISO_OFFSET_TIME);
                OffsetTime offsetTimeWithZone = offsetTime.withOffsetSameInstant(zoneOffset);
                updatedFieldValue = ZonedTime.toIsoString(offsetTimeWithZone, null);
                break;
            }
            case "io.debezium.time.MicroTimestamp": {
                long microTimestamp = (Long)fieldValue;
                Instant microInstant = Instant.ofEpochSecond(microTimestamp / 1000000L, microTimestamp % 1000000L * 1000L);
                LocalDateTime microLocalDateTime = microInstant.atOffset(ZoneOffset.UTC).toLocalDateTime();
                zoneOffset = zoneId.getRules().getOffset(microLocalDateTime);
                updatedFieldValue = microLocalDateTime.toEpochSecond(zoneOffset) * 1000000L + (long)(microLocalDateTime.getNano() / 1000);
                break;
            }
            case "io.debezium.time.NanoTimestamp": {
                long nanoTimestamp = (Long)fieldValue;
                Instant nanoInstant = Instant.ofEpochSecond(nanoTimestamp / 1000000000L, nanoTimestamp % 1000000000L);
                LocalDateTime nanoLocalDateTime = nanoInstant.atOffset(ZoneOffset.UTC).toLocalDateTime();
                zoneOffset = zoneId.getRules().getOffset(nanoLocalDateTime);
                updatedFieldValue = nanoLocalDateTime.toEpochSecond(zoneOffset) * 1000000000L + (long)nanoLocalDateTime.getNano();
                break;
            }
            case "io.debezium.time.Timestamp": {
                Instant instant = Instant.ofEpochMilli((Long)fieldValue);
                LocalDateTime localDateTime = instant.atOffset(ZoneOffset.UTC).toLocalDateTime();
                zoneOffset = zoneId.getRules().getOffset(localDateTime);
                updatedFieldValue = localDateTime.atOffset(zoneOffset).toInstant().toEpochMilli();
                break;
            }
            case "org.apache.kafka.connect.data.Timestamp": {
                Date date = (Date)fieldValue;
                Instant timestampInstant = date.toInstant();
                LocalDateTime timestampLocalDateTime = timestampInstant.atOffset(ZoneOffset.UTC).toLocalDateTime();
                zoneOffset = zoneId.getRules().getOffset(timestampLocalDateTime);
                updatedFieldValue = Date.from(timestampLocalDateTime.atOffset(zoneOffset).toInstant());
            }
        }
        return updatedFieldValue;
    }

    private void handleStructs(Struct value, Type type, String matchName, Set<String> fields) {
        if (type == null || matchName == null) {
            return;
        }
        Struct before = this.getStruct(value, "before");
        Struct after = this.getStruct(value, "after");
        Struct source = this.getStruct(value, SOURCE);
        HashSet<String> beforeFields = new HashSet<String>();
        HashSet<String> afterFields = new HashSet<String>();
        HashSet<String> sourceFields = new HashSet<String>();
        if (!fields.isEmpty()) {
            for (String field : fields) {
                if (field.startsWith(FIELD_SOURCE_PREFIX)) {
                    sourceFields.add(field.substring(FIELD_SOURCE_PREFIX.length()));
                    continue;
                }
                if (field.startsWith(FIELD_BEFORE_PREFIX)) {
                    beforeFields.add(field.substring(FIELD_BEFORE_PREFIX.length()));
                    continue;
                }
                if (field.startsWith(FIELD_AFTER_PREFIX)) {
                    afterFields.add(field.substring(FIELD_AFTER_PREFIX.length()));
                    continue;
                }
                beforeFields.add(field);
                afterFields.add(field);
            }
        }
        if (before != null) {
            this.handleValueForFields(before, type, beforeFields);
        }
        if (after != null) {
            this.handleValueForFields(after, type, afterFields);
        }
        if (source != null && !sourceFields.isEmpty()) {
            this.handleValueForFields(source, type, sourceFields);
        }
    }

    private void handleValueForFields(Struct value, Type type, Set<String> fields) {
        Schema schema = value.schema();
        for (org.apache.kafka.connect.data.Field field : schema.fields()) {
            boolean shouldIncludeField;
            String schemaName = field.schema().name();
            if (schemaName == null) continue;
            boolean isUnsupportedLogicalType = UNSUPPORTED_LOGICAL_NAMES.contains(schemaName);
            boolean supportedLogicalType = SUPPORTED_TIMESTAMP_LOGICAL_NAMES.contains(schemaName);
            boolean bl = shouldIncludeField = type == Type.ALL || type == Type.INCLUDE && fields.contains(field.name()) || type == Type.EXCLUDE && !fields.contains(field.name());
            if (isUnsupportedLogicalType && shouldIncludeField) {
                LOGGER.warn("Skipping conversion for unsupported logical type: " + schemaName + " for field: " + field.name());
                continue;
            }
            if (!shouldIncludeField || !supportedLogicalType || value.get(field) == null) continue;
            this.handleValueForField(value, field);
        }
    }

    private void handleValueForField(Struct struct, org.apache.kafka.connect.data.Field field) {
        String fieldName = field.name();
        Schema schema = field.schema();
        Object newValue = this.getTimestampWithTimezone(schema.name(), struct.get(fieldName));
        struct.put(fieldName, newValue);
    }

    private Struct getStruct(Struct struct, String structName) {
        try {
            return Requirements.requireStructOrNull((Object)struct.getStruct(structName), (String)"");
        }
        catch (DataException dataException) {
            return null;
        }
    }

    private String getTableFromSource(Struct value) {
        try {
            Struct source = value.getStruct(SOURCE);
            return source.getString("table");
        }
        catch (DataException dataException) {
            return null;
        }
    }

    private FieldItem parseItem(String item) {
        String prefix = null;
        String matchName = null;
        String fieldName = null;
        String[] parts = item.split(":");
        if (parts.length == 1) {
            matchName = parts[0];
        } else if (parts.length >= 2 && parts.length <= 3) {
            if (parts[0].equalsIgnoreCase(SOURCE) || parts[0].equalsIgnoreCase(TOPIC)) {
                prefix = parts[0];
                matchName = parts[1];
                if (parts.length == 3) {
                    fieldName = parts[2];
                }
            } else {
                matchName = parts[0];
                fieldName = parts[1];
            }
        }
        return new FieldItem(prefix, matchName, fieldName);
    }

    private MatchFieldsResult handleMatchNameAndFields(String table, String topic) {
        String matchName = null;
        Set<Object> fields = Collections.emptySet();
        if (this.topicFieldsMap.containsKey(topic)) {
            matchName = topic;
            fields = this.topicFieldsMap.get(topic);
        } else if (this.tableFieldsMap.containsKey(table)) {
            matchName = table;
            fields = this.tableFieldsMap.get(table);
        } else if (this.noPrefixFieldsMap.containsKey(topic)) {
            matchName = topic;
            fields = this.noPrefixFieldsMap.get(topic);
        } else if (this.noPrefixFieldsMap.containsKey(table)) {
            matchName = table;
            fields = this.noPrefixFieldsMap.get(table);
        }
        return new MatchFieldsResult(matchName, fields);
    }

    private void handleInclude(Struct value, String table, String topic) {
        MatchFieldsResult matchFieldsResult = this.handleMatchNameAndFields(table, topic);
        String matchName = matchFieldsResult.getMatchName();
        Set<String> fields = matchFieldsResult.getFields();
        if (matchName != null) {
            if (!fields.isEmpty()) {
                this.handleStructs(value, Type.INCLUDE, matchName, fields);
            } else {
                this.handleStructs(value, Type.ALL, matchName, fields);
            }
        } else {
            this.handleStructs(value, Type.ALL, table, Collections.emptySet());
        }
    }

    private void handleExclude(Struct value, String table, String topic) {
        MatchFieldsResult matchFieldsResult = this.handleMatchNameAndFields(table, topic);
        String matchName = matchFieldsResult.getMatchName();
        Set<String> fields = matchFieldsResult.getFields();
        if (matchName == null) {
            this.handleStructs(value, Type.ALL, table != null ? table : topic, Collections.emptySet());
        } else if (!fields.isEmpty()) {
            this.handleStructs(value, Type.EXCLUDE, matchName, fields);
        }
    }

    private void handleAllRecords(Struct value, String table, String topic) {
        if (!(this.topicFieldsMap.containsKey(topic) || this.tableFieldsMap.containsKey(table) || this.noPrefixFieldsMap.containsKey(table))) {
            this.handleStructs(value, Type.ALL, table != null ? table : topic, Collections.emptySet());
        }
    }

    private static class FieldItem {
        private final String prefix;
        private final String matchName;
        private final String fieldName;

        FieldItem(String prefix, String matchName, String fieldName) {
            this.prefix = prefix;
            this.matchName = matchName;
            this.fieldName = fieldName;
        }

        public String getPrefix() {
            return this.prefix;
        }

        public String getMatchName() {
            return this.matchName;
        }

        public String getFieldName() {
            return this.fieldName;
        }
    }

    private static enum Type {
        ALL,
        INCLUDE,
        EXCLUDE;

    }

    private static class MatchFieldsResult {
        private final String matchName;
        private final Set<String> fields;

        MatchFieldsResult(String matchName, Set<String> fields) {
            this.matchName = matchName;
            this.fields = fields;
        }

        public String getMatchName() {
            return this.matchName;
        }

        public Set<String> getFields() {
            return this.fields;
        }
    }
}

