/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.transforms;

import io.debezium.config.Configuration;
import io.debezium.config.Field;
import io.debezium.data.Envelope;
import io.debezium.util.SchemaNameAdjuster;
import io.debezium.util.Strings;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.kafka.common.cache.Cache;
import org.apache.kafka.common.cache.LRUCache;
import org.apache.kafka.common.cache.SynchronizedCache;
import org.apache.kafka.common.config.ConfigDef;
import org.apache.kafka.connect.connector.ConnectRecord;
import org.apache.kafka.connect.data.Schema;
import org.apache.kafka.connect.data.SchemaBuilder;
import org.apache.kafka.connect.data.Struct;
import org.apache.kafka.connect.errors.ConnectException;
import org.apache.kafka.connect.transforms.Transformation;
import org.apache.kafka.connect.transforms.util.Requirements;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ByLogicalTableRouter<R extends ConnectRecord<R>>
implements Transformation<R> {
    private static final Field TOPIC_REGEX = Field.create("topic.regex").withDisplayName("Topic regex").withType(ConfigDef.Type.STRING).withWidth(ConfigDef.Width.LONG).withImportance(ConfigDef.Importance.LOW).withValidation(Field::isRequired, Field::isRegex).withDescription("The regex used for extracting the name of the logical table from the original topic name.");
    private static final Field TOPIC_REPLACEMENT = Field.create("topic.replacement").withDisplayName("Topic replacement").withType(ConfigDef.Type.STRING).withWidth(ConfigDef.Width.LONG).withImportance(ConfigDef.Importance.LOW).withValidation(Field::isRequired).withDescription("The replacement string used in conjunction with " + TOPIC_REGEX.name() + ". This will be used to create the new topic name.");
    private static final Field KEY_FIELD_REGEX = Field.create("key.field.regex").withDisplayName("Key field regex").withType(ConfigDef.Type.STRING).withWidth(ConfigDef.Width.LONG).withImportance(ConfigDef.Importance.LOW).withValidation(Field::isRegex).withDescription("The regex used for extracting the physical table identifier from the original topic name. Now that multiple physical tables can share a topic, the event's key may need to be augmented to include fields other than just those for the record's primary/unique key, since these are not guaranteed to be unique across tables. We need some identifier added to the key that distinguishes the different physical tables.");
    private static final Field KEY_FIELD_NAME = Field.create("key.field.name").withDisplayName("Key field name").withType(ConfigDef.Type.STRING).withWidth(ConfigDef.Width.LONG).withImportance(ConfigDef.Importance.LOW).withDefault("__dbz__physicalTableIdentifier").withDescription("Each record's key schema will be augmented with this field name. The purpose of this field is to distinguish the different physical tables that can now share a single topic. Make sure not to configure a field name that is at risk of conflict with existing key schema field names.");
    private static final Field KEY_FIELD_REPLACEMENT = Field.create("key.field.replacement").withDisplayName("Key field replacement").withType(ConfigDef.Type.STRING).withWidth(ConfigDef.Width.LONG).withImportance(ConfigDef.Importance.LOW).withValidation(ByLogicalTableRouter::validateKeyFieldReplacement).withDescription("The replacement string used in conjunction with " + KEY_FIELD_REGEX.name() + ". This will be used to create the physical table identifier in the record's key.");
    private static final Logger logger = LoggerFactory.getLogger(ByLogicalTableRouter.class);
    private final SchemaNameAdjuster schemaNameAdjuster = SchemaNameAdjuster.create(logger);
    private Pattern topicRegex;
    private String topicReplacement;
    private Pattern keyFieldRegex;
    private String keyFieldReplacement;
    private String keyFieldName;
    private final Cache<Schema, Schema> keySchemaUpdateCache = new SynchronizedCache((Cache)new LRUCache(16));
    private final Cache<Schema, Schema> envelopeSchemaUpdateCache = new SynchronizedCache((Cache)new LRUCache(16));
    private final Cache<String, String> keyRegexReplaceCache = new SynchronizedCache((Cache)new LRUCache(16));
    private final Cache<String, String> topicRegexReplaceCache = new SynchronizedCache((Cache)new LRUCache(16));

    private static int validateKeyFieldReplacement(Configuration config, Field field, Field.ValidationOutput problems) {
        String keyFieldReplacement;
        String keyFieldRegex = config.getString(KEY_FIELD_REGEX);
        if (keyFieldRegex != null) {
            keyFieldRegex = keyFieldRegex.trim();
        }
        if ((keyFieldReplacement = config.getString(KEY_FIELD_REPLACEMENT)) != null) {
            keyFieldReplacement = keyFieldReplacement.trim();
        }
        if (!Strings.isNullOrEmpty(keyFieldRegex) && Strings.isNullOrEmpty(keyFieldReplacement)) {
            problems.accept(KEY_FIELD_REPLACEMENT, null, String.format("%s must be non-empty if %s is set.", KEY_FIELD_REPLACEMENT.name(), KEY_FIELD_REGEX.name()));
            return 1;
        }
        return 0;
    }

    public void configure(Map<String, ?> props) {
        Configuration config = Configuration.from(props);
        Field.Set configFields = Field.setOf(TOPIC_REGEX, TOPIC_REPLACEMENT, KEY_FIELD_REGEX, KEY_FIELD_REPLACEMENT);
        if (!config.validateAndRecord(configFields, arg_0 -> ((Logger)logger).error(arg_0))) {
            throw new ConnectException("Unable to validate config.");
        }
        this.topicRegex = Pattern.compile(config.getString(TOPIC_REGEX));
        this.topicReplacement = config.getString(TOPIC_REPLACEMENT);
        String keyFieldRegexString = config.getString(KEY_FIELD_REGEX);
        if (keyFieldRegexString != null) {
            keyFieldRegexString = keyFieldRegexString.trim();
        }
        if (keyFieldRegexString != null && !keyFieldRegexString.isEmpty()) {
            this.keyFieldRegex = Pattern.compile(config.getString(KEY_FIELD_REGEX));
            this.keyFieldReplacement = config.getString(KEY_FIELD_REPLACEMENT);
        }
        this.keyFieldName = config.getString(KEY_FIELD_NAME);
    }

    public R apply(R record) {
        String oldTopic = record.topic();
        String newTopic = this.determineNewTopic(oldTopic);
        if (newTopic == null) {
            return record;
        }
        logger.debug("Applying topic name transformation from {} to {}", (Object)oldTopic, (Object)newTopic);
        Schema newKeySchema = null;
        Struct newKey = null;
        if (record.key() != null) {
            Struct oldKey = Requirements.requireStruct((Object)record.key(), (String)"Updating schema");
            newKeySchema = this.updateKeySchema(oldKey.schema(), newTopic);
            newKey = this.updateKey(newKeySchema, oldKey, oldTopic);
        }
        if (record.value() == null) {
            return (R)record.newRecord(newTopic, record.kafkaPartition(), newKeySchema, newKey, record.valueSchema(), record.value(), record.timestamp());
        }
        Struct oldEnvelope = Requirements.requireStruct((Object)record.value(), (String)"Updating schema");
        Schema newEnvelopeSchema = this.updateEnvelopeSchema(oldEnvelope.schema(), newTopic);
        Struct newEnvelope = this.updateEnvelope(newEnvelopeSchema, oldEnvelope);
        return (R)record.newRecord(newTopic, record.kafkaPartition(), newKeySchema, (Object)newKey, newEnvelopeSchema, (Object)newEnvelope, record.timestamp());
    }

    public void close() {
    }

    public ConfigDef config() {
        ConfigDef config = new ConfigDef();
        Field.group(config, null, TOPIC_REGEX, TOPIC_REPLACEMENT, KEY_FIELD_REGEX, KEY_FIELD_REPLACEMENT);
        return config;
    }

    private String determineNewTopic(String oldTopic) {
        String newTopic = (String)this.topicRegexReplaceCache.get((Object)oldTopic);
        if (newTopic != null) {
            return newTopic;
        }
        Matcher matcher = this.topicRegex.matcher(oldTopic);
        if (matcher.matches()) {
            newTopic = matcher.replaceFirst(this.topicReplacement);
            this.topicRegexReplaceCache.put((Object)oldTopic, (Object)newTopic);
            return newTopic;
        }
        return null;
    }

    private Schema updateKeySchema(Schema oldKeySchema, String newTopicName) {
        Schema newKeySchema = (Schema)this.keySchemaUpdateCache.get((Object)oldKeySchema);
        if (newKeySchema != null) {
            return newKeySchema;
        }
        SchemaBuilder builder = this.copySchemaExcludingName(oldKeySchema, SchemaBuilder.struct());
        builder.name(this.schemaNameAdjuster.adjust(newTopicName + ".Key"));
        builder.field(this.keyFieldName, Schema.STRING_SCHEMA);
        newKeySchema = builder.build();
        this.keySchemaUpdateCache.put((Object)oldKeySchema, (Object)newKeySchema);
        return newKeySchema;
    }

    private Struct updateKey(Schema newKeySchema, Struct oldKey, String oldTopic) {
        Struct newKey = new Struct(newKeySchema);
        for (org.apache.kafka.connect.data.Field field : oldKey.schema().fields()) {
            newKey.put(field.name(), oldKey.get(field));
        }
        String physicalTableIdentifier = oldTopic;
        if (this.keyFieldRegex != null && (physicalTableIdentifier = (String)this.keyRegexReplaceCache.get((Object)oldTopic)) == null) {
            Matcher matcher = this.keyFieldRegex.matcher(oldTopic);
            if (matcher.matches()) {
                physicalTableIdentifier = matcher.replaceFirst(this.keyFieldReplacement);
                this.keyRegexReplaceCache.put((Object)oldTopic, (Object)physicalTableIdentifier);
            } else {
                physicalTableIdentifier = oldTopic;
            }
        }
        newKey.put(this.keyFieldName, (Object)physicalTableIdentifier);
        return newKey;
    }

    private Schema updateEnvelopeSchema(Schema oldEnvelopeSchema, String newTopicName) {
        Schema newEnvelopeSchema = (Schema)this.envelopeSchemaUpdateCache.get((Object)oldEnvelopeSchema);
        if (newEnvelopeSchema != null) {
            return newEnvelopeSchema;
        }
        Schema oldValueSchema = oldEnvelopeSchema.field("before").schema();
        SchemaBuilder valueBuilder = this.copySchemaExcludingName(oldValueSchema, SchemaBuilder.struct());
        valueBuilder.name(this.schemaNameAdjuster.adjust(newTopicName + ".Value"));
        Schema newValueSchema = valueBuilder.build();
        SchemaBuilder envelopeBuilder = this.copySchemaExcludingName(oldEnvelopeSchema, SchemaBuilder.struct(), false);
        for (org.apache.kafka.connect.data.Field field : oldEnvelopeSchema.fields()) {
            String fieldName = field.name();
            Schema fieldSchema = field.schema();
            if (Objects.equals(fieldName, "before") || Objects.equals(fieldName, "after")) {
                fieldSchema = newValueSchema;
            }
            envelopeBuilder.field(fieldName, fieldSchema);
        }
        envelopeBuilder.name(this.schemaNameAdjuster.adjust(Envelope.schemaName(newTopicName)));
        newEnvelopeSchema = envelopeBuilder.build();
        this.envelopeSchemaUpdateCache.put((Object)oldEnvelopeSchema, (Object)newEnvelopeSchema);
        return newEnvelopeSchema;
    }

    private Struct updateEnvelope(Schema newEnvelopeSchema, Struct oldEnvelope) {
        Struct newEnvelope = new Struct(newEnvelopeSchema);
        Schema newValueSchema = newEnvelopeSchema.field("before").schema();
        for (org.apache.kafka.connect.data.Field field : oldEnvelope.schema().fields()) {
            String fieldName = field.name();
            Object fieldValue = oldEnvelope.get(field);
            if ((Objects.equals(fieldName, "before") || Objects.equals(fieldName, "after")) && fieldValue != null) {
                fieldValue = this.updateValue(newValueSchema, Requirements.requireStruct((Object)fieldValue, (String)"Updating schema"));
            }
            newEnvelope.put(fieldName, fieldValue);
        }
        return newEnvelope;
    }

    private Struct updateValue(Schema newValueSchema, Struct oldValue) {
        Struct newValue = new Struct(newValueSchema);
        for (org.apache.kafka.connect.data.Field field : oldValue.schema().fields()) {
            newValue.put(field.name(), oldValue.get(field));
        }
        return newValue;
    }

    private SchemaBuilder copySchemaExcludingName(Schema source, SchemaBuilder builder) {
        return this.copySchemaExcludingName(source, builder, true);
    }

    private SchemaBuilder copySchemaExcludingName(Schema source, SchemaBuilder builder, boolean copyFields) {
        builder.version(source.version());
        builder.doc(source.doc());
        Map params = source.parameters();
        if (params != null) {
            builder.parameters(params);
        }
        if (source.isOptional()) {
            builder.optional();
        } else {
            builder.required();
        }
        if (copyFields) {
            for (org.apache.kafka.connect.data.Field field : source.fields()) {
                builder.field(field.name(), field.schema());
            }
        }
        return builder;
    }
}

