/*
 * Decompiled with CFR 0.152.
 */
package org.infinispan.distribution.topologyaware;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import org.infinispan.distribution.topologyaware.TopologyLevel;
import org.infinispan.remoting.transport.Address;
import org.infinispan.remoting.transport.TopologyAwareAddress;

public class TopologyInfo {
    private final int numSegments;
    private final int numOwners;
    private final Cluster cluster = new Cluster();
    private final List<Rack> allRacks = new ArrayList<Rack>();
    private final List<Machine> allMachines = new ArrayList<Machine>();
    private final List<Node> allNodes = new ArrayList<Node>();
    private final Map<Address, Node> addressMap = new HashMap<Address, Node>();

    public TopologyInfo(int numSegments, int numOwners, Collection<Address> members, Map<Address, Float> capacityFactors) {
        this.numOwners = Math.min(numOwners, members.size());
        this.numSegments = numSegments;
        for (Address node : members) {
            float capacityFactor = capacityFactors != null ? capacityFactors.get(node).floatValue() : 1.0f;
            if (capacityFactor == 0.0f) continue;
            this.addNode(node, capacityFactor);
        }
        if (this.cluster.totalCapacity == 0.0f) {
            throw new IllegalArgumentException("At least one node should have non-zero capacity");
        }
        Collections.sort(this.cluster.sites);
        for (Site site : this.cluster.sites) {
            Collections.sort(site.racks);
            for (Rack rack : site.racks) {
                this.allRacks.add(rack);
                Collections.sort(rack.machines);
                for (Machine machine : rack.machines) {
                    this.allMachines.add(machine);
                    Collections.sort(machine.nodes);
                    for (Node node : machine.nodes) {
                        this.allNodes.add(node);
                        this.addressMap.put(node.address, node);
                    }
                }
            }
        }
        this.computeExpectedSegments();
    }

    public int getDistinctLocationsCount(TopologyLevel level) {
        switch (level) {
            case NODE: {
                return this.allNodes.size();
            }
            case MACHINE: {
                return this.allMachines.size();
            }
            case RACK: {
                return this.allRacks.size();
            }
            case SITE: {
                return this.cluster.sites.size();
            }
        }
        throw new IllegalArgumentException("Unknown level: " + String.valueOf((Object)level));
    }

    public int getDistinctLocationsCount(TopologyLevel level, Collection<Address> addresses) {
        HashSet<Object> locations = new HashSet<Object>();
        for (Address address : addresses) {
            locations.add(this.getLocationId(level, address));
        }
        return locations.size();
    }

    public boolean duplicateLocation(TopologyLevel level, Collection<Address> addresses, Address candidate, boolean excludeCandidate) {
        Object newLocationId = this.getLocationId(level, candidate);
        for (Address address : addresses) {
            if (excludeCandidate && address.equals(candidate) || !newLocationId.equals(this.getLocationId(level, address))) continue;
            return true;
        }
        return false;
    }

    public Object getLocationId(TopologyLevel level, Address address) {
        Location locationId;
        Node node = this.addressMap.get(address);
        switch (level) {
            case SITE: {
                locationId = node.machine.rack.site;
                break;
            }
            case RACK: {
                locationId = node.machine.rack;
                break;
            }
            case MACHINE: {
                locationId = node.machine;
                break;
            }
            case NODE: {
                locationId = node;
                break;
            }
            default: {
                throw new IllegalStateException("Unexpected value: " + String.valueOf((Object)level));
            }
        }
        return locationId;
    }

    private void addNode(Address address, float capacityFactor) {
        TopologyAwareAddress taa = (TopologyAwareAddress)address;
        String siteId = taa.getSiteId();
        String rackId = taa.getRackId();
        String machineId = taa.getMachineId();
        this.cluster.addNode(siteId, rackId, machineId, address, capacityFactor);
    }

    public Collection<Address> getSiteNodes(String site) {
        ArrayList<Address> addresses = new ArrayList<Address>();
        this.cluster.getSite(site).collectNodes(addresses);
        return addresses;
    }

    public Collection<Address> getRackNodes(String site, String rack) {
        ArrayList<Address> addresses = new ArrayList<Address>();
        this.cluster.getSite(site).getRack(rack).collectNodes(addresses);
        return addresses;
    }

    public Collection<Address> getMachineNodes(String site, String rack, String machine) {
        ArrayList<Address> addresses = new ArrayList<Address>();
        this.cluster.getSite(site).getRack(rack).getMachine(machine).collectNodes(addresses);
        return addresses;
    }

    public Collection<String> getAllSites() {
        return this.cluster.getChildNames();
    }

    public Collection<String> getSiteRacks(String site) {
        return this.cluster.getSite(site).getChildNames();
    }

    public Collection<String> getRackMachines(String site, String rack) {
        return this.cluster.getSite(site).getRack(rack).getChildNames();
    }

    public String toString() {
        DecimalFormat df = new DecimalFormat("0", DecimalFormatSymbols.getInstance(Locale.ENGLISH));
        df.setMaximumFractionDigits(2);
        StringBuilder sb = new StringBuilder("TopologyInfo{\n");
        sb.append(this.formatLocation(df, this.cluster, ""));
        for (Site site : this.cluster.sites) {
            sb.append(this.formatLocation(df, site, "  "));
            for (Rack rack : site.racks) {
                sb.append(this.formatLocation(df, rack, "    "));
                for (Machine machine : rack.machines) {
                    sb.append(this.formatLocation(df, machine, "      "));
                    for (Node node : machine.nodes) {
                        sb.append(this.formatLocation(df, node, "        "));
                    }
                }
            }
        }
        sb.append("}");
        return sb.toString();
    }

    public String formatLocation(DecimalFormat df, Location location, String prefix) {
        return String.format("%s%s * %s: %s+%s %n", prefix, location.getName(), df.format(location.totalCapacity), df.format(location.expectedPrimarySegments), df.format(location.getExpectedBackupSegments()));
    }

    private void computeExpectedSegments() {
        this.splitPrimarySegments();
        this.splitExpectedOwnedSegments(this.cluster.getChildren(), this.numSegments * this.numOwners, this.cluster.totalCapacity);
    }

    private void splitPrimarySegments() {
        for (Node node : this.allNodes) {
            float fraction = node.totalCapacity / this.cluster.totalCapacity;
            node.addPrimarySegments((float)this.numSegments * fraction);
        }
    }

    private void splitExpectedOwnedSegments(Collection<? extends Location> locations, float totalOwnedSegments, float totalCapacity) {
        float locationOwned;
        Location location;
        float remainingCapacity = totalCapacity;
        float remainingOwned = totalOwnedSegments;
        ArrayList<? extends Location> remainingLocations = new ArrayList<Location>(locations);
        Iterator it = remainingLocations.listIterator(locations.size());
        while (it.hasPrevious()) {
            int n;
            location = (Location)it.previous();
            if (remainingOwned < (float)(this.numSegments * remainingLocations.size()) || (locationOwned = remainingOwned * location.totalCapacity / remainingCapacity) > (float)(n = this.numSegments)) break;
            this.splitExpectedOwnedSegments2(location.getChildren(), n, location.totalCapacity);
            remainingCapacity -= location.totalCapacity;
            remainingOwned -= location.expectedOwnedSegments;
            it.remove();
        }
        it = remainingLocations.iterator();
        while (it.hasNext()) {
            location = (Location)it.next();
            locationOwned = remainingOwned * location.totalCapacity / remainingCapacity;
            float f = this.computeMaxOwned(remainingOwned, remainingLocations.size());
            if (locationOwned < f) break;
            this.splitExpectedOwnedSegments2(location.getChildren(), f, location.totalCapacity);
            remainingCapacity -= location.totalCapacity;
            remainingOwned -= f;
            it.remove();
        }
        if (remainingLocations.isEmpty()) {
            return;
        }
        if ((float)(remainingLocations.size() * this.numSegments) < remainingOwned) {
            ArrayList<? extends Location> childrenLocations = new ArrayList<Location>(remainingLocations.size() * 2);
            for (Location location2 : remainingLocations) {
                childrenLocations.addAll(location2.getChildren());
            }
            Collections.sort(childrenLocations);
            this.splitExpectedOwnedSegments2(childrenLocations, remainingOwned, remainingCapacity);
        } else {
            float fraction = remainingOwned / remainingCapacity;
            for (Location location3 : remainingLocations) {
                locationOwned = location3.totalCapacity * fraction;
                this.splitExpectedOwnedSegments2(location3.getChildren(), locationOwned, location3.totalCapacity);
            }
        }
    }

    private float computeMaxOwned(float remainingOwned, int locationsCount) {
        float maxOwned = remainingOwned < (float)this.numSegments ? remainingOwned : (remainingOwned < (float)(this.numSegments * locationsCount) ? (float)this.numSegments : remainingOwned - (float)(this.numSegments * (locationsCount - 1)));
        return maxOwned;
    }

    private void splitExpectedOwnedSegments2(Collection<? extends Location> locations, float totalOwnedSegments, float totalCapacity) {
        Location first = locations.iterator().next();
        if (locations.size() == 1 && first instanceof Node) {
            ((Node)first).addOwnedSegments(totalOwnedSegments);
        } else {
            this.splitExpectedOwnedSegments(locations, totalOwnedSegments, totalCapacity);
        }
    }

    public float computeTotalCapacity(Collection<Address> nodes, Map<Address, Float> capacityFactors) {
        if (capacityFactors == null) {
            return nodes.size();
        }
        float totalCapacity = 0.0f;
        for (Address node : nodes) {
            totalCapacity += capacityFactors.get(node).floatValue();
        }
        return totalCapacity;
    }

    public float getExpectedPrimarySegments(Address address) {
        Node node = this.addressMap.get(address);
        return node != null ? node.expectedPrimarySegments : 0.0f;
    }

    public float getExpectedOwnedSegments(Address address) {
        Node node = this.addressMap.get(address);
        return node != null ? node.expectedOwnedSegments : 0.0f;
    }

    public int getSiteIndex(Address address) {
        Site site = this.addressMap.get(address).getSite();
        return this.cluster.sites.indexOf(site);
    }

    public int getRackIndex(Address address) {
        Rack rack = this.addressMap.get(address).getRack();
        return this.allRacks.indexOf(rack);
    }

    public int getMachineIndex(Address address) {
        Machine machine = this.addressMap.get(address).getMachine();
        return this.allMachines.indexOf(machine);
    }

    public static class Cluster
    extends Location {
        final List<Site> sites = new ArrayList<Site>();
        final Map<String, Site> siteMap = new HashMap<String, Site>();

        void addNode(String siteId, String rackId, String machineId, Address address, float capacityFactor) {
            Site site = this.siteMap.get(siteId);
            if (site == null) {
                site = new Site(this, siteId);
                this.sites.add(site);
                this.siteMap.put(siteId, site);
            }
            site.addNode(rackId, machineId, address, capacityFactor);
            this.totalCapacity += capacityFactor;
            ++this.nodeCount;
        }

        Site getSite(String siteId) {
            return this.siteMap.get(siteId);
        }

        Collection<Site> getChildren() {
            return this.sites;
        }

        @Override
        String getName() {
            return "cluster";
        }

        @Override
        String getFullName() {
            return "";
        }
    }

    public static class Site
    extends Location {
        final Cluster cluster;
        final String siteId;
        final List<Rack> racks = new ArrayList<Rack>();
        final Map<String, Rack> rackMap = new HashMap<String, Rack>();

        Site(Cluster cluster, String siteId) {
            this.cluster = cluster;
            this.siteId = siteId;
        }

        void addNode(String rackId, String machineId, Address address, float capacityFactor) {
            Rack rack = this.rackMap.get(rackId);
            if (rack == null) {
                rack = new Rack(this, rackId);
                this.racks.add(rack);
                this.rackMap.put(rackId, rack);
            }
            rack.addNode(machineId, address, capacityFactor);
            this.totalCapacity += capacityFactor;
            ++this.nodeCount;
        }

        Rack getRack(String rackId) {
            return this.rackMap.get(rackId);
        }

        Collection<Rack> getChildren() {
            return this.racks;
        }

        @Override
        String getName() {
            return this.siteId;
        }

        @Override
        String getFullName() {
            return this.siteId;
        }
    }

    public static class Rack
    extends Location {
        final Site site;
        final String rackId;
        final List<Machine> machines = new ArrayList<Machine>();
        final Map<String, Machine> machineMap = new HashMap<String, Machine>();

        Rack(Site site, String rackId) {
            this.site = site;
            this.rackId = rackId;
        }

        void addNode(String machineId, Address address, float capacityFactor) {
            Machine machine = this.machineMap.get(machineId);
            if (machine == null) {
                machine = new Machine(this, machineId);
                this.machines.add(machine);
                this.machineMap.put(machineId, machine);
            }
            machine.addNode(address, capacityFactor);
            this.totalCapacity += capacityFactor;
            ++this.nodeCount;
        }

        Machine getMachine(String machineId) {
            return this.machineMap.get(machineId);
        }

        Collection<Machine> getChildren() {
            return this.machines;
        }

        @Override
        String getName() {
            return this.rackId;
        }

        @Override
        String getFullName() {
            return this.rackId + "|" + this.site.siteId;
        }
    }

    public static class Machine
    extends Location {
        final Rack rack;
        final String machineId;
        final List<Node> nodes = new ArrayList<Node>();

        Machine(Rack rack, String machineId) {
            this.rack = rack;
            this.machineId = machineId;
        }

        void addNode(Address address, float capacityFactor) {
            this.nodes.add(new Node(this, address, capacityFactor));
            this.totalCapacity += capacityFactor;
            ++this.nodeCount;
        }

        Collection<Node> getChildren() {
            return this.nodes;
        }

        @Override
        String getName() {
            return this.machineId;
        }

        @Override
        String getFullName() {
            return this.machineId + "|" + this.rack.rackId + "|" + this.rack.site.siteId;
        }
    }

    public static class Node
    extends Location {
        final Machine machine;
        final Address address;

        Node(Machine machine, Address address, float capacityFactor) {
            this.machine = machine;
            this.address = address;
            this.totalCapacity = capacityFactor;
        }

        public Machine getMachine() {
            return this.machine;
        }

        public Rack getRack() {
            return this.machine.rack;
        }

        public Site getSite() {
            return this.machine.rack.site;
        }

        Collection<Node> getChildren() {
            return Collections.singletonList(this);
        }

        @Override
        void collectNodes(Collection<Address> addressCollection) {
            addressCollection.add(this.address);
        }

        @Override
        String getName() {
            return this.address.toString();
        }

        @Override
        String getFullName() {
            return this.address.toString() + "|" + this.machine.machineId + "|" + this.machine.rack.rackId + "|" + this.machine.rack.site.siteId;
        }

        void addPrimarySegments(float segments) {
            this.expectedPrimarySegments += segments;
            this.machine.expectedPrimarySegments += segments;
            this.machine.rack.expectedPrimarySegments += segments;
            this.machine.rack.site.expectedPrimarySegments += segments;
            this.machine.rack.site.cluster.expectedPrimarySegments += segments;
        }

        void addOwnedSegments(float segments) {
            this.expectedOwnedSegments += segments;
            this.machine.expectedOwnedSegments += segments;
            this.machine.rack.expectedOwnedSegments += segments;
            this.machine.rack.site.expectedOwnedSegments += segments;
            this.machine.rack.site.cluster.expectedOwnedSegments += segments;
        }

        @Override
        public String toString() {
            return this.address.toString();
        }
    }

    public static abstract class Location
    implements Comparable<Location> {
        float totalCapacity;
        int nodeCount;
        float expectedPrimarySegments;
        float expectedOwnedSegments;

        abstract Collection<? extends Location> getChildren();

        abstract String getName();

        abstract String getFullName();

        float getCapacityPerNode() {
            return this.totalCapacity / (float)this.nodeCount;
        }

        float getExpectedBackupSegments() {
            return this.expectedOwnedSegments - this.expectedPrimarySegments;
        }

        void collectNodes(Collection<Address> addressCollection) {
            for (Location location : this.getChildren()) {
                location.collectNodes(addressCollection);
            }
        }

        public Collection<String> getChildNames() {
            ArrayList<String> names = new ArrayList<String>();
            for (Location location : this.getChildren()) {
                names.add(location.getName());
            }
            return names;
        }

        @Override
        public int compareTo(Location o) {
            return Float.compare(o.totalCapacity, this.totalCapacity);
        }

        public String toString() {
            String name = this.getFullName();
            return String.format("%s * %f: %.2f+%.2f", name != null ? name : "/", Float.valueOf(this.totalCapacity), Float.valueOf(this.expectedPrimarySegments), Float.valueOf(this.getExpectedBackupSegments()));
        }
    }
}

