/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.connector.mongodb.connection;

import com.mongodb.client.MongoClient;
import io.debezium.DebeziumException;
import io.debezium.config.Configuration;
import io.debezium.connector.mongodb.CollectionId;
import io.debezium.connector.mongodb.Filters;
import io.debezium.connector.mongodb.MongoDbConnectorConfig;
import io.debezium.connector.mongodb.MongoUtils;
import io.debezium.connector.mongodb.connection.MongoDbConnectionContext;
import io.debezium.function.BlockingConsumer;
import io.debezium.function.BlockingFunction;
import io.debezium.util.Clock;
import io.debezium.util.Metronome;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import org.bson.BsonTimestamp;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class MongoDbConnection
implements AutoCloseable {
    public static final Logger LOGGER = LoggerFactory.getLogger(MongoDbConnection.class);
    public static final String AUTHORIZATION_FAILURE_MESSAGE = "Command failed with error 13";
    private static final Duration PAUSE_AFTER_ERROR = Duration.ofMillis(500L);
    private final Filters filters;
    private final ErrorHandler errorHandler;
    private final AtomicBoolean running = new AtomicBoolean(true);
    private final MongoDbConnectorConfig connectorConfig;
    private final MongoDbConnectionContext connectionContext;

    MongoDbConnection(Configuration config, ErrorHandler errorHandler) {
        this.connectionContext = new MongoDbConnectionContext(config);
        this.connectorConfig = this.connectionContext.getConnectorConfig();
        this.filters = new Filters(config);
        this.errorHandler = errorHandler;
    }

    public MongoClient getMongoClient() {
        return this.connectionContext.getMongoClient();
    }

    public void execute(String desc, BlockingConsumer<MongoClient> operation) throws InterruptedException {
        this.execute(desc, client -> {
            operation.accept(client);
            return null;
        });
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public <T> T execute(String desc, BlockingFunction<MongoClient, T> operation) throws InterruptedException {
        Metronome errorMetronome = Metronome.sleeper((Duration)PAUSE_AFTER_ERROR, (Clock)Clock.SYSTEM);
        while (true) {
            try (MongoClient client = this.getMongoClient();){
                Object object = operation.apply((Object)client);
                return (T)object;
            }
            catch (InterruptedException e) {
                throw e;
            }
            catch (Throwable t) {
                this.errorHandler.onError(desc, t);
                if (!this.isRunning()) {
                    throw new DebeziumException("Operation failed and MongoDB connection to '" + this.connectionContext.getMaskedConnectionString() + "' termination requested", t);
                }
                errorMetronome.pause();
                continue;
            }
            break;
        }
    }

    public Set<String> databaseNames() throws InterruptedException {
        if (this.connectorConfig.getCaptureScope() == MongoDbConnectorConfig.CaptureScope.DATABASE) {
            return this.connectorConfig.getCaptureTarget().filter(dbName -> this.filters.databaseFilter().test((String)dbName)).map(Set::of).orElse(Set.of());
        }
        return (Set)this.execute("get database names", client -> {
            HashSet databaseNames = new HashSet();
            MongoUtils.forEachDatabaseName(client, dbName -> {
                if (this.filters.databaseFilter().test((String)dbName)) {
                    databaseNames.add(dbName);
                }
            });
            return databaseNames;
        });
    }

    public List<CollectionId> collections() throws InterruptedException {
        return (List)this.execute("get collections in databases", client -> {
            ArrayList collections = new ArrayList();
            Set<String> databaseNames = this.databaseNames();
            for (String dbName : databaseNames) {
                MongoUtils.forEachCollectionNameInDatabase(client, dbName, collectionName -> {
                    CollectionId collectionId = new CollectionId(dbName, (String)collectionName);
                    if (this.filters.collectionFilter().test(collectionId)) {
                        collections.add(collectionId);
                    }
                });
            }
            return collections;
        });
    }

    public BsonTimestamp hello() throws InterruptedException {
        return (BsonTimestamp)this.execute("ping on first available database", client -> {
            String dbName = this.databaseNames().stream().findFirst().orElse("admin");
            return MongoUtils.hello(client, dbName);
        });
    }

    private boolean isRunning() {
        return this.running.get();
    }

    @Override
    public void close() {
        this.running.set(false);
    }

    @FunctionalInterface
    public static interface ErrorHandler {
        public void onError(String var1, Throwable var2);
    }
}

