/*
 * Decompiled with CFR 0.152.
 */
package com.mongodb.internal.connection;

import com.mongodb.MongoConnectionPoolClearedException;
import com.mongodb.MongoException;
import com.mongodb.MongoInterruptedException;
import com.mongodb.MongoTimeoutException;
import com.mongodb.RequestContext;
import com.mongodb.annotations.NotThreadSafe;
import com.mongodb.annotations.ThreadSafe;
import com.mongodb.assertions.Assertions;
import com.mongodb.connection.ConnectionDescription;
import com.mongodb.connection.ConnectionId;
import com.mongodb.connection.ConnectionPoolSettings;
import com.mongodb.connection.ServerDescription;
import com.mongodb.connection.ServerId;
import com.mongodb.diagnostics.logging.Logger;
import com.mongodb.diagnostics.logging.Loggers;
import com.mongodb.event.ConnectionAddedEvent;
import com.mongodb.event.ConnectionCheckOutFailedEvent;
import com.mongodb.event.ConnectionCheckOutStartedEvent;
import com.mongodb.event.ConnectionCheckedInEvent;
import com.mongodb.event.ConnectionCheckedOutEvent;
import com.mongodb.event.ConnectionClosedEvent;
import com.mongodb.event.ConnectionCreatedEvent;
import com.mongodb.event.ConnectionPoolClearedEvent;
import com.mongodb.event.ConnectionPoolClosedEvent;
import com.mongodb.event.ConnectionPoolCreatedEvent;
import com.mongodb.event.ConnectionPoolListener;
import com.mongodb.event.ConnectionPoolOpenedEvent;
import com.mongodb.event.ConnectionPoolReadyEvent;
import com.mongodb.event.ConnectionReadyEvent;
import com.mongodb.event.ConnectionRemovedEvent;
import com.mongodb.internal.Timeout;
import com.mongodb.internal.async.ErrorHandlingResultCallback;
import com.mongodb.internal.async.SingleResultCallback;
import com.mongodb.internal.connection.CommandMessage;
import com.mongodb.internal.connection.ConcurrentPool;
import com.mongodb.internal.connection.Connection;
import com.mongodb.internal.connection.ConnectionGenerationSupplier;
import com.mongodb.internal.connection.ConnectionPool;
import com.mongodb.internal.connection.InternalConnection;
import com.mongodb.internal.connection.InternalConnectionFactory;
import com.mongodb.internal.connection.InternalConnectionPoolSettings;
import com.mongodb.internal.connection.ResponseBuffers;
import com.mongodb.internal.connection.SdamServerDescriptionManager;
import com.mongodb.internal.connection.UsageTrackingInternalConnection;
import com.mongodb.internal.event.EventListenerHelper;
import com.mongodb.internal.inject.OptionalProvider;
import com.mongodb.internal.session.SessionContext;
import com.mongodb.internal.thread.DaemonThreadFactory;
import com.mongodb.lang.NonNull;
import com.mongodb.lang.Nullable;
import java.util.Deque;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.LongAdder;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.StampedLock;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.function.Supplier;
import org.bson.ByteBuf;
import org.bson.codecs.Decoder;
import org.bson.types.ObjectId;

class DefaultConnectionPool
implements ConnectionPool {
    private static final Logger LOGGER = Loggers.getLogger("connection");
    private final ConcurrentPool<UsageTrackingInternalConnection> pool;
    private final ConnectionPoolSettings settings;
    private final BackgroundMaintenanceManager backgroundMaintenance;
    private final AsyncWorkManager asyncWorkManager;
    private final ConnectionPoolListener connectionPoolListener;
    private final ServerId serverId;
    private final PinnedStatsManager pinnedStatsManager = new PinnedStatsManager();
    private final ServiceStateManager serviceStateManager = new ServiceStateManager();
    private final ConnectionGenerationSupplier connectionGenerationSupplier;
    private final OpenConcurrencyLimiter openConcurrencyLimiter;
    private final StateAndGeneration stateAndGeneration;
    private final OptionalProvider<SdamServerDescriptionManager> sdamProvider;

    DefaultConnectionPool(ServerId serverId, InternalConnectionFactory internalConnectionFactory, ConnectionPoolSettings settings, OptionalProvider<SdamServerDescriptionManager> sdamProvider) {
        this(serverId, internalConnectionFactory, settings, InternalConnectionPoolSettings.builder().build(), sdamProvider);
    }

    DefaultConnectionPool(ServerId serverId, InternalConnectionFactory internalConnectionFactory, ConnectionPoolSettings settings, InternalConnectionPoolSettings internalSettings, OptionalProvider<SdamServerDescriptionManager> sdamProvider) {
        this.serverId = Assertions.notNull("serverId", serverId);
        this.settings = Assertions.notNull("settings", settings);
        UsageTrackingInternalConnectionItemFactory connectionItemFactory = new UsageTrackingInternalConnectionItemFactory(internalConnectionFactory);
        this.pool = new ConcurrentPool<UsageTrackingInternalConnection>(DefaultConnectionPool.maxSize(settings), connectionItemFactory, String.format("The server at %s is no longer available", serverId.getAddress()));
        this.sdamProvider = Assertions.assertNotNull(sdamProvider);
        this.connectionPoolListener = EventListenerHelper.getConnectionPoolListener(settings);
        this.backgroundMaintenance = new BackgroundMaintenanceManager();
        this.connectionPoolCreated(this.connectionPoolListener, serverId, settings);
        this.openConcurrencyLimiter = new OpenConcurrencyLimiter(settings.getMaxConnecting());
        this.asyncWorkManager = new AsyncWorkManager(internalSettings.isPrestartAsyncWorkManager());
        this.stateAndGeneration = new StateAndGeneration();
        this.connectionGenerationSupplier = new ConnectionGenerationSupplier(){

            @Override
            public int getGeneration() {
                return DefaultConnectionPool.this.stateAndGeneration.generation();
            }

            @Override
            public int getGeneration(@NonNull ObjectId serviceId) {
                return DefaultConnectionPool.this.serviceStateManager.getGeneration(serviceId);
            }
        };
    }

    @Override
    public InternalConnection get() {
        return this.get(this.settings.getMaxWaitTime(TimeUnit.MILLISECONDS), TimeUnit.MILLISECONDS);
    }

    @Override
    public InternalConnection get(long timeoutValue, TimeUnit timeUnit) {
        this.connectionPoolListener.connectionCheckOutStarted(new ConnectionCheckOutStartedEvent(this.serverId));
        Timeout timeout = Timeout.startNow(timeoutValue, timeUnit);
        try {
            this.stateAndGeneration.throwIfClosedOrPaused();
            PooledConnection connection = this.getPooledConnection(timeout);
            if (!connection.opened()) {
                connection = this.openConcurrencyLimiter.openOrGetAvailable(connection, timeout);
            }
            this.connectionPoolListener.connectionCheckedOut(new ConnectionCheckedOutEvent(this.getId(connection)));
            return connection;
        }
        catch (Exception e) {
            throw (RuntimeException)this.checkOutFailed(e);
        }
    }

    @Override
    public void getAsync(SingleResultCallback<InternalConnection> callback) {
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace(String.format("Asynchronously getting a connection from the pool for server %s", this.serverId));
        }
        this.connectionPoolListener.connectionCheckOutStarted(new ConnectionCheckOutStartedEvent(this.serverId));
        Timeout timeout = Timeout.startNow(this.settings.getMaxWaitTime(TimeUnit.NANOSECONDS));
        SingleResultCallback<InternalConnection> eventSendingCallback = (result, failure) -> {
            SingleResultCallback<InternalConnection> errHandlingCallback = ErrorHandlingResultCallback.errorHandlingCallback(callback, LOGGER);
            if (failure == null) {
                this.connectionPoolListener.connectionCheckedOut(new ConnectionCheckedOutEvent(this.getId((InternalConnection)result)));
                errHandlingCallback.onResult((InternalConnection)result, null);
            } else {
                errHandlingCallback.onResult(null, this.checkOutFailed(failure));
            }
        };
        try {
            this.stateAndGeneration.throwIfClosedOrPaused();
        }
        catch (Exception e) {
            eventSendingCallback.onResult(null, e);
            return;
        }
        this.asyncWorkManager.enqueue(new Task(timeout, t -> {
            if (t != null) {
                eventSendingCallback.onResult((InternalConnection)null, (Throwable)t);
            } else {
                PooledConnection connection;
                try {
                    connection = this.getPooledConnection(timeout);
                }
                catch (Exception e) {
                    eventSendingCallback.onResult(null, e);
                    return;
                }
                if (connection.opened()) {
                    if (LOGGER.isTraceEnabled()) {
                        LOGGER.trace(String.format("Pooled connection %s to server %s is already open", this.getId(connection), this.serverId));
                    }
                    eventSendingCallback.onResult(connection, null);
                } else {
                    if (LOGGER.isTraceEnabled()) {
                        LOGGER.trace(String.format("Pooled connection %s to server %s is not yet open", this.getId(connection), this.serverId));
                    }
                    this.openConcurrencyLimiter.openAsyncWithConcurrencyLimit(connection, timeout, eventSendingCallback);
                }
            }
        }));
    }

    private Throwable checkOutFailed(Throwable t) {
        Throwable result = t;
        if (t instanceof MongoTimeoutException) {
            this.connectionPoolListener.connectionCheckOutFailed(new ConnectionCheckOutFailedEvent(this.serverId, ConnectionCheckOutFailedEvent.Reason.TIMEOUT));
        } else if (t instanceof MongoOpenConnectionInternalException) {
            this.connectionPoolListener.connectionCheckOutFailed(new ConnectionCheckOutFailedEvent(this.serverId, ConnectionCheckOutFailedEvent.Reason.CONNECTION_ERROR));
            result = t.getCause();
        } else if (t instanceof MongoConnectionPoolClearedException) {
            this.connectionPoolListener.connectionCheckOutFailed(new ConnectionCheckOutFailedEvent(this.serverId, ConnectionCheckOutFailedEvent.Reason.CONNECTION_ERROR));
        } else if (ConcurrentPool.isPoolClosedException(t)) {
            this.connectionPoolListener.connectionCheckOutFailed(new ConnectionCheckOutFailedEvent(this.serverId, ConnectionCheckOutFailedEvent.Reason.POOL_CLOSED));
        } else {
            this.connectionPoolListener.connectionCheckOutFailed(new ConnectionCheckOutFailedEvent(this.serverId, ConnectionCheckOutFailedEvent.Reason.UNKNOWN));
        }
        return result;
    }

    @Override
    public void invalidate(@Nullable Throwable cause) {
        Assertions.assertFalse(this.isLoadBalanced());
        if (this.stateAndGeneration.pauseAndIncrementGeneration(cause)) {
            LOGGER.debug("Invalidating the connection pool for " + this.serverId + " and marking it as 'paused'" + (cause == null ? "" : " due to " + cause));
            this.openConcurrencyLimiter.signalClosedOrPaused();
        }
    }

    @Override
    public void ready() {
        if (this.stateAndGeneration.ready()) {
            LOGGER.debug("Marking the connection pool for " + this.serverId + " as 'ready'");
        }
    }

    @Override
    public void invalidate(ObjectId serviceId, int generation) {
        Assertions.assertTrue(this.isLoadBalanced());
        if (generation == -1) {
            return;
        }
        if (this.serviceStateManager.incrementGeneration(serviceId, generation)) {
            LOGGER.debug("Invalidating the connection pool for server id " + serviceId);
            this.connectionPoolListener.connectionPoolCleared(new ConnectionPoolClearedEvent(this.serverId, serviceId));
        }
    }

    @Override
    public void close() {
        if (this.stateAndGeneration.close()) {
            this.pool.close();
            this.backgroundMaintenance.close();
            this.asyncWorkManager.close();
            this.openConcurrencyLimiter.signalClosedOrPaused();
            this.connectionPoolListener.connectionPoolClosed(new ConnectionPoolClosedEvent(this.serverId));
        }
    }

    @Override
    public int getGeneration() {
        return this.stateAndGeneration.generation();
    }

    private PooledConnection getPooledConnection(Timeout timeout) throws MongoTimeoutException {
        try {
            UsageTrackingInternalConnection internalConnection = this.pool.get(timeout.remainingOrInfinite(TimeUnit.NANOSECONDS), TimeUnit.NANOSECONDS);
            while (this.shouldPrune(internalConnection)) {
                this.pool.release(internalConnection, true);
                internalConnection = this.pool.get(timeout.remainingOrInfinite(TimeUnit.NANOSECONDS), TimeUnit.NANOSECONDS);
            }
            return new PooledConnection(internalConnection);
        }
        catch (MongoTimeoutException e) {
            throw this.createTimeoutException(timeout);
        }
    }

    @Nullable
    private PooledConnection getPooledConnectionImmediateUnfair() {
        UsageTrackingInternalConnection internalConnection = this.pool.getImmediateUnfair();
        while (internalConnection != null && this.shouldPrune(internalConnection)) {
            this.pool.release(internalConnection, true);
            internalConnection = this.pool.getImmediateUnfair();
        }
        return internalConnection == null ? null : new PooledConnection(internalConnection);
    }

    private MongoTimeoutException createTimeoutException(Timeout timeout) {
        int numOtherInUse;
        int numPinnedToCursor = this.pinnedStatsManager.getNumPinnedToCursor();
        int numPinnedToTransaction = this.pinnedStatsManager.getNumPinnedToTransaction();
        if (numPinnedToCursor == 0 && numPinnedToTransaction == 0) {
            return new MongoTimeoutException(String.format("Timed out after %s while waiting for a connection to server %s.", timeout.toUserString(), this.serverId.getAddress()));
        }
        int maxSize = this.pool.getMaxSize();
        int numInUse = this.pool.getInUseCount();
        if (numInUse == 0) {
            numInUse = Math.min(numPinnedToCursor + numPinnedToTransaction, maxSize);
        }
        Assertions.assertTrue((numOtherInUse = numInUse - (numPinnedToCursor = Math.min(numPinnedToCursor, numInUse)) - (numPinnedToTransaction = Math.min(numPinnedToTransaction, numInUse - numPinnedToCursor))) >= 0);
        Assertions.assertTrue(numPinnedToCursor + numPinnedToTransaction + numOtherInUse <= maxSize);
        return new MongoTimeoutException(String.format("Timed out after %s while waiting for a connection to server %s. Details: maxPoolSize: %s, connections in use by cursors: %d, connections in use by transactions: %d, connections in use by other operations: %d", timeout.toUserString(), this.serverId.getAddress(), ConcurrentPool.sizeToString(maxSize), numPinnedToCursor, numPinnedToTransaction, numOtherInUse));
    }

    ConcurrentPool<UsageTrackingInternalConnection> getPool() {
        return this.pool;
    }

    void doMaintenance() {
        block5: {
            Predicate<Exception> silentlyComplete = e -> e instanceof MongoInterruptedException || e instanceof MongoTimeoutException || e instanceof MongoConnectionPoolClearedException || ConcurrentPool.isPoolClosedException(e);
            try {
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug(String.format("Pruning pooled connections to %s", this.serverId.getAddress()));
                }
                this.pool.prune();
                if (this.shouldEnsureMinSize()) {
                    if (LOGGER.isDebugEnabled()) {
                        LOGGER.debug(String.format("Ensuring minimum pooled connections to %s", this.serverId.getAddress()));
                    }
                    this.pool.ensureMinSize(this.settings.getMinSize(), newConnection -> {
                        try {
                            this.openConcurrencyLimiter.openImmediatelyAndTryHandOverOrRelease(new PooledConnection((UsageTrackingInternalConnection)newConnection));
                        }
                        catch (MongoException | MongoOpenConnectionInternalException e) {
                            RuntimeException actualException = e instanceof MongoOpenConnectionInternalException ? (RuntimeException)e.getCause() : e;
                            this.sdamProvider.optional().ifPresent(sdam -> {
                                if (!silentlyComplete.test(actualException)) {
                                    sdam.handleExceptionBeforeHandshake(SdamServerDescriptionManager.SdamIssue.specific(actualException, sdam.context((InternalConnection)newConnection)));
                                }
                            });
                            throw actualException;
                        }
                    });
                }
            }
            catch (Exception e2) {
                if (silentlyComplete.test(e2)) break block5;
                LOGGER.warn("Exception thrown during connection pool background maintenance task", e2);
                throw e2;
            }
        }
    }

    private boolean shouldEnsureMinSize() {
        return this.settings.getMinSize() > 0;
    }

    private boolean shouldPrune(UsageTrackingInternalConnection connection) {
        return this.fromPreviousGeneration(connection) || this.pastMaxLifeTime(connection) || this.pastMaxIdleTime(connection);
    }

    private boolean pastMaxIdleTime(UsageTrackingInternalConnection connection) {
        return this.expired(connection.getLastUsedAt(), System.currentTimeMillis(), this.settings.getMaxConnectionIdleTime(TimeUnit.MILLISECONDS));
    }

    private boolean pastMaxLifeTime(UsageTrackingInternalConnection connection) {
        return this.expired(connection.getOpenedAt(), System.currentTimeMillis(), this.settings.getMaxConnectionLifeTime(TimeUnit.MILLISECONDS));
    }

    private boolean fromPreviousGeneration(UsageTrackingInternalConnection connection) {
        int generation = connection.getGeneration();
        if (generation == -1) {
            return false;
        }
        ObjectId serviceId = connection.getDescription().getServiceId();
        if (serviceId != null) {
            return this.serviceStateManager.getGeneration(serviceId) > generation;
        }
        return this.stateAndGeneration.generation() > generation;
    }

    private boolean expired(long startTime, long curTime, long maxTime) {
        return maxTime != 0L && curTime - startTime > maxTime;
    }

    private void connectionPoolCreated(ConnectionPoolListener connectionPoolListener, ServerId serverId, ConnectionPoolSettings settings) {
        connectionPoolListener.connectionPoolCreated(new ConnectionPoolCreatedEvent(serverId, settings));
        connectionPoolListener.connectionPoolOpened(new ConnectionPoolOpenedEvent(serverId, settings));
    }

    private void connectionCreated(ConnectionPoolListener connectionPoolListener, ConnectionId connectionId) {
        connectionPoolListener.connectionAdded(new ConnectionAddedEvent(connectionId));
        connectionPoolListener.connectionCreated(new ConnectionCreatedEvent(connectionId));
    }

    private void connectionClosed(ConnectionPoolListener connectionPoolListener, ConnectionId connectionId, ConnectionClosedEvent.Reason reason) {
        connectionPoolListener.connectionRemoved(new ConnectionRemovedEvent(connectionId, this.getReasonForRemoved(reason)));
        connectionPoolListener.connectionClosed(new ConnectionClosedEvent(connectionId, reason));
    }

    private ConnectionRemovedEvent.Reason getReasonForRemoved(ConnectionClosedEvent.Reason reason) {
        ConnectionRemovedEvent.Reason removedReason = ConnectionRemovedEvent.Reason.UNKNOWN;
        switch (reason) {
            case STALE: {
                removedReason = ConnectionRemovedEvent.Reason.STALE;
                break;
            }
            case IDLE: {
                removedReason = ConnectionRemovedEvent.Reason.MAX_IDLE_TIME_EXCEEDED;
                break;
            }
            case ERROR: {
                removedReason = ConnectionRemovedEvent.Reason.ERROR;
                break;
            }
            case POOL_CLOSED: {
                removedReason = ConnectionRemovedEvent.Reason.POOL_CLOSED;
                break;
            }
        }
        return removedReason;
    }

    private ConnectionId getId(InternalConnection internalConnection) {
        return internalConnection.getDescription().getConnectionId();
    }

    private boolean isLoadBalanced() {
        return !this.sdamProvider.optional().isPresent();
    }

    private static int maxSize(ConnectionPoolSettings settings) {
        return settings.getMaxSize() == 0 ? Integer.MAX_VALUE : settings.getMaxSize();
    }

    @NotThreadSafe
    private final class BackgroundMaintenanceManager
    implements AutoCloseable {
        @Nullable
        private final ScheduledExecutorService maintainer;
        @Nullable
        private Future<?> cancellationHandle;
        private boolean initialStart;

        private BackgroundMaintenanceManager() {
            this.maintainer = DefaultConnectionPool.this.settings.getMaintenanceInitialDelay(TimeUnit.NANOSECONDS) < Long.MAX_VALUE ? Executors.newSingleThreadScheduledExecutor(new DaemonThreadFactory("MaintenanceTimer")) : null;
            this.cancellationHandle = null;
            this.initialStart = true;
        }

        void start() {
            if (this.maintainer != null) {
                Assertions.assertTrue(this.cancellationHandle == null);
                this.cancellationHandle = this.ignoreRejectedExectution(() -> this.maintainer.scheduleAtFixedRate(DefaultConnectionPool.this::doMaintenance, this.initialStart ? DefaultConnectionPool.this.settings.getMaintenanceInitialDelay(TimeUnit.MILLISECONDS) : 0L, DefaultConnectionPool.this.settings.getMaintenanceFrequency(TimeUnit.MILLISECONDS), TimeUnit.MILLISECONDS));
                this.initialStart = false;
            }
        }

        void runOnceAndStop() {
            if (this.maintainer != null) {
                if (this.cancellationHandle != null) {
                    this.cancellationHandle.cancel(false);
                    this.cancellationHandle = null;
                }
                this.ignoreRejectedExectution(() -> this.maintainer.execute(DefaultConnectionPool.this::doMaintenance));
            }
        }

        @Override
        public void close() {
            if (this.maintainer != null) {
                this.maintainer.shutdownNow();
            }
        }

        private void ignoreRejectedExectution(Runnable action) {
            this.ignoreRejectedExectution(() -> {
                action.run();
                return null;
            });
        }

        @Nullable
        private <T> T ignoreRejectedExectution(Supplier<T> action) {
            try {
                return action.get();
            }
            catch (RejectedExecutionException ignored) {
                return null;
            }
        }
    }

    private class PooledConnection
    implements InternalConnection {
        private final UsageTrackingInternalConnection wrapped;
        private final AtomicBoolean isClosed = new AtomicBoolean();
        private Connection.PinningMode pinningMode;

        PooledConnection(UsageTrackingInternalConnection wrapped) {
            this.wrapped = Assertions.notNull("wrapped", wrapped);
        }

        @Override
        public int getGeneration() {
            return this.wrapped.getGeneration();
        }

        @Override
        public void open() {
            Assertions.assertFalse(this.isClosed.get());
            try {
                DefaultConnectionPool.this.connectionCreated(DefaultConnectionPool.this.connectionPoolListener, this.wrapped.getDescription().getConnectionId());
                this.wrapped.open();
            }
            catch (Exception e) {
                this.closeAndHandleOpenFailure();
                throw new MongoOpenConnectionInternalException(e);
            }
            this.handleOpenSuccess();
        }

        @Override
        public void openAsync(SingleResultCallback<Void> callback) {
            Assertions.assertFalse(this.isClosed.get());
            DefaultConnectionPool.this.connectionCreated(DefaultConnectionPool.this.connectionPoolListener, this.wrapped.getDescription().getConnectionId());
            this.wrapped.openAsync((nullResult, failure) -> {
                if (failure != null) {
                    this.closeAndHandleOpenFailure();
                    callback.onResult(null, new MongoOpenConnectionInternalException(failure));
                } else {
                    this.handleOpenSuccess();
                    callback.onResult((Void)nullResult, null);
                }
            });
        }

        @Override
        public void close() {
            if (!this.isClosed.getAndSet(true)) {
                this.unmarkAsPinned();
                DefaultConnectionPool.this.connectionPoolListener.connectionCheckedIn(new ConnectionCheckedInEvent(DefaultConnectionPool.this.getId(this.wrapped)));
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace(String.format("Checked in connection [%s] to server %s", DefaultConnectionPool.this.getId(this.wrapped), DefaultConnectionPool.this.serverId.getAddress()));
                }
                if (this.wrapped.isClosed() || DefaultConnectionPool.this.shouldPrune(this.wrapped)) {
                    DefaultConnectionPool.this.pool.release(this.wrapped, true);
                } else {
                    DefaultConnectionPool.this.openConcurrencyLimiter.tryHandOverOrRelease(this.wrapped);
                }
            }
        }

        void release() {
            if (!this.isClosed.getAndSet(true)) {
                DefaultConnectionPool.this.pool.release(this.wrapped);
            }
        }

        void closeSilently() {
            if (!this.isClosed.getAndSet(true)) {
                this.wrapped.setCloseSilently();
                DefaultConnectionPool.this.pool.release(this.wrapped, true);
            }
        }

        private void closeAndHandleOpenFailure() {
            if (!this.isClosed.getAndSet(true)) {
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace(String.format("Pooled connection %s to server %s failed to open", DefaultConnectionPool.this.getId(this), DefaultConnectionPool.this.serverId));
                }
                if (this.wrapped.getDescription().getServiceId() != null) {
                    DefaultConnectionPool.this.invalidate(this.wrapped.getDescription().getServiceId(), this.wrapped.getGeneration());
                }
                DefaultConnectionPool.this.pool.release(this.wrapped, true);
            }
        }

        private void handleOpenSuccess() {
            if (LOGGER.isTraceEnabled()) {
                LOGGER.trace(String.format("Pooled connection %s to server %s is now open", DefaultConnectionPool.this.getId(this), DefaultConnectionPool.this.serverId));
            }
            DefaultConnectionPool.this.connectionPoolListener.connectionReady(new ConnectionReadyEvent(DefaultConnectionPool.this.getId(this)));
        }

        @Override
        public boolean opened() {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.opened();
        }

        @Override
        public boolean isClosed() {
            return this.isClosed.get() || this.wrapped.isClosed();
        }

        @Override
        public ByteBuf getBuffer(int capacity) {
            return this.wrapped.getBuffer(capacity);
        }

        @Override
        public void sendMessage(List<ByteBuf> byteBuffers, int lastRequestId) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.sendMessage(byteBuffers, lastRequestId);
        }

        @Override
        public <T> T sendAndReceive(CommandMessage message, Decoder<T> decoder, SessionContext sessionContext, RequestContext requestContext) {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.sendAndReceive(message, decoder, sessionContext, requestContext);
        }

        @Override
        public <T> void send(CommandMessage message, Decoder<T> decoder, SessionContext sessionContext) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.send(message, decoder, sessionContext);
        }

        @Override
        public <T> T receive(Decoder<T> decoder, SessionContext sessionContext) {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.receive(decoder, sessionContext);
        }

        @Override
        public boolean supportsAdditionalTimeout() {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.supportsAdditionalTimeout();
        }

        @Override
        public <T> T receive(Decoder<T> decoder, SessionContext sessionContext, int additionalTimeout) {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.receive(decoder, sessionContext, additionalTimeout);
        }

        @Override
        public boolean hasMoreToCome() {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.hasMoreToCome();
        }

        @Override
        public <T> void sendAndReceiveAsync(CommandMessage message, Decoder<T> decoder, SessionContext sessionContext, RequestContext requestContext, final SingleResultCallback<T> callback) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.sendAndReceiveAsync(message, decoder, sessionContext, requestContext, new SingleResultCallback<T>(){

                @Override
                public void onResult(T result, Throwable t) {
                    callback.onResult(result, t);
                }
            });
        }

        @Override
        public ResponseBuffers receiveMessage(int responseTo) {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.receiveMessage(responseTo);
        }

        @Override
        public void sendMessageAsync(List<ByteBuf> byteBuffers, int lastRequestId, final SingleResultCallback<Void> callback) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.sendMessageAsync(byteBuffers, lastRequestId, new SingleResultCallback<Void>(){

                @Override
                public void onResult(Void result, Throwable t) {
                    callback.onResult(null, t);
                }
            });
        }

        @Override
        public void receiveMessageAsync(int responseTo, final SingleResultCallback<ResponseBuffers> callback) {
            Assertions.isTrue("open", !this.isClosed.get());
            this.wrapped.receiveMessageAsync(responseTo, new SingleResultCallback<ResponseBuffers>(){

                @Override
                public void onResult(ResponseBuffers result, Throwable t) {
                    callback.onResult(result, t);
                }
            });
        }

        @Override
        public void markAsPinned(Connection.PinningMode pinningMode) {
            Assertions.assertNotNull(pinningMode);
            if (this.pinningMode == null) {
                this.pinningMode = pinningMode;
                DefaultConnectionPool.this.pinnedStatsManager.increment(pinningMode);
            }
        }

        void unmarkAsPinned() {
            if (this.pinningMode != null) {
                DefaultConnectionPool.this.pinnedStatsManager.decrement(this.pinningMode);
            }
        }

        @Override
        public ConnectionDescription getDescription() {
            return this.wrapped.getDescription();
        }

        @Override
        public ServerDescription getInitialServerDescription() {
            Assertions.isTrue("open", !this.isClosed.get());
            return this.wrapped.getInitialServerDescription();
        }
    }

    private static final class PinnedStatsManager {
        private final LongAdder numPinnedToCursor = new LongAdder();
        private final LongAdder numPinnedToTransaction = new LongAdder();

        private PinnedStatsManager() {
        }

        void increment(Connection.PinningMode pinningMode) {
            switch (pinningMode) {
                case CURSOR: {
                    this.numPinnedToCursor.increment();
                    break;
                }
                case TRANSACTION: {
                    this.numPinnedToTransaction.increment();
                    break;
                }
                default: {
                    Assertions.fail();
                }
            }
        }

        void decrement(Connection.PinningMode pinningMode) {
            switch (pinningMode) {
                case CURSOR: {
                    this.numPinnedToCursor.decrement();
                    break;
                }
                case TRANSACTION: {
                    this.numPinnedToTransaction.decrement();
                    break;
                }
                default: {
                    Assertions.fail();
                }
            }
        }

        int getNumPinnedToCursor() {
            return this.numPinnedToCursor.intValue();
        }

        int getNumPinnedToTransaction() {
            return this.numPinnedToTransaction.intValue();
        }
    }

    @ThreadSafe
    private final class OpenConcurrencyLimiter {
        private final ReentrantLock lock = new ReentrantLock(true);
        private final Condition permitAvailableOrHandedOverOrClosedOrPausedCondition = this.lock.newCondition();
        private final int maxPermits;
        private int permits;
        private final Deque<MutableReference<PooledConnection>> desiredConnectionSlots;

        OpenConcurrencyLimiter(int maxConnecting) {
            this.permits = this.maxPermits = maxConnecting;
            this.desiredConnectionSlots = new LinkedList<MutableReference<PooledConnection>>();
        }

        PooledConnection openOrGetAvailable(PooledConnection connection, Timeout timeout) throws MongoTimeoutException {
            PooledConnection result = this.openWithConcurrencyLimit(connection, OpenWithConcurrencyLimitMode.TRY_GET_AVAILABLE, timeout);
            return Assertions.assertNotNull(result);
        }

        void openImmediatelyAndTryHandOverOrRelease(PooledConnection connection) throws MongoTimeoutException {
            Assertions.assertNull(this.openWithConcurrencyLimit(connection, OpenWithConcurrencyLimitMode.TRY_HAND_OVER_OR_RELEASE, Timeout.immediate()));
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Nullable
        private PooledConnection openWithConcurrencyLimit(PooledConnection connection, OpenWithConcurrencyLimitMode mode, Timeout timeout) throws MongoTimeoutException {
            PooledConnection availableConnection;
            try {
                availableConnection = this.acquirePermitOrGetAvailableOpenedConnection(mode == OpenWithConcurrencyLimitMode.TRY_GET_AVAILABLE, timeout);
            }
            catch (Exception e) {
                connection.closeSilently();
                throw e;
            }
            if (availableConnection != null) {
                connection.closeSilently();
                return availableConnection;
            }
            try {
                connection.open();
                if (mode == OpenWithConcurrencyLimitMode.TRY_HAND_OVER_OR_RELEASE) {
                    this.tryHandOverOrRelease(connection.wrapped);
                    PooledConnection pooledConnection = null;
                    return pooledConnection;
                }
                PooledConnection pooledConnection = connection;
                return pooledConnection;
            }
            finally {
                this.releasePermit();
            }
        }

        void openAsyncWithConcurrencyLimit(PooledConnection connection, Timeout timeout, SingleResultCallback<InternalConnection> callback) {
            PooledConnection availableConnection;
            try {
                availableConnection = this.acquirePermitOrGetAvailableOpenedConnection(true, timeout);
            }
            catch (Exception e) {
                connection.closeSilently();
                callback.onResult(null, e);
                return;
            }
            if (availableConnection != null) {
                connection.closeSilently();
                callback.onResult(availableConnection, null);
            } else {
                connection.openAsync((nullResult, failure) -> {
                    this.releasePermit();
                    if (failure != null) {
                        callback.onResult(null, failure);
                    } else {
                        callback.onResult(connection, null);
                    }
                });
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Nullable
        private PooledConnection acquirePermitOrGetAvailableOpenedConnection(boolean tryGetAvailable, Timeout timeout) throws MongoTimeoutException, MongoInterruptedException {
            PooledConnection availableConnection = null;
            boolean expressedDesireToGetAvailableConnection = false;
            ConcurrentPool.lockInterruptibly(this.lock);
            try {
                if (tryGetAvailable) {
                    availableConnection = DefaultConnectionPool.this.getPooledConnectionImmediateUnfair();
                    if (availableConnection != null) {
                        PooledConnection pooledConnection = availableConnection;
                        return pooledConnection;
                    }
                    this.expressDesireToGetAvailableConnection();
                    expressedDesireToGetAvailableConnection = true;
                }
                long remainingNanos = timeout.remainingOrInfinite(TimeUnit.NANOSECONDS);
                while (true) {
                    availableConnection = tryGetAvailable ? this.tryGetAvailableConnection() : null;
                    if (!(this.permits == 0 & !DefaultConnectionPool.this.stateAndGeneration.throwIfClosedOrPaused() & availableConnection == null)) break;
                    if (Timeout.expired(remainingNanos)) {
                        throw DefaultConnectionPool.this.createTimeoutException(timeout);
                    }
                    remainingNanos = this.awaitNanos(this.permitAvailableOrHandedOverOrClosedOrPausedCondition, remainingNanos);
                }
                if (availableConnection == null) {
                    Assertions.assertTrue(this.permits > 0);
                    --this.permits;
                }
                PooledConnection pooledConnection = availableConnection;
                return pooledConnection;
            }
            finally {
                try {
                    if (expressedDesireToGetAvailableConnection && availableConnection == null) {
                        this.giveUpOnTryingToGetAvailableConnection();
                    }
                }
                finally {
                    this.lock.unlock();
                }
            }
        }

        private void releasePermit() {
            ConcurrentPool.lockUnfair(this.lock);
            try {
                Assertions.assertTrue(this.permits < this.maxPermits);
                ++this.permits;
                this.permitAvailableOrHandedOverOrClosedOrPausedCondition.signal();
            }
            finally {
                this.lock.unlock();
            }
        }

        private void expressDesireToGetAvailableConnection() {
            this.desiredConnectionSlots.addLast(new MutableReference());
        }

        @Nullable
        private PooledConnection tryGetAvailableConnection() {
            Assertions.assertFalse(this.desiredConnectionSlots.isEmpty());
            PooledConnection result = (PooledConnection)((MutableReference)this.desiredConnectionSlots.peekFirst()).reference;
            if (result != null) {
                this.desiredConnectionSlots.removeFirst();
                Assertions.assertTrue(result.opened());
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace(String.format("Received opened connection [%s] to server %s", DefaultConnectionPool.this.getId(result), DefaultConnectionPool.this.serverId.getAddress()));
                }
            }
            return result;
        }

        private void giveUpOnTryingToGetAvailableConnection() {
            Assertions.assertFalse(this.desiredConnectionSlots.isEmpty());
            PooledConnection connection = (PooledConnection)((MutableReference)this.desiredConnectionSlots.removeLast()).reference;
            if (connection != null) {
                connection.release();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void tryHandOverOrRelease(UsageTrackingInternalConnection openConnection) {
            ConcurrentPool.lockUnfair(this.lock);
            try {
                for (MutableReference<PooledConnection> desiredConnectionSlot : this.desiredConnectionSlots) {
                    if (((MutableReference)desiredConnectionSlot).reference != null) continue;
                    ((MutableReference)desiredConnectionSlot).reference = new PooledConnection(openConnection);
                    this.permitAvailableOrHandedOverOrClosedOrPausedCondition.signal();
                    if (LOGGER.isTraceEnabled()) {
                        LOGGER.trace(String.format("Handed over opened connection [%s] to server %s", DefaultConnectionPool.this.getId(openConnection), DefaultConnectionPool.this.serverId.getAddress()));
                    }
                    return;
                }
            }
            finally {
                this.lock.unlock();
            }
            DefaultConnectionPool.this.pool.release(openConnection);
        }

        void signalClosedOrPaused() {
            ConcurrentPool.lockUnfair(this.lock);
            try {
                this.permitAvailableOrHandedOverOrClosedOrPausedCondition.signalAll();
            }
            finally {
                this.lock.unlock();
            }
        }

        private long awaitNanos(Condition condition, long timeoutNanos) throws MongoInterruptedException {
            try {
                if (timeoutNanos < 0L || timeoutNanos == Long.MAX_VALUE) {
                    condition.await();
                    return -1L;
                }
                return Math.max(0L, condition.awaitNanos(timeoutNanos));
            }
            catch (InterruptedException e) {
                throw new MongoInterruptedException(null, e);
            }
        }
    }

    static final class ServiceStateManager {
        private final ConcurrentHashMap<ObjectId, ServiceState> stateByServiceId = new ConcurrentHashMap();

        ServiceStateManager() {
        }

        void addConnection(ObjectId serviceId) {
            this.stateByServiceId.compute(serviceId, (k, v) -> {
                if (v == null) {
                    v = new ServiceState();
                }
                v.incrementConnectionCount();
                return v;
            });
        }

        void removeConnection(ObjectId serviceId) {
            this.stateByServiceId.compute(serviceId, (k, v) -> {
                Assertions.assertNotNull(v);
                return v.decrementAndGetConnectionCount() == 0 ? null : v;
            });
        }

        boolean incrementGeneration(ObjectId serviceId, int expectedGeneration) {
            ServiceState state = this.stateByServiceId.get(serviceId);
            return state == null || state.incrementGeneration(expectedGeneration);
        }

        int getGeneration(ObjectId serviceId) {
            ServiceState state = this.stateByServiceId.get(serviceId);
            return state == null ? 0 : state.getGeneration();
        }

        private static final class ServiceState {
            private final AtomicInteger generation = new AtomicInteger();
            private final AtomicInteger connectionCount = new AtomicInteger();

            private ServiceState() {
            }

            void incrementConnectionCount() {
                this.connectionCount.incrementAndGet();
            }

            int decrementAndGetConnectionCount() {
                return this.connectionCount.decrementAndGet();
            }

            boolean incrementGeneration(int expectedGeneration) {
                return this.generation.compareAndSet(expectedGeneration, expectedGeneration + 1);
            }

            public int getGeneration() {
                return this.generation.get();
            }
        }
    }

    @ThreadSafe
    private final class StateAndGeneration {
        private final ReadWriteLock lock = new StampedLock().asReadWriteLock();
        private volatile boolean paused = true;
        private final AtomicBoolean closed = new AtomicBoolean();
        private volatile int generation = 0;
        @Nullable
        private Throwable cause = null;

        StateAndGeneration() {
        }

        int generation() {
            return this.generation;
        }

        boolean pauseAndIncrementGeneration(@Nullable Throwable cause) {
            boolean result = false;
            this.lock.writeLock().lock();
            try {
                if (!this.paused) {
                    this.paused = true;
                    DefaultConnectionPool.this.pool.pause(() -> new MongoConnectionPoolClearedException(DefaultConnectionPool.this.serverId, cause));
                    result = true;
                }
                this.cause = cause;
                ++this.generation;
                if (result) {
                    DefaultConnectionPool.this.connectionPoolListener.connectionPoolCleared(new ConnectionPoolClearedEvent(DefaultConnectionPool.this.serverId));
                    DefaultConnectionPool.this.backgroundMaintenance.runOnceAndStop();
                }
            }
            finally {
                this.lock.writeLock().unlock();
            }
            return result;
        }

        boolean ready() {
            boolean result = false;
            if (this.paused) {
                this.lock.writeLock().lock();
                try {
                    if (this.paused) {
                        this.paused = false;
                        this.cause = null;
                        DefaultConnectionPool.this.pool.ready();
                        DefaultConnectionPool.this.connectionPoolListener.connectionPoolReady(new ConnectionPoolReadyEvent(DefaultConnectionPool.this.serverId));
                        DefaultConnectionPool.this.backgroundMaintenance.start();
                        result = true;
                    }
                }
                finally {
                    this.lock.writeLock().unlock();
                }
            }
            return result;
        }

        boolean close() {
            return this.closed.compareAndSet(false, true);
        }

        boolean throwIfClosedOrPaused() {
            if (this.closed.get()) {
                throw DefaultConnectionPool.this.pool.poolClosedException();
            }
            if (this.paused) {
                this.lock.readLock().lock();
                try {
                    if (this.paused) {
                        throw new MongoConnectionPoolClearedException(DefaultConnectionPool.this.serverId, this.cause);
                    }
                }
                finally {
                    this.lock.readLock().unlock();
                }
            }
            return false;
        }
    }

    private class UsageTrackingInternalConnectionItemFactory
    implements ConcurrentPool.ItemFactory<UsageTrackingInternalConnection> {
        private final InternalConnectionFactory internalConnectionFactory;

        UsageTrackingInternalConnectionItemFactory(InternalConnectionFactory internalConnectionFactory) {
            this.internalConnectionFactory = internalConnectionFactory;
        }

        @Override
        public UsageTrackingInternalConnection create() {
            return new UsageTrackingInternalConnection(this.internalConnectionFactory.create(DefaultConnectionPool.this.serverId, DefaultConnectionPool.this.connectionGenerationSupplier), DefaultConnectionPool.this.serviceStateManager);
        }

        @Override
        public void close(UsageTrackingInternalConnection connection) {
            if (connection.isCloseSilently()) {
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace(String.format("Silently closed connection [%s] to server %s", DefaultConnectionPool.this.getId(connection), DefaultConnectionPool.this.serverId.getAddress()));
                }
            } else {
                DefaultConnectionPool.this.connectionClosed(DefaultConnectionPool.this.connectionPoolListener, DefaultConnectionPool.this.getId(connection), this.getReasonForClosing(connection));
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug(String.format("Closed connection [%s] to %s because %s.", DefaultConnectionPool.this.getId(connection), DefaultConnectionPool.this.serverId.getAddress(), this.getReasonStringForClosing(connection)));
                }
            }
            connection.close();
        }

        private String getReasonStringForClosing(UsageTrackingInternalConnection connection) {
            String reason = connection.isClosed() ? "there was a socket exception raised by this connection" : (DefaultConnectionPool.this.fromPreviousGeneration(connection) ? "there was a socket exception raised on another connection from this pool" : (DefaultConnectionPool.this.pastMaxLifeTime(connection) ? "it is past its maximum allowed life time" : (DefaultConnectionPool.this.pastMaxIdleTime(connection) ? "it is past its maximum allowed idle time" : "the pool has been closed")));
            return reason;
        }

        private ConnectionClosedEvent.Reason getReasonForClosing(UsageTrackingInternalConnection connection) {
            ConnectionClosedEvent.Reason reason = connection.isClosed() ? ConnectionClosedEvent.Reason.ERROR : (DefaultConnectionPool.this.fromPreviousGeneration(connection) ? ConnectionClosedEvent.Reason.STALE : (DefaultConnectionPool.this.pastMaxIdleTime(connection) ? ConnectionClosedEvent.Reason.IDLE : ConnectionClosedEvent.Reason.POOL_CLOSED));
            return reason;
        }

        @Override
        public boolean shouldPrune(UsageTrackingInternalConnection usageTrackingConnection) {
            return DefaultConnectionPool.this.shouldPrune(usageTrackingConnection);
        }
    }

    @ThreadSafe
    private static class AsyncWorkManager
    implements AutoCloseable {
        private volatile State state = State.NEW;
        private volatile BlockingQueue<Task> tasks = new LinkedBlockingQueue<Task>();
        private final Lock lock = new StampedLock().asWriteLock();
        @Nullable
        private ExecutorService worker;

        AsyncWorkManager(boolean prestart) {
            if (prestart) {
                Assertions.assertTrue(this.initUnlessClosed());
            }
        }

        void enqueue(Task task) {
            this.lock.lock();
            try {
                if (this.initUnlessClosed()) {
                    this.tasks.add(task);
                    return;
                }
            }
            finally {
                this.lock.unlock();
            }
            task.failAsClosed();
        }

        private boolean initUnlessClosed() {
            boolean result = true;
            if (this.state == State.NEW) {
                this.worker = Executors.newSingleThreadExecutor(new DaemonThreadFactory("AsyncGetter"));
                this.worker.submit(() -> this.runAndLogUncaught(this::workerRun));
                this.state = State.INITIALIZED;
            } else if (this.state == State.CLOSED) {
                result = false;
            }
            return result;
        }

        @Override
        public void close() {
            this.lock.lock();
            try {
                if (this.state != State.CLOSED) {
                    this.state = State.CLOSED;
                    if (this.worker != null) {
                        this.worker.shutdownNow();
                    }
                }
            }
            finally {
                this.lock.unlock();
            }
        }

        private void workerRun() {
            while (this.state != State.CLOSED) {
                try {
                    Task task = this.tasks.take();
                    if (task.timeout().expired()) {
                        task.failAsTimedOut();
                        continue;
                    }
                    task.execute();
                }
                catch (InterruptedException task) {
                }
                catch (Exception e) {
                    LOGGER.error(null, e);
                }
            }
            this.failAllTasksAfterClosing();
        }

        private void failAllTasksAfterClosing() {
            BlockingQueue<Task> localGets;
            this.lock.lock();
            try {
                Assertions.assertTrue(this.state == State.CLOSED);
                localGets = this.tasks;
                if (!this.tasks.isEmpty()) {
                    this.tasks = new LinkedBlockingQueue<Task>();
                }
            }
            finally {
                this.lock.unlock();
            }
            localGets.forEach(Task::failAsClosed);
            localGets.clear();
        }

        private void runAndLogUncaught(Runnable runnable) {
            try {
                runnable.run();
            }
            catch (Throwable t) {
                LOGGER.error("The pool is not going to work correctly from now on. You may want to recreate the MongoClient", t);
                throw t;
            }
        }

        private static enum State {
            NEW,
            INITIALIZED,
            CLOSED;

        }
    }

    @NotThreadSafe
    final class Task {
        private final Timeout timeout;
        private final Consumer<RuntimeException> action;
        private boolean completed;

        Task(Timeout timeout, Consumer<RuntimeException> action) {
            this.timeout = timeout;
            this.action = action;
        }

        void execute() {
            this.doComplete(() -> null);
        }

        void failAsClosed() {
            this.doComplete(DefaultConnectionPool.this.pool::poolClosedException);
        }

        void failAsTimedOut() {
            this.doComplete(() -> DefaultConnectionPool.this.createTimeoutException(this.timeout));
        }

        private void doComplete(Supplier<RuntimeException> failureSupplier) {
            Assertions.assertFalse(this.completed);
            this.completed = true;
            this.action.accept(failureSupplier.get());
        }

        Timeout timeout() {
            return this.timeout;
        }
    }

    private static final class MongoOpenConnectionInternalException
    extends RuntimeException {
        private static final long serialVersionUID = 1L;

        MongoOpenConnectionInternalException(@NonNull Throwable cause) {
            super(cause);
        }

        @Override
        @NonNull
        public Throwable getCause() {
            return Assertions.assertNotNull(super.getCause());
        }
    }

    @NotThreadSafe
    private static final class MutableReference<T> {
        @Nullable
        private T reference;

        private MutableReference() {
        }
    }

    private static enum OpenWithConcurrencyLimitMode {
        TRY_GET_AVAILABLE,
        TRY_HAND_OVER_OR_RELEASE;

    }
}

