/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.connector.mongodb;

import com.mongodb.ReadPreference;
import com.mongodb.client.ChangeStreamIterable;
import com.mongodb.client.MongoChangeStreamCursor;
import com.mongodb.client.MongoClient;
import com.mongodb.client.model.changestream.ChangeStreamDocument;
import com.mongodb.client.model.changestream.FullDocument;
import com.mongodb.client.model.changestream.FullDocumentBeforeChange;
import io.debezium.DebeziumException;
import io.debezium.connector.mongodb.ChangeStreamPipeline;
import io.debezium.connector.mongodb.ChangeStreamPipelineFactory;
import io.debezium.connector.mongodb.CollectionId;
import io.debezium.connector.mongodb.DisconnectEvent;
import io.debezium.connector.mongodb.MongoDbConnector;
import io.debezium.connector.mongodb.MongoDbConnectorConfig;
import io.debezium.connector.mongodb.MongoDbIncrementalSnapshotContext;
import io.debezium.connector.mongodb.MongoDbOffsetContext;
import io.debezium.connector.mongodb.MongoDbPartition;
import io.debezium.connector.mongodb.MongoDbTaskContext;
import io.debezium.connector.mongodb.MongoUtil;
import io.debezium.connector.mongodb.ReplicaSetOffsetContext;
import io.debezium.connector.mongodb.ReplicaSetPartition;
import io.debezium.connector.mongodb.ReplicaSets;
import io.debezium.connector.mongodb.SourceInfo;
import io.debezium.connector.mongodb.connection.ConnectionContext;
import io.debezium.connector.mongodb.connection.MongoDbConnection;
import io.debezium.connector.mongodb.connection.ReplicaSet;
import io.debezium.connector.mongodb.recordemitter.MongoDbChangeRecordEmitter;
import io.debezium.function.BlockingConsumer;
import io.debezium.pipeline.ConnectorEvent;
import io.debezium.pipeline.ErrorHandler;
import io.debezium.pipeline.EventDispatcher;
import io.debezium.pipeline.source.spi.ChangeEventSource;
import io.debezium.pipeline.source.spi.StreamingChangeEventSource;
import io.debezium.pipeline.spi.ChangeRecordEmitter;
import io.debezium.pipeline.spi.OffsetContext;
import io.debezium.pipeline.spi.Partition;
import io.debezium.pipeline.txmetadata.TransactionContext;
import io.debezium.spi.schema.DataCollectionId;
import io.debezium.util.Clock;
import io.debezium.util.Metronome;
import io.debezium.util.Threads;
import java.time.Duration;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.bson.BsonDocument;
import org.bson.BsonString;
import org.bson.BsonTimestamp;
import org.bson.BsonValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MongoDbStreamingChangeEventSource
implements StreamingChangeEventSource<MongoDbPartition, MongoDbOffsetContext> {
    private static final Logger LOGGER = LoggerFactory.getLogger(MongoDbStreamingChangeEventSource.class);
    private static final String AUTHORIZATION_FAILURE_MESSAGE = "Command failed with error 13";
    private static final String OPERATION_FIELD = "op";
    private static final String OBJECT_FIELD = "o";
    private static final String OPERATION_CONTROL = "c";
    private static final String TX_OPS = "applyOps";
    private final MongoDbConnectorConfig connectorConfig;
    private final EventDispatcher<MongoDbPartition, CollectionId> dispatcher;
    private final ErrorHandler errorHandler;
    private final Clock clock;
    private final ConnectionContext connectionContext;
    private final ReplicaSets replicaSets;
    private final MongoDbTaskContext taskContext;

    public MongoDbStreamingChangeEventSource(MongoDbConnectorConfig connectorConfig, MongoDbTaskContext taskContext, ReplicaSets replicaSets, EventDispatcher<MongoDbPartition, CollectionId> dispatcher, ErrorHandler errorHandler, Clock clock) {
        this.connectorConfig = connectorConfig;
        this.connectionContext = taskContext.getConnectionContext();
        this.dispatcher = dispatcher;
        this.errorHandler = errorHandler;
        this.clock = clock;
        this.replicaSets = replicaSets;
        this.taskContext = taskContext;
    }

    public void execute(ChangeEventSource.ChangeEventSourceContext context, MongoDbPartition partition, MongoDbOffsetContext offsetContext) throws InterruptedException {
        List<ReplicaSet> validReplicaSets = this.replicaSets.all();
        if (offsetContext == null) {
            offsetContext = this.initializeOffsets(this.connectorConfig, partition, this.replicaSets);
        }
        if (validReplicaSets.size() == 1) {
            this.streamChangesForReplicaSet(context, partition, validReplicaSets.get(0), offsetContext);
        } else if (validReplicaSets.size() > 1) {
            this.streamChangesForReplicaSets(context, partition, validReplicaSets, offsetContext);
        }
    }

    private void streamChangesForReplicaSet(ChangeEventSource.ChangeEventSourceContext context, MongoDbPartition partition, ReplicaSet replicaSet, MongoDbOffsetContext offsetContext) {
        try (MongoDbConnection mongo = this.establishConnection(partition, replicaSet, ReadPreference.secondaryPreferred());){
            AtomicReference<MongoDbConnection> mongoReference = new AtomicReference<MongoDbConnection>(mongo);
            mongo.execute("read from change stream on '" + replicaSet + "'", (BlockingConsumer<MongoClient>)((BlockingConsumer)client -> this.readChangeStream((MongoClient)client, (MongoDbConnection)mongoReference.get(), replicaSet, context, offsetContext)));
        }
        catch (Throwable t) {
            LOGGER.error("Streaming for replica set {} failed", (Object)replicaSet.replicaSetName(), (Object)t);
            this.errorHandler.setProducerThrowable(t);
        }
    }

    private void streamChangesForReplicaSets(ChangeEventSource.ChangeEventSourceContext context, MongoDbPartition partition, List<ReplicaSet> replicaSets, MongoDbOffsetContext offsetContext) {
        int threads = replicaSets.size();
        ExecutorService executor = Threads.newFixedThreadPool(MongoDbConnector.class, (String)this.taskContext.serverName(), (String)"replicator-streaming", (int)threads);
        CountDownLatch latch = new CountDownLatch(threads);
        LOGGER.info("Starting {} thread(s) to stream changes for replica sets: {}", (Object)threads, replicaSets);
        replicaSets.forEach(replicaSet -> executor.submit(() -> {
            try {
                this.streamChangesForReplicaSet(context, partition, (ReplicaSet)replicaSet, offsetContext);
            }
            finally {
                latch.countDown();
            }
        }));
        try {
            latch.await();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        executor.shutdown();
    }

    private MongoDbConnection establishConnection(MongoDbPartition partition, ReplicaSet replicaSet, ReadPreference preference) {
        return this.connectionContext.connect(replicaSet, preference, this.taskContext.filters(), (desc, error) -> {
            if (error.getMessage() != null && error.getMessage().startsWith(AUTHORIZATION_FAILURE_MESSAGE)) {
                throw new DebeziumException("Error while attempting to " + desc, error);
            }
            this.dispatcher.dispatchConnectorEvent((Partition)partition, (ConnectorEvent)new DisconnectEvent());
            LOGGER.error("Error while attempting to {}: {}", new Object[]{desc, error.getMessage(), error});
            throw new DebeziumException("Error while attempting to " + desc, error);
        });
    }

    private void readChangeStream(MongoClient client, MongoDbConnection mongo, ReplicaSet replicaSet, ChangeEventSource.ChangeEventSourceContext context, MongoDbOffsetContext offsetContext) {
        ReplicaSetPartition rsPartition = offsetContext.getReplicaSetPartition(replicaSet);
        ReplicaSetOffsetContext rsOffsetContext = offsetContext.getReplicaSetOffsetContext(replicaSet);
        BsonTimestamp oplogStart = rsOffsetContext.lastOffsetTimestamp();
        ReplicaSetChangeStreamsContext oplogContext = new ReplicaSetChangeStreamsContext(rsPartition, rsOffsetContext, mongo, replicaSet);
        LOGGER.info("Reading change stream for '{}' starting at {}", (Object)replicaSet, (Object)oplogStart);
        ChangeStreamPipeline pipeline = new ChangeStreamPipelineFactory(rsOffsetContext, this.taskContext.getConnectorConfig(), this.taskContext.filters().getConfig()).create();
        ChangeStreamIterable rsChangeStream = client.watch(pipeline.getStages(), BsonDocument.class);
        if (this.taskContext.getCaptureMode().isFullUpdate()) {
            rsChangeStream.fullDocument(FullDocument.UPDATE_LOOKUP);
        }
        if (this.taskContext.getCaptureMode().isIncludePreImage()) {
            rsChangeStream.fullDocumentBeforeChange(FullDocumentBeforeChange.WHEN_AVAILABLE);
        }
        if (rsOffsetContext.lastResumeToken() != null) {
            LOGGER.info("Resuming streaming from token '{}'", (Object)rsOffsetContext.lastResumeToken());
            BsonDocument doc = new BsonDocument();
            doc.put("_data", (BsonValue)new BsonString(rsOffsetContext.lastResumeToken()));
            rsChangeStream.resumeAfter(doc);
        } else if (oplogStart.getTime() > 0) {
            LOGGER.info("Resume token not available, starting streaming from time '{}'", (Object)oplogStart);
            rsChangeStream.startAtOperationTime(oplogStart);
        }
        if (this.connectorConfig.getCursorMaxAwaitTime() > 0) {
            rsChangeStream.maxAwaitTime((long)this.connectorConfig.getCursorMaxAwaitTime(), TimeUnit.MILLISECONDS);
        }
        try (MongoChangeStreamCursor cursor = rsChangeStream.cursor();){
            Metronome pause = Metronome.sleeper((Duration)Duration.ofMillis(500L), (Clock)this.clock);
            while (context.isRunning()) {
                ChangeStreamDocument event = (ChangeStreamDocument)cursor.tryNext();
                if (event != null) {
                    LOGGER.trace("Arrived Change Stream event: {}", (Object)event);
                    oplogContext.getOffset().changeStreamEvent((ChangeStreamDocument<BsonDocument>)event);
                    oplogContext.getOffset().getOffset();
                    CollectionId collectionId = new CollectionId(replicaSet.replicaSetName(), event.getNamespace().getDatabaseName(), event.getNamespace().getCollectionName());
                    try {
                        this.dispatcher.dispatchDataChangeEvent((Partition)oplogContext.getPartition(), (DataCollectionId)collectionId, (ChangeRecordEmitter)new MongoDbChangeRecordEmitter(oplogContext.getPartition(), (OffsetContext)oplogContext.getOffset(), this.clock, (ChangeStreamDocument<BsonDocument>)event));
                        continue;
                    }
                    catch (Exception e) {
                        this.errorHandler.setProducerThrowable((Throwable)e);
                        if (cursor != null) {
                            cursor.close();
                        }
                        return;
                    }
                }
                try {
                    if (cursor.getResumeToken() != null) {
                        oplogContext.getOffset().noEvent(cursor);
                        this.dispatcher.dispatchHeartbeatEvent((Partition)oplogContext.getPartition(), (OffsetContext)oplogContext.getOffset());
                    }
                }
                catch (InterruptedException e) {
                    LOGGER.info("Replicator thread is interrupted");
                    Thread.currentThread().interrupt();
                    if (cursor != null) {
                        cursor.close();
                    }
                    return;
                }
                try {
                    pause.pause();
                }
                catch (InterruptedException e) {
                    break;
                }
            }
        }
    }

    protected MongoDbOffsetContext initializeOffsets(MongoDbConnectorConfig connectorConfig, MongoDbPartition partition, ReplicaSets replicaSets) throws InterruptedException {
        LinkedHashMap<ReplicaSet, BsonDocument> positions = new LinkedHashMap<ReplicaSet, BsonDocument>();
        for (ReplicaSet replicaSet : replicaSets.all()) {
            LOGGER.info("Determine Snapshot Offset for replica-set {}", (Object)replicaSet.replicaSetName());
            MongoDbConnection mongo = this.establishConnection(partition, replicaSet, ReadPreference.primaryPreferred());
            try {
                mongo.execute("get oplog position", (BlockingConsumer<MongoClient>)((BlockingConsumer)client -> positions.put(replicaSet, MongoUtil.getOplogEntry(client, -1, LOGGER))));
            }
            finally {
                if (mongo == null) continue;
                mongo.close();
            }
        }
        return new MongoDbOffsetContext(new SourceInfo(connectorConfig), new TransactionContext(), new MongoDbIncrementalSnapshotContext<CollectionId>(false), positions);
    }

    private static class ReplicaSetChangeStreamsContext {
        private final ReplicaSetPartition partition;
        private final ReplicaSetOffsetContext offset;
        private final MongoDbConnection mongo;
        private final ReplicaSet replicaSet;

        ReplicaSetChangeStreamsContext(ReplicaSetPartition partition, ReplicaSetOffsetContext offsetContext, MongoDbConnection mongo, ReplicaSet replicaSet) {
            this.partition = partition;
            this.offset = offsetContext;
            this.mongo = mongo;
            this.replicaSet = replicaSet;
        }

        ReplicaSetPartition getPartition() {
            return this.partition;
        }

        ReplicaSetOffsetContext getOffset() {
            return this.offset;
        }

        MongoDbConnection getMongo() {
            return this.mongo;
        }

        String getReplicaSetName() {
            return this.replicaSet.replicaSetName();
        }
    }
}

