/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.connector.mongodb;

import com.mongodb.ConnectionString;
import com.mongodb.MongoCredential;
import com.mongodb.ReadPreference;
import com.mongodb.ServerAddress;
import com.mongodb.client.MongoClient;
import com.mongodb.connection.ClusterDescription;
import com.mongodb.connection.ClusterType;
import io.debezium.config.Configuration;
import io.debezium.connector.mongodb.CollectionId;
import io.debezium.connector.mongodb.Filters;
import io.debezium.connector.mongodb.MongoClients;
import io.debezium.connector.mongodb.MongoDbConnectorConfig;
import io.debezium.connector.mongodb.MongoUtil;
import io.debezium.connector.mongodb.ReplicaSet;
import io.debezium.function.BlockingConsumer;
import io.debezium.util.Clock;
import io.debezium.util.DelayStrategy;
import io.debezium.util.Metronome;
import io.debezium.util.Strings;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import org.apache.kafka.connect.errors.ConnectException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConnectionContext
implements AutoCloseable {
    private static final Duration PAUSE_AFTER_ERROR = Duration.ofMillis(500L);
    private static final Logger LOGGER = LoggerFactory.getLogger(ConnectionContext.class);
    protected final Configuration config;
    protected final MongoClients pool;
    protected final DelayStrategy backoffStrategy;
    protected final boolean useHostsAsSeeds;

    public ConnectionContext(Configuration config) {
        this.config = config;
        this.useHostsAsSeeds = config.getBoolean(MongoDbConnectorConfig.AUTO_DISCOVER_MEMBERS);
        String username = config.getString(MongoDbConnectorConfig.USER);
        String password = config.getString(MongoDbConnectorConfig.PASSWORD);
        String adminDbName = config.getString(MongoDbConnectorConfig.AUTH_SOURCE);
        boolean useSSL = config.getBoolean(MongoDbConnectorConfig.SSL_ENABLED);
        boolean sslAllowInvalidHostnames = config.getBoolean(MongoDbConnectorConfig.SSL_ALLOW_INVALID_HOSTNAMES);
        int connectTimeoutMs = config.getInteger(MongoDbConnectorConfig.CONNECT_TIMEOUT_MS);
        int heartbeatFrequencyMs = config.getInteger(MongoDbConnectorConfig.HEARTBEAT_FREQUENCY_MS);
        int socketTimeoutMs = config.getInteger(MongoDbConnectorConfig.SOCKET_TIMEOUT_MS);
        int serverSelectionTimeoutMs = config.getInteger(MongoDbConnectorConfig.SERVER_SELECTION_TIMEOUT_MS);
        MongoClients.Builder clientBuilder = MongoClients.create();
        clientBuilder.settings().applyToSocketSettings(builder -> builder.connectTimeout(connectTimeoutMs, TimeUnit.MILLISECONDS).readTimeout(socketTimeoutMs, TimeUnit.MILLISECONDS)).applyToClusterSettings(builder -> builder.serverSelectionTimeout((long)serverSelectionTimeoutMs, TimeUnit.MILLISECONDS)).applyToServerSettings(builder -> builder.heartbeatFrequency((long)heartbeatFrequencyMs, TimeUnit.MILLISECONDS));
        Optional<ConnectionString> cs = this.connectionString();
        cs.map(ConnectionString::getCredential).ifPresent(clientBuilder::withCredential);
        if (username != null || password != null) {
            clientBuilder.withCredential(MongoCredential.createCredential((String)username, (String)adminDbName, (char[])password.toCharArray()));
        }
        if (useSSL) {
            clientBuilder.settings().applyToSslSettings(builder -> builder.enabled(true).invalidHostNameAllowed(sslAllowInvalidHostnames));
        }
        clientBuilder.settings().applyToSocketSettings(builder -> builder.connectTimeout(connectTimeoutMs, TimeUnit.MILLISECONDS).readTimeout(socketTimeoutMs, TimeUnit.MILLISECONDS)).applyToClusterSettings(builder -> builder.serverSelectionTimeout((long)serverSelectionTimeoutMs, TimeUnit.MILLISECONDS));
        this.pool = clientBuilder.build();
        int initialDelayInMs = config.getInteger(MongoDbConnectorConfig.CONNECT_BACKOFF_INITIAL_DELAY_MS);
        long maxDelayInMs = config.getLong(MongoDbConnectorConfig.CONNECT_BACKOFF_MAX_DELAY_MS);
        this.backoffStrategy = DelayStrategy.exponential((Duration)Duration.ofMillis(initialDelayInMs), (Duration)Duration.ofMillis(maxDelayInMs));
    }

    public void shutdown() {
        try {
            this.logger().info("Closing all connections to {}", (Object)this.maskedConnectionSeed());
            this.pool.clear();
        }
        catch (Throwable e) {
            this.logger().error("Unexpected error shutting down the MongoDB clients", e);
        }
    }

    @Override
    public final void close() {
        this.shutdown();
    }

    protected Logger logger() {
        return LOGGER;
    }

    public MongoClients clients() {
        return this.pool;
    }

    public boolean performSnapshotEvenIfNotNeeded() {
        return false;
    }

    public MongoClient clientFor(ReplicaSet replicaSet) {
        return this.clientFor(replicaSet.addresses());
    }

    public MongoClient clientFor(String seedAddresses) {
        List<ServerAddress> addresses = MongoUtil.parseAddresses(seedAddresses);
        return this.clientFor(addresses);
    }

    public MongoClient clientForSeedConnection() {
        return this.connectionString().map(this::clientFor).orElseGet(() -> this.clientFor(this.hosts()));
    }

    public MongoClient clientFor(ConnectionString connectionString) {
        return this.pool.clientForMembers(connectionString);
    }

    public MongoClient clientFor(List<ServerAddress> addresses) {
        if (this.useHostsAsSeeds || addresses.isEmpty()) {
            return this.pool.clientForMembers(addresses);
        }
        return this.pool.clientFor(addresses.get(0));
    }

    public String hosts() {
        return this.config.getString(MongoDbConnectorConfig.HOSTS);
    }

    public String connectionSeed() {
        return this.connectionString().map(ConnectionString::toString).orElse(this.config.getString(MongoDbConnectorConfig.HOSTS));
    }

    public String maskedConnectionSeed() {
        String connectionSeed = this.connectionSeed();
        return this.connectionString().map(ConnectionString::getCredential).map(creds -> Strings.mask((String)connectionSeed, (String)creds.getUserName(), (String[])new String[]{creds.getSource(), creds.getPassword() != null ? String.valueOf(creds.getPassword()) : null})).orElse(connectionSeed);
    }

    public Optional<ConnectionString> connectionString() {
        String raw = this.config.getString(MongoDbConnectorConfig.CONNECTION_STRING);
        return Optional.ofNullable(raw).map(ConnectionString::new);
    }

    public Duration pollInterval() {
        return Duration.ofMillis(this.config.getLong(MongoDbConnectorConfig.MONGODB_POLL_INTERVAL_MS));
    }

    public MongoPrimary primaryFor(ReplicaSet replicaSet, Filters filters, BiConsumer<String, Throwable> errorHandler) {
        return new MongoPrimary(this, replicaSet, filters, errorHandler);
    }

    public MongoPreferredNode preferredFor(ReplicaSet replicaSet, ReadPreference preference, Filters filters, BiConsumer<String, Throwable> errorHandler) {
        return new MongoPreferredNode(this, replicaSet, preference, filters, errorHandler);
    }

    protected Supplier<MongoClient> preferredClientFor(ReplicaSet replicaSet, ReadPreference preference) {
        return this.preferredClientFor(replicaSet, preference, (attempts, remaining, error) -> {
            if (error == null) {
                this.logger().info("Unable to connect to {} node of '{}' after attempt #{} ({} remaining)", new Object[]{preference.getName(), replicaSet, attempts, remaining});
            } else {
                this.logger().error("Error while attempting to connect to {} node of '{}' after attempt #{} ({} remaining): {}", new Object[]{preference.getName(), replicaSet, attempts, remaining, error.getMessage(), error});
            }
        });
    }

    protected Supplier<MongoClient> preferredClientFor(ReplicaSet replicaSet, ReadPreference preference, PreferredConnectFailed handler) {
        Supplier<MongoClient> factory = () -> this.clientForPreferred(replicaSet, preference);
        int maxAttempts = this.config.getInteger(MongoDbConnectorConfig.MAX_FAILED_CONNECTIONS);
        return () -> {
            int attempts = 0;
            MongoClient client = null;
            while (client == null) {
                ++attempts;
                try {
                    client = (MongoClient)factory.get();
                    if (client != null) {
                        break;
                    }
                }
                catch (Throwable t) {
                    handler.failed(attempts, maxAttempts - attempts, t);
                }
                if (attempts > maxAttempts) {
                    throw new ConnectException("Unable to connect to " + preference.getName() + " node of '" + replicaSet + "' after " + attempts + " failed attempts");
                }
                handler.failed(attempts, maxAttempts - attempts, null);
                this.backoffStrategy.sleepWhen(true);
            }
            return client;
        };
    }

    protected MongoClient clientForPreferred(ReplicaSet replicaSet, ReadPreference preference) {
        MongoClient replicaSetClient = this.clientFor(replicaSet);
        ClusterDescription clusterDescription = MongoUtil.clusterDescription(replicaSetClient);
        if (clusterDescription.getType() == ClusterType.UNKNOWN) {
            if (!this.useHostsAsSeeds) {
                return replicaSetClient;
            }
            throw new ConnectException("The MongoDB server(s) at '" + replicaSet + "' is not a valid replica set and cannot be used");
        }
        ServerAddress preferredAddress = MongoUtil.getPreferredAddress(replicaSetClient, preference);
        if (preferredAddress != null) {
            return this.pool.clientFor(preferredAddress);
        }
        return null;
    }

    public static class MongoPrimary
    extends MongoPreferredNode {
        protected MongoPrimary(ConnectionContext context, ReplicaSet replicaSet, Filters filters, BiConsumer<String, Throwable> errorHandler) {
            super(context, replicaSet, ReadPreference.primary(), filters, errorHandler);
        }
    }

    public static class MongoPreferredNode {
        private final ReplicaSet replicaSet;
        private final Supplier<MongoClient> connectionSupplier;
        private final Filters filters;
        private final BiConsumer<String, Throwable> errorHandler;
        private final AtomicBoolean running = new AtomicBoolean(true);
        private final ReadPreference preference;

        protected MongoPreferredNode(ConnectionContext context, ReplicaSet replicaSet, ReadPreference preference, Filters filters, BiConsumer<String, Throwable> errorHandler) {
            this.replicaSet = replicaSet;
            this.preference = preference;
            this.connectionSupplier = context.preferredClientFor(replicaSet, preference);
            this.filters = filters;
            this.errorHandler = errorHandler;
        }

        public ReplicaSet replicaSet() {
            return this.replicaSet;
        }

        public ReadPreference getPreference() {
            return this.preference;
        }

        public ServerAddress address() {
            return this.execute("get replica set " + this.preference.getName(), (MongoClient client) -> MongoUtil.getPreferredAddress(client, this.preference));
        }

        public void execute(String desc, Consumer<MongoClient> operation) {
            Metronome errorMetronome = Metronome.sleeper((Duration)PAUSE_AFTER_ERROR, (Clock)Clock.SYSTEM);
            while (true) {
                MongoClient client = this.connectionSupplier.get();
                try {
                    operation.accept(client);
                    return;
                }
                catch (Throwable t) {
                    this.errorHandler.accept(desc, t);
                    if (!this.isRunning()) {
                        throw new ConnectException("Operation failed and MongoDB " + this.preference.getName() + " termination requested", t);
                    }
                    try {
                        errorMetronome.pause();
                    }
                    catch (InterruptedException interruptedException) {
                    }
                    continue;
                }
                break;
            }
        }

        public <T> T execute(String desc, Function<MongoClient, T> operation) {
            Metronome errorMetronome = Metronome.sleeper((Duration)PAUSE_AFTER_ERROR, (Clock)Clock.SYSTEM);
            while (true) {
                MongoClient client = this.connectionSupplier.get();
                try {
                    return operation.apply(client);
                }
                catch (Throwable t) {
                    this.errorHandler.accept(desc, t);
                    if (!this.isRunning()) {
                        throw new ConnectException("Operation failed and MongoDB " + this.preference.getName() + " termination requested", t);
                    }
                    try {
                        errorMetronome.pause();
                    }
                    catch (InterruptedException interruptedException) {
                    }
                    continue;
                }
                break;
            }
        }

        public void executeBlocking(String desc, BlockingConsumer<MongoClient> operation) throws InterruptedException {
            Metronome errorMetronome = Metronome.sleeper((Duration)PAUSE_AFTER_ERROR, (Clock)Clock.SYSTEM);
            while (true) {
                MongoClient client = this.connectionSupplier.get();
                try {
                    operation.accept((Object)client);
                    return;
                }
                catch (InterruptedException e) {
                    throw e;
                }
                catch (Throwable t) {
                    this.errorHandler.accept(desc, t);
                    if (!this.isRunning()) {
                        throw new ConnectException("Operation failed and MongoDB " + this.preference.getName() + " termination requested", t);
                    }
                    errorMetronome.pause();
                    continue;
                }
                break;
            }
        }

        public Set<String> databaseNames() {
            return this.execute("get database names", (MongoClient client) -> {
                HashSet databaseNames = new HashSet();
                MongoUtil.forEachDatabaseName(client, dbName -> {
                    if (this.filters.databaseFilter().test((String)dbName)) {
                        databaseNames.add(dbName);
                    }
                });
                return databaseNames;
            });
        }

        public List<CollectionId> collections() {
            String replicaSetName = this.replicaSet.replicaSetName();
            return this.execute("get collections in databases", (MongoClient client) -> {
                ArrayList collections = new ArrayList();
                Set<String> databaseNames = this.databaseNames();
                for (String dbName : databaseNames) {
                    MongoUtil.forEachCollectionNameInDatabase(client, dbName, collectionName -> {
                        CollectionId collectionId = new CollectionId(replicaSetName, dbName, (String)collectionName);
                        if (this.filters.collectionFilter().test(collectionId)) {
                            collections.add(collectionId);
                        }
                    });
                }
                return collections;
            });
        }

        private boolean isRunning() {
            return this.running.get();
        }

        public void stop() {
            this.running.set(false);
        }
    }

    @FunctionalInterface
    public static interface PreferredConnectFailed {
        public void failed(int var1, int var2, Throwable var3);
    }
}

