/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.connector.mongodb;

import com.mongodb.ServerAddress;
import com.mongodb.client.ChangeStreamIterable;
import com.mongodb.client.MongoClient;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoCursor;
import com.mongodb.client.model.Aggregates;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.changestream.ChangeStreamDocument;
import com.mongodb.client.model.changestream.FullDocument;
import io.debezium.connector.mongodb.CollectionId;
import io.debezium.connector.mongodb.ConnectionContext;
import io.debezium.connector.mongodb.DisconnectEvent;
import io.debezium.connector.mongodb.MongoDbChangeRecordEmitter;
import io.debezium.connector.mongodb.MongoDbConnector;
import io.debezium.connector.mongodb.MongoDbConnectorConfig;
import io.debezium.connector.mongodb.MongoDbIncrementalSnapshotContext;
import io.debezium.connector.mongodb.MongoDbOffsetContext;
import io.debezium.connector.mongodb.MongoDbPartition;
import io.debezium.connector.mongodb.MongoDbTaskContext;
import io.debezium.connector.mongodb.MongoUtil;
import io.debezium.connector.mongodb.ReplicaSet;
import io.debezium.connector.mongodb.ReplicaSetOffsetContext;
import io.debezium.connector.mongodb.ReplicaSetPartition;
import io.debezium.connector.mongodb.ReplicaSets;
import io.debezium.connector.mongodb.SourceInfo;
import io.debezium.data.Envelope;
import io.debezium.pipeline.ConnectorEvent;
import io.debezium.pipeline.ErrorHandler;
import io.debezium.pipeline.EventDispatcher;
import io.debezium.pipeline.source.spi.ChangeEventSource;
import io.debezium.pipeline.source.spi.StreamingChangeEventSource;
import io.debezium.pipeline.spi.ChangeRecordEmitter;
import io.debezium.pipeline.spi.OffsetContext;
import io.debezium.pipeline.spi.Partition;
import io.debezium.pipeline.txmetadata.TransactionContext;
import io.debezium.spi.schema.DataCollectionId;
import io.debezium.util.Clock;
import io.debezium.util.Metronome;
import io.debezium.util.Threads;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.EnumSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.kafka.connect.errors.ConnectException;
import org.bson.BsonDocument;
import org.bson.BsonString;
import org.bson.BsonTimestamp;
import org.bson.BsonValue;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MongoDbStreamingChangeEventSource
implements StreamingChangeEventSource<MongoDbPartition, MongoDbOffsetContext> {
    private static final Logger LOGGER = LoggerFactory.getLogger(MongoDbStreamingChangeEventSource.class);
    private static final String AUTHORIZATION_FAILURE_MESSAGE = "Command failed with error 13";
    private static final String OPERATION_FIELD = "op";
    private static final String OBJECT_FIELD = "o";
    private static final String OPERATION_CONTROL = "c";
    private static final String TX_OPS = "applyOps";
    private final MongoDbConnectorConfig connectorConfig;
    private final EventDispatcher<MongoDbPartition, CollectionId> dispatcher;
    private final ErrorHandler errorHandler;
    private final Clock clock;
    private final ConnectionContext connectionContext;
    private final ReplicaSets replicaSets;
    private final MongoDbTaskContext taskContext;

    public MongoDbStreamingChangeEventSource(MongoDbConnectorConfig connectorConfig, MongoDbTaskContext taskContext, ReplicaSets replicaSets, EventDispatcher<MongoDbPartition, CollectionId> dispatcher, ErrorHandler errorHandler, Clock clock) {
        this.connectorConfig = connectorConfig;
        this.connectionContext = taskContext.getConnectionContext();
        this.dispatcher = dispatcher;
        this.errorHandler = errorHandler;
        this.clock = clock;
        this.replicaSets = replicaSets;
        this.taskContext = taskContext;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void execute(ChangeEventSource.ChangeEventSourceContext context, MongoDbPartition partition, MongoDbOffsetContext offsetContext) throws InterruptedException {
        List<ReplicaSet> validReplicaSets = this.replicaSets.validReplicaSets();
        if (offsetContext == null) {
            offsetContext = this.initializeOffsets(this.connectorConfig, partition, this.replicaSets);
        }
        try {
            if (validReplicaSets.size() == 1) {
                this.streamChangesForReplicaSet(context, partition, validReplicaSets.get(0), offsetContext);
            } else if (validReplicaSets.size() > 1) {
                this.streamChangesForReplicaSets(context, partition, validReplicaSets, offsetContext);
            }
        }
        finally {
            this.taskContext.getConnectionContext().shutdown();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void streamChangesForReplicaSet(ChangeEventSource.ChangeEventSourceContext context, MongoDbPartition partition, ReplicaSet replicaSet, MongoDbOffsetContext offsetContext) {
        ConnectionContext.MongoPrimary primaryClient = null;
        try {
            primaryClient = this.establishConnectionToPrimary(partition, replicaSet);
            if (primaryClient != null) {
                AtomicReference<ConnectionContext.MongoPrimary> primaryReference = new AtomicReference<ConnectionContext.MongoPrimary>(primaryClient);
                primaryClient.execute("read from change stream on '" + replicaSet + "'", primary -> this.readChangeStream((MongoClient)primary, (ConnectionContext.MongoPrimary)primaryReference.get(), replicaSet, context, offsetContext));
            }
        }
        catch (Throwable t) {
            LOGGER.error("Streaming for replica set {} failed", (Object)replicaSet.replicaSetName(), (Object)t);
            this.errorHandler.setProducerThrowable(t);
        }
        finally {
            if (primaryClient != null) {
                primaryClient.stop();
            }
        }
    }

    private void streamChangesForReplicaSets(ChangeEventSource.ChangeEventSourceContext context, MongoDbPartition partition, List<ReplicaSet> replicaSets, MongoDbOffsetContext offsetContext) {
        int threads = replicaSets.size();
        ExecutorService executor = Threads.newFixedThreadPool(MongoDbConnector.class, (String)this.taskContext.serverName(), (String)"replicator-streaming", (int)threads);
        CountDownLatch latch = new CountDownLatch(threads);
        LOGGER.info("Starting {} thread(s) to stream changes for replica sets: {}", (Object)threads, replicaSets);
        replicaSets.forEach(replicaSet -> executor.submit(() -> {
            try {
                this.streamChangesForReplicaSet(context, partition, (ReplicaSet)replicaSet, offsetContext);
            }
            finally {
                latch.countDown();
            }
        }));
        try {
            latch.await();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
        executor.shutdown();
    }

    private ConnectionContext.MongoPrimary establishConnectionToPrimary(MongoDbPartition partition, ReplicaSet replicaSet) {
        return this.connectionContext.primaryFor(replicaSet, this.taskContext.filters(), (desc, error) -> {
            if (error.getMessage() != null && error.getMessage().startsWith(AUTHORIZATION_FAILURE_MESSAGE)) {
                throw new ConnectException("Error while attempting to " + desc, error);
            }
            this.dispatcher.dispatchConnectorEvent((Partition)partition, (ConnectorEvent)new DisconnectEvent());
            LOGGER.error("Error while attempting to {}: {}", new Object[]{desc, error.getMessage(), error});
            throw new ConnectException("Error while attempting to " + desc, error);
        });
    }

    private List<String> getChangeStreamSkippedOperationsFilter() {
        EnumSet skippedOperations = this.taskContext.getConnectorConfig().getSkippedOperations();
        ArrayList<String> includedOperations = new ArrayList<String>();
        if (!skippedOperations.contains(Envelope.Operation.CREATE)) {
            includedOperations.add("insert");
        }
        if (!skippedOperations.contains(Envelope.Operation.UPDATE)) {
            includedOperations.add("update");
            includedOperations.add("replace");
        }
        if (!skippedOperations.contains(Envelope.Operation.DELETE)) {
            includedOperations.add("delete");
        }
        return includedOperations;
    }

    private void readChangeStream(MongoClient primary, ConnectionContext.MongoPrimary primaryClient, ReplicaSet replicaSet, ChangeEventSource.ChangeEventSourceContext context, MongoDbOffsetContext offsetContext) {
        ReplicaSetPartition rsPartition = offsetContext.getReplicaSetPartition(replicaSet);
        ReplicaSetOffsetContext rsOffsetContext = offsetContext.getReplicaSetOffsetContext(replicaSet);
        BsonTimestamp oplogStart = rsOffsetContext.lastOffsetTimestamp();
        ReplicaSetChangeStreamsContext oplogContext = new ReplicaSetChangeStreamsContext(rsPartition, rsOffsetContext, primaryClient, replicaSet);
        ServerAddress primaryAddress = MongoUtil.getPrimaryAddress(primary);
        LOGGER.info("Reading change stream for '{}' primary {} starting at {}", new Object[]{replicaSet, primaryAddress, oplogStart});
        Bson filters = Filters.in((String)"operationType", this.getChangeStreamSkippedOperationsFilter());
        if (rsOffsetContext.lastResumeToken() == null) {
            filters = Filters.and((Bson[])new Bson[]{filters, Filters.ne((String)"clusterTime", (Object)oplogStart)});
        }
        ChangeStreamIterable rsChangeStream = primary.watch(Arrays.asList(Aggregates.match((Bson)filters)), BsonDocument.class);
        if (this.taskContext.getCaptureMode().isFullUpdate()) {
            rsChangeStream.fullDocument(FullDocument.UPDATE_LOOKUP);
        }
        if (rsOffsetContext.lastResumeToken() != null) {
            LOGGER.info("Resuming streaming from token '{}'", (Object)rsOffsetContext.lastResumeToken());
            BsonDocument doc = new BsonDocument();
            doc.put("_data", (BsonValue)new BsonString(rsOffsetContext.lastResumeToken()));
            rsChangeStream.resumeAfter(doc);
        } else {
            LOGGER.info("Resume token not available, starting streaming from time '{}'", (Object)oplogStart);
            rsChangeStream.startAtOperationTime(oplogStart);
        }
        if (this.connectorConfig.getCursorMaxAwaitTime() > 0) {
            rsChangeStream.maxAwaitTime((long)this.connectorConfig.getCursorMaxAwaitTime(), TimeUnit.MILLISECONDS);
        }
        try (MongoCursor cursor = rsChangeStream.iterator();){
            Metronome pause = Metronome.sleeper((Duration)Duration.ofMillis(500L), (Clock)this.clock);
            while (context.isRunning()) {
                ChangeStreamDocument event = (ChangeStreamDocument)cursor.tryNext();
                if (event != null) {
                    LOGGER.trace("Arrived Change Stream event: {}", (Object)event);
                    if (!this.taskContext.filters().databaseFilter().test(event.getDatabaseName())) {
                        LOGGER.debug("Skipping the event for database '{}' based on database include/exclude list", (Object)event.getDatabaseName());
                    } else {
                        oplogContext.getOffset().changeStreamEvent((ChangeStreamDocument<BsonDocument>)event);
                        oplogContext.getOffset().getOffset();
                        CollectionId collectionId = new CollectionId(replicaSet.replicaSetName(), event.getNamespace().getDatabaseName(), event.getNamespace().getCollectionName());
                        if (this.taskContext.filters().collectionFilter().test(collectionId)) {
                            try {
                                this.dispatcher.dispatchDataChangeEvent((Partition)oplogContext.getPartition(), (DataCollectionId)collectionId, (ChangeRecordEmitter)new MongoDbChangeRecordEmitter(oplogContext.getPartition(), (OffsetContext)oplogContext.getOffset(), this.clock, (ChangeStreamDocument<BsonDocument>)event));
                            }
                            catch (Exception e) {
                                this.errorHandler.setProducerThrowable((Throwable)e);
                                if (cursor != null) {
                                    cursor.close();
                                }
                                return;
                            }
                        }
                    }
                    try {
                        this.dispatcher.dispatchHeartbeatEvent((Partition)oplogContext.getPartition(), (OffsetContext)oplogContext.getOffset());
                        continue;
                    }
                    catch (InterruptedException e) {
                        LOGGER.info("Replicator thread is interrupted");
                        Thread.currentThread().interrupt();
                        if (cursor != null) {
                            cursor.close();
                        }
                        return;
                    }
                }
                try {
                    pause.pause();
                }
                catch (InterruptedException e) {
                    break;
                }
            }
        }
    }

    protected MongoDbOffsetContext initializeOffsets(MongoDbConnectorConfig connectorConfig, MongoDbPartition partition, ReplicaSets replicaSets) {
        LinkedHashMap<ReplicaSet, BsonDocument> positions = new LinkedHashMap<ReplicaSet, BsonDocument>();
        replicaSets.onEachReplicaSet(replicaSet -> {
            LOGGER.info("Determine Snapshot Offset for replica-set {}", (Object)replicaSet.replicaSetName());
            ConnectionContext.MongoPrimary primaryClient = this.establishConnectionToPrimary(partition, (ReplicaSet)replicaSet);
            if (primaryClient != null) {
                try {
                    primaryClient.execute("get oplog position", primary -> {
                        MongoCollection oplog = primary.getDatabase("local").getCollection("oplog.rs", BsonDocument.class);
                        BsonDocument last = (BsonDocument)oplog.find().sort((Bson)new Document("$natural", (Object)-1)).limit(1).first();
                        positions.put((ReplicaSet)replicaSet, last);
                    });
                }
                finally {
                    LOGGER.info("Stopping primary client");
                    primaryClient.stop();
                }
            }
        });
        return new MongoDbOffsetContext(new SourceInfo(connectorConfig), new TransactionContext(), new MongoDbIncrementalSnapshotContext<CollectionId>(false), positions);
    }

    private class ReplicaSetChangeStreamsContext {
        private final ReplicaSetPartition partition;
        private final ReplicaSetOffsetContext offset;
        private final ConnectionContext.MongoPrimary primary;
        private final ReplicaSet replicaSet;

        ReplicaSetChangeStreamsContext(ReplicaSetPartition partition, ReplicaSetOffsetContext offsetContext, ConnectionContext.MongoPrimary primary, ReplicaSet replicaSet) {
            this.partition = partition;
            this.offset = offsetContext;
            this.primary = primary;
            this.replicaSet = replicaSet;
        }

        ReplicaSetPartition getPartition() {
            return this.partition;
        }

        ReplicaSetOffsetContext getOffset() {
            return this.offset;
        }

        ConnectionContext.MongoPrimary getPrimary() {
            return this.primary;
        }

        String getReplicaSetName() {
            return this.replicaSet.replicaSetName();
        }
    }
}

