/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.connector.mongodb;

import com.mongodb.MongoClient;
import com.mongodb.MongoCredential;
import com.mongodb.ReplicaSetStatus;
import com.mongodb.ServerAddress;
import io.debezium.config.Configuration;
import io.debezium.connector.mongodb.CollectionId;
import io.debezium.connector.mongodb.Filters;
import io.debezium.connector.mongodb.MongoClients;
import io.debezium.connector.mongodb.MongoDbConnectorConfig;
import io.debezium.connector.mongodb.MongoUtil;
import io.debezium.connector.mongodb.ReplicaSet;
import io.debezium.connector.mongodb.ReplicaSets;
import io.debezium.function.BlockingConsumer;
import io.debezium.util.Clock;
import io.debezium.util.DelayStrategy;
import io.debezium.util.Metronome;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;
import org.apache.kafka.connect.errors.ConnectException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConnectionContext
implements AutoCloseable {
    private static final Duration PAUSE_AFTER_ERROR = Duration.ofMillis(500L);
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    protected final Configuration config;
    protected final MongoClients pool;
    protected final ReplicaSets replicaSets;
    protected final DelayStrategy primaryBackoffStrategy;
    protected final boolean useHostsAsSeeds;

    public ConnectionContext(Configuration config) {
        this.config = config;
        this.useHostsAsSeeds = config.getBoolean(MongoDbConnectorConfig.AUTO_DISCOVER_MEMBERS);
        String username = config.getString(MongoDbConnectorConfig.USER);
        String password = config.getString(MongoDbConnectorConfig.PASSWORD);
        String adminDbName = config.getString(MongoDbConnectorConfig.AUTH_SOURCE);
        boolean useSSL = config.getBoolean(MongoDbConnectorConfig.SSL_ENABLED);
        boolean sslAllowInvalidHostnames = config.getBoolean(MongoDbConnectorConfig.SSL_ALLOW_INVALID_HOSTNAMES);
        int connectTimeoutMs = config.getInteger(MongoDbConnectorConfig.CONNECT_TIMEOUT_MS);
        int socketTimeoutMs = config.getInteger(MongoDbConnectorConfig.SOCKET_TIMEOUT_MS);
        int serverSelectionTimeoutMs = config.getInteger(MongoDbConnectorConfig.SERVER_SELECTION_TIMEOUT_MS);
        MongoClients.Builder clientBuilder = MongoClients.create();
        if (username != null || password != null) {
            clientBuilder.withCredential(MongoCredential.createCredential((String)username, (String)adminDbName, (char[])password.toCharArray()));
        }
        if (useSSL) {
            clientBuilder.options().sslEnabled(true).sslInvalidHostNameAllowed(sslAllowInvalidHostnames);
        }
        clientBuilder.options().serverSelectionTimeout(serverSelectionTimeoutMs).socketTimeout(socketTimeoutMs).connectTimeout(connectTimeoutMs);
        this.pool = clientBuilder.build();
        this.replicaSets = ReplicaSets.parse(this.hosts());
        int initialDelayInMs = config.getInteger(MongoDbConnectorConfig.CONNECT_BACKOFF_INITIAL_DELAY_MS);
        long maxDelayInMs = config.getLong(MongoDbConnectorConfig.CONNECT_BACKOFF_MAX_DELAY_MS);
        this.primaryBackoffStrategy = DelayStrategy.exponential((long)initialDelayInMs, (long)maxDelayInMs);
    }

    public void shutdown() {
        try {
            this.logger().info("Closing all connections to {}", (Object)this.replicaSets);
            this.pool.clear();
        }
        catch (Throwable e) {
            this.logger().error("Unexpected error shutting down the MongoDB clients", e);
        }
    }

    @Override
    public final void close() {
        this.shutdown();
    }

    protected Logger logger() {
        return this.logger;
    }

    public MongoClients clients() {
        return this.pool;
    }

    public ReplicaSets replicaSets() {
        return this.replicaSets;
    }

    public boolean performSnapshotEvenIfNotNeeded() {
        return false;
    }

    public MongoClient clientForReplicaSet(ReplicaSet replicaSet) {
        return this.clientFor(replicaSet.addresses());
    }

    public MongoClient clientFor(String seedAddresses) {
        List<ServerAddress> addresses = MongoUtil.parseAddresses(seedAddresses);
        return this.clientFor(addresses);
    }

    public MongoClient clientFor(List<ServerAddress> addresses) {
        if (this.useHostsAsSeeds || addresses.isEmpty()) {
            return this.pool.clientForMembers(addresses);
        }
        return this.pool.clientFor(addresses.get(0));
    }

    public String hosts() {
        return this.config.getString(MongoDbConnectorConfig.HOSTS);
    }

    public Duration pollInterval() {
        if (this.config.hasKey(MongoDbConnectorConfig.MONGODB_POLL_INTERVAL_MS.name())) {
            return Duration.ofMillis(this.config.getLong(MongoDbConnectorConfig.MONGODB_POLL_INTERVAL_MS));
        }
        if (this.config.hasKey(MongoDbConnectorConfig.POLL_INTERVAL_SEC.name())) {
            this.logger.warn("The option `mongodb.poll.interval.sec` is deprecated. Use `mongodb.poll.interval.ms` instead.");
        }
        return Duration.ofSeconds(this.config.getInteger(MongoDbConnectorConfig.POLL_INTERVAL_SEC));
    }

    public int maxConnectionAttemptsForPrimary() {
        return this.config.getInteger(MongoDbConnectorConfig.MAX_FAILED_CONNECTIONS);
    }

    public int maxNumberOfCopyThreads() {
        return this.config.getInteger(MongoDbConnectorConfig.MAX_COPY_THREADS);
    }

    public MongoPrimary primaryFor(ReplicaSet replicaSet, Filters filters, BiConsumer<String, Throwable> errorHandler) {
        return new MongoPrimary(this, replicaSet, filters, errorHandler);
    }

    protected Supplier<MongoClient> primaryClientFor(ReplicaSet replicaSet) {
        return this.primaryClientFor(replicaSet, (attempts, remaining, error) -> {
            if (error == null) {
                this.logger().info("Unable to connect to primary node of '{}' after attempt #{} ({} remaining)", new Object[]{replicaSet, attempts, remaining});
            } else {
                this.logger().error("Error while attempting to connect to primary node of '{}' after attempt #{} ({} remaining): {}", new Object[]{replicaSet, attempts, remaining, error.getMessage(), error});
            }
        });
    }

    protected Supplier<MongoClient> primaryClientFor(ReplicaSet replicaSet, PrimaryConnectFailed handler) {
        Supplier<MongoClient> factory = () -> this.clientForPrimary(replicaSet);
        int maxAttempts = this.maxConnectionAttemptsForPrimary();
        return () -> {
            int attempts = 0;
            MongoClient primary = null;
            while (primary == null) {
                ++attempts;
                try {
                    primary = (MongoClient)factory.get();
                    if (primary != null) {
                        break;
                    }
                }
                catch (Throwable t) {
                    handler.failed(attempts, maxAttempts - attempts, t);
                }
                if (attempts > maxAttempts) {
                    throw new ConnectException("Unable to connect to primary node of '" + replicaSet + "' after " + attempts + " failed attempts");
                }
                handler.failed(attempts, maxAttempts - attempts, null);
                this.primaryBackoffStrategy.sleepWhen(true);
            }
            return primary;
        };
    }

    protected MongoClient clientForPrimary(ReplicaSet replicaSet) {
        MongoClient replicaSetClient = this.clientForReplicaSet(replicaSet);
        ReplicaSetStatus rsStatus = replicaSetClient.getReplicaSetStatus();
        if (rsStatus == null) {
            if (!this.useHostsAsSeeds) {
                return replicaSetClient;
            }
            throw new ConnectException("The MongoDB server(s) at '" + replicaSet + "' is not a valid replica set and cannot be used");
        }
        ServerAddress primaryAddress = rsStatus.getMaster();
        if (primaryAddress != null) {
            return this.pool.clientFor(primaryAddress);
        }
        return null;
    }

    public static class MongoPrimary {
        private final ReplicaSet replicaSet;
        private final Supplier<MongoClient> primaryConnectionSupplier;
        private final Filters filters;
        private final BiConsumer<String, Throwable> errorHandler;
        private final AtomicBoolean running = new AtomicBoolean(true);

        protected MongoPrimary(ConnectionContext context, ReplicaSet replicaSet, Filters filters, BiConsumer<String, Throwable> errorHandler) {
            this.replicaSet = replicaSet;
            this.primaryConnectionSupplier = context.primaryClientFor(replicaSet);
            this.filters = filters;
            this.errorHandler = errorHandler;
        }

        public ReplicaSet replicaSet() {
            return this.replicaSet;
        }

        public ServerAddress address() {
            return this.execute("get replica set primary", (MongoClient primary) -> {
                ReplicaSetStatus rsStatus = primary.getReplicaSetStatus();
                if (rsStatus != null) {
                    return rsStatus.getMaster();
                }
                return null;
            });
        }

        public void execute(String desc, Consumer<MongoClient> operation) {
            Metronome errorMetronome = Metronome.sleeper((Duration)PAUSE_AFTER_ERROR, (Clock)Clock.SYSTEM);
            while (true) {
                MongoClient primary = this.primaryConnectionSupplier.get();
                try {
                    operation.accept(primary);
                    return;
                }
                catch (Throwable t) {
                    this.errorHandler.accept(desc, t);
                    if (!this.isRunning()) {
                        throw new ConnectException("Operation failed and MongoDB primary termination requested", t);
                    }
                    try {
                        errorMetronome.pause();
                    }
                    catch (InterruptedException interruptedException) {
                    }
                    continue;
                }
                break;
            }
        }

        public <T> T execute(String desc, Function<MongoClient, T> operation) {
            Metronome errorMetronome = Metronome.sleeper((Duration)PAUSE_AFTER_ERROR, (Clock)Clock.SYSTEM);
            while (true) {
                MongoClient primary = this.primaryConnectionSupplier.get();
                try {
                    return operation.apply(primary);
                }
                catch (Throwable t) {
                    this.errorHandler.accept(desc, t);
                    if (!this.isRunning()) {
                        throw new ConnectException("Operation failed and MongoDB primary termination requested", t);
                    }
                    try {
                        errorMetronome.pause();
                    }
                    catch (InterruptedException interruptedException) {
                    }
                    continue;
                }
                break;
            }
        }

        public void executeBlocking(String desc, BlockingConsumer<MongoClient> operation) throws InterruptedException {
            Metronome errorMetronome = Metronome.sleeper((Duration)PAUSE_AFTER_ERROR, (Clock)Clock.SYSTEM);
            while (true) {
                MongoClient primary = this.primaryConnectionSupplier.get();
                try {
                    operation.accept((Object)primary);
                    return;
                }
                catch (InterruptedException e) {
                    throw e;
                }
                catch (Throwable t) {
                    this.errorHandler.accept(desc, t);
                    if (!this.isRunning()) {
                        throw new ConnectException("Operation failed and MongoDB primary termination requested", t);
                    }
                    errorMetronome.pause();
                    continue;
                }
                break;
            }
        }

        public Set<String> databaseNames() {
            return this.execute("get database names", (MongoClient primary) -> {
                HashSet databaseNames = new HashSet();
                MongoUtil.forEachDatabaseName(primary, dbName -> {
                    if (this.filters.databaseFilter().test((String)dbName)) {
                        databaseNames.add(dbName);
                    }
                });
                return databaseNames;
            });
        }

        public List<CollectionId> collections() {
            String replicaSetName = this.replicaSet.replicaSetName();
            return this.execute("get collections in databases", (MongoClient primary) -> {
                ArrayList collections = new ArrayList();
                Set<String> databaseNames = this.databaseNames();
                for (String dbName : databaseNames) {
                    MongoUtil.forEachCollectionNameInDatabase(primary, dbName, collectionName -> {
                        CollectionId collectionId = new CollectionId(replicaSetName, dbName, (String)collectionName);
                        if (this.filters.collectionFilter().test(collectionId)) {
                            collections.add(collectionId);
                        }
                    });
                }
                return collections;
            });
        }

        private boolean isRunning() {
            return this.running.get();
        }

        public void stop() {
            this.running.set(false);
        }
    }

    @FunctionalInterface
    public static interface PrimaryConnectFailed {
        public void failed(int var1, int var2, Throwable var3);
    }
}

