/*
 * Copyright Debezium Authors.
 *
 * Licensed under the Apache Software License version 2.0, available at http://www.apache.org/licenses/LICENSE-2.0
 */
package io.debezium.connector.mongodb.transforms;

import static org.fest.assertions.Assertions.assertThat;
import static org.junit.Assert.fail;

import java.io.IOException;
import java.util.Collections;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiConsumer;

import org.apache.kafka.connect.data.Schema;
import org.apache.kafka.connect.data.Struct;
import org.apache.kafka.connect.source.SourceRecord;
import org.bson.Document;
import org.bson.RawBsonDocument;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import io.debezium.config.Configuration;
import io.debezium.connector.mongodb.ConnectionContext.MongoPrimary;
import io.debezium.connector.mongodb.MongoDbConnector;
import io.debezium.connector.mongodb.MongoDbConnectorConfig;
import io.debezium.connector.mongodb.MongoDbTaskContext;
import io.debezium.connector.mongodb.ReplicaSet;
import io.debezium.connector.mongodb.TestHelper;
import io.debezium.embedded.AbstractConnectorTest;
import io.debezium.util.Testing;

/**
 * Integration test for {@link UnwrapFromMongoDbEnvelope}. It sends operations into
 * MongoDB and listens on messages that are generated by Debezium plug-in. The messages
 * are then run through the SMT itself.
 *
 * @author Jiri Pechanec
 */
public class UnwrapFromMongoDbEnvelopeTestIT extends AbstractConnectorTest {

    private static final String DB_NAME = "transform";
    private static final String COLLECTION_NAME = "source";
    private static final String TOPIC_NAME = "mongo.transform.source";

    private Configuration config;
    private MongoDbTaskContext context;
    private UnwrapFromMongoDbEnvelope<SourceRecord> transformation;

    @Before
    public void beforeEach() {
        Testing.Debug.disable();
        Testing.Print.disable();
        stopConnector();
        initializeConnectorTestFramework();

        transformation = new UnwrapFromMongoDbEnvelope<SourceRecord>();
        transformation.configure(Collections.emptyMap());
    }

    @After
    public void afterEach() {
        try {
            stopConnector();
        }
        finally {
            if (context != null) context.getConnectionContext().shutdown();
        }
        transformation.close();
    }

    @Test
    public void shouldTransformEvents() throws InterruptedException, IOException {

        // Use the DB configuration to define the connector's configuration ...
        config = TestHelper.getConfiguration().edit()
                              .with(MongoDbConnectorConfig.POLL_INTERVAL_MS, 10)
                              .with(MongoDbConnectorConfig.COLLECTION_WHITELIST, "transform.*")
                              .with(MongoDbConnectorConfig.LOGICAL_NAME, "mongo")
                              .build();

        // Set up the replication context for connections ...
        context = new MongoDbTaskContext(config);

        // Cleanup database
        TestHelper.cleanDatabase(primary(), DB_NAME);

        // Start the connector ...
        start(MongoDbConnector.class, config);

        // Test insert
        primary().execute("insert", client -> {
            client.getDatabase(DB_NAME).getCollection(COLLECTION_NAME)
                    .insertOne(Document.parse("{'_id': 1, 'dataStr': 'hello', 'dataInt': 123, 'dataLong': 80000000000}"));
        });

        SourceRecords records = consumeRecordsByTopic(1);

        assertThat(records.recordsForTopic(TOPIC_NAME).size()).isEqualTo(1);
        final SourceRecord insertRecord = records.recordsForTopic(TOPIC_NAME).get(0);
        final SourceRecord transformedInsert = transformation.apply(insertRecord);
        final Struct transformedInsertValue = (Struct)transformedInsert.value();

        assertThat(transformedInsert.valueSchema().field("id").schema()).isEqualTo(Schema.OPTIONAL_INT32_SCHEMA);
        assertThat(transformedInsert.valueSchema().field("dataStr").schema()).isEqualTo(Schema.OPTIONAL_STRING_SCHEMA);
        assertThat(transformedInsert.valueSchema().field("dataInt").schema()).isEqualTo(Schema.OPTIONAL_INT32_SCHEMA);
        assertThat(transformedInsert.valueSchema().field("dataLong").schema()).isEqualTo(Schema.OPTIONAL_INT64_SCHEMA);
        assertThat(transformedInsertValue.get("id")).isEqualTo(1);
        assertThat(transformedInsertValue.get("dataStr")).isEqualTo("hello");
        assertThat(transformedInsertValue.get("dataInt")).isEqualTo(123);
        assertThat(transformedInsertValue.get("dataLong")).isEqualTo(80_000_000_000l);

        // Test update
        primary().execute("update", client -> {
            client.getDatabase(DB_NAME).getCollection(COLLECTION_NAME).updateOne(RawBsonDocument.parse("{'_id' : 1}"),
                    RawBsonDocument.parse("{'$set': {'dataStr': 'bye'}}"));
        });

        records = consumeRecordsByTopic(1);
        final SourceRecord candidateRecord = records.recordsForTopic(TOPIC_NAME).get(0);
        if (((Struct)candidateRecord.value()).get("op").equals("c")) {
            // MongoDB is not providing really consistent snapshot, so the initial insert
            // can arrive both in initial sync snapshot and in oplog
            records = consumeRecordsByTopic(1);
        }

        assertThat(records.recordsForTopic(TOPIC_NAME).size()).isEqualTo(1);
        final SourceRecord updateRecord = records.recordsForTopic(TOPIC_NAME).get(0);
        final SourceRecord transformedUpdate = transformation.apply(updateRecord);
        final Struct transformedUpdateValue = (Struct)transformedUpdate.value();

        assertThat(transformedUpdate.valueSchema().field("id").schema()).isEqualTo(Schema.OPTIONAL_INT32_SCHEMA);
        assertThat(transformedUpdate.valueSchema().field("dataStr").schema()).isEqualTo(Schema.OPTIONAL_STRING_SCHEMA);
        assertThat(transformedUpdateValue.get("id")).isEqualTo(1);
        assertThat(transformedUpdateValue.get("dataStr")).isEqualTo("bye");

        // Test update
        primary().execute("delete", client -> {
            client.getDatabase(DB_NAME).getCollection(COLLECTION_NAME).deleteOne(RawBsonDocument.parse("{'_id' : 1}"));
        });

        records = consumeRecordsByTopic(2);

        assertThat(records.recordsForTopic(TOPIC_NAME).size()).isEqualTo(2);
        final SourceRecord deleteRecord = records.recordsForTopic(TOPIC_NAME).get(0);
        final SourceRecord transformedDelete = transformation.apply(deleteRecord);
        final Struct transformedDeleteValue = (Struct)transformedDelete.value();

        assertThat(transformedDeleteValue).isNull();
        assertThat(records.recordsForTopic(TOPIC_NAME).get(1).value()).isNull();
}

    private MongoPrimary primary() {
        ReplicaSet replicaSet = ReplicaSet.parse(context.getConnectionContext().hosts());
        return context.getConnectionContext().primaryFor(replicaSet, context.filters(), connectionErrorHandler(3));
    }

    private BiConsumer<String, Throwable> connectionErrorHandler(int numErrorsBeforeFailing) {
        AtomicInteger attempts = new AtomicInteger();
        return (desc, error) -> {
            if (attempts.incrementAndGet() > numErrorsBeforeFailing) {
                fail("Unable to connect to primary after " + numErrorsBeforeFailing + " errors trying to " + desc + ": " + error);
            }
            logger.error("Error while attempting to {}: {}", desc, error.getMessage(), error);
        };
    }
}
