/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.connector.mongodb;

import io.debezium.annotation.Immutable;
import io.debezium.annotation.ThreadSafe;
import io.debezium.config.Configuration;
import io.debezium.config.ConfigurationDefaults;
import io.debezium.connector.mongodb.Module;
import io.debezium.connector.mongodb.MongoDbConnectorConfig;
import io.debezium.connector.mongodb.ReplicaSet;
import io.debezium.connector.mongodb.ReplicaSets;
import io.debezium.connector.mongodb.ReplicationContext;
import io.debezium.connector.mongodb.Replicator;
import io.debezium.connector.mongodb.SourceInfo;
import io.debezium.function.BlockingConsumer;
import io.debezium.time.Temporals;
import io.debezium.util.Clock;
import io.debezium.util.LoggingContext;
import io.debezium.util.Metronome;
import io.debezium.util.Threads;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Deque;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BooleanSupplier;
import java.util.function.Consumer;
import org.apache.kafka.connect.errors.ConnectException;
import org.apache.kafka.connect.source.SourceRecord;
import org.apache.kafka.connect.source.SourceTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public final class MongoDbConnectorTask
extends SourceTask {
    private final Logger logger = LoggerFactory.getLogger(((Object)((Object)this)).getClass());
    private final AtomicBoolean running = new AtomicBoolean(false);
    private final Deque<Replicator> replicators = new ConcurrentLinkedDeque<Replicator>();
    private final RecordBatchSummarizer recordSummarizer = new RecordBatchSummarizer();
    private volatile TaskRecordQueue queue;
    private volatile String taskName;
    private volatile ReplicationContext replContext;

    public String version() {
        return Module.version();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void start(Map<String, String> props) {
        ReplicationContext replicationContext;
        if (!this.running.compareAndSet(false, true)) {
            return;
        }
        if (this.context == null) {
            throw new ConnectException("Unexpected null context");
        }
        Configuration config = Configuration.from(props);
        this.taskName = "task" + config.getInteger(MongoDbConnectorConfig.TASK_ID);
        this.replContext = replicationContext = new ReplicationContext(config);
        LoggingContext.PreviousContext previousLogContext = replicationContext.configureLoggingContext(this.taskName);
        try {
            this.logger.info("Starting MongoDB connector task with configuration:");
            config.forEach((propName, propValue) -> this.logger.info("   {} = {}", propName, propValue));
            if (!config.validateAndRecord((Iterable)MongoDbConnectorConfig.ALL_FIELDS, arg_0 -> ((Logger)this.logger).error(arg_0))) {
                throw new ConnectException("Error configuring an instance of " + ((Object)((Object)this)).getClass().getSimpleName() + "; check the logs for details");
            }
            String hosts = config.getString(MongoDbConnectorConfig.HOSTS);
            ReplicaSets replicaSets = ReplicaSets.parse(hosts);
            if (replicaSets.validReplicaSetCount() == 0) {
                this.logger.info("Unable to start MongoDB connector task since no replica sets were found at {}", (Object)hosts);
                return;
            }
            this.queue = new TaskRecordQueue(config, replicaSets.replicaSetCount(), this.running::get, this.recordSummarizer);
            SourceInfo source = replicationContext.source();
            ArrayList partitions = new ArrayList();
            replicaSets.onEachReplicaSet(replicaSet -> {
                String replicaSetName = replicaSet.replicaSetName();
                if (replicaSetName != null) {
                    partitions.add(source.partition(replicaSetName));
                }
            });
            this.context.offsetStorageReader().offsets(partitions).forEach(source::setOffsetFor);
            int numThreads = replicaSets.replicaSetCount();
            ExecutorService executor = Executors.newFixedThreadPool(numThreads);
            AtomicInteger stillRunning = new AtomicInteger(numThreads);
            this.logger.info("Ignoring unnamed replica sets: {}", replicaSets.unnamedReplicaSets());
            this.logger.info("Starting {} thread(s) to replicate replica sets: {}", (Object)numThreads, (Object)replicaSets);
            replicaSets.validReplicaSets().forEach(replicaSet -> {
                Replicator replicator = new Replicator(replicationContext, (ReplicaSet)replicaSet, (BlockingConsumer<SourceRecord>)((BlockingConsumer)this.queue::enqueue));
                this.replicators.add(replicator);
                executor.submit(() -> {
                    try {
                        replicationContext.configureLoggingContext(replicaSet.replicaSetName());
                        replicator.run();
                    }
                    finally {
                        try {
                            this.replicators.remove(replicator);
                        }
                        finally {
                            if (stillRunning.decrementAndGet() == 0) {
                                try {
                                    executor.shutdown();
                                }
                                finally {
                                    replicationContext.shutdown();
                                }
                            }
                        }
                    }
                });
            });
            this.logger.info("Successfully started MongoDB connector task with {} thread(s) for replica sets {}", (Object)numThreads, (Object)replicaSets);
        }
        finally {
            previousLogContext.restore();
        }
    }

    public List<SourceRecord> poll() throws InterruptedException {
        return this.queue.poll();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void stop() {
        LoggingContext.PreviousContext previousLogContext = this.replContext.configureLoggingContext(this.taskName);
        try {
            if (this.running.compareAndSet(true, false)) {
                this.logger.info("Stopping MongoDB task");
                Replicator replicator = null;
                int counter = 0;
                while ((replicator = this.replicators.poll()) != null) {
                    replicator.stop();
                    ++counter;
                }
                this.logger.info("Stopped MongoDB replication task by stopping {} replicator threads", (Object)counter);
            }
        }
        catch (Throwable e) {
            this.logger.error("Unexpected error shutting down the MongoDB replication task", e);
        }
        finally {
            previousLogContext.restore();
        }
    }

    protected static final class ReplicaSetSummary {
        private int numRecords = 0;
        private Map<String, ?> lastOffset;

        protected ReplicaSetSummary() {
        }

        public void add(SourceRecord record) {
            ++this.numRecords;
            this.lastOffset = record.sourceOffset();
        }

        public int recordCount() {
            return this.numRecords;
        }

        public Map<String, ?> lastOffset() {
            return this.lastOffset;
        }
    }

    protected final class RecordBatchSummarizer
    implements Consumer<List<SourceRecord>> {
        private final Map<String, ReplicaSetSummary> summaryByReplicaSet = new HashMap<String, ReplicaSetSummary>();

        protected RecordBatchSummarizer() {
        }

        @Override
        public void accept(List<SourceRecord> records) {
            if (records.isEmpty()) {
                return;
            }
            if (!MongoDbConnectorTask.this.logger.isInfoEnabled()) {
                return;
            }
            this.summaryByReplicaSet.clear();
            records.forEach(record -> {
                String replicaSetName = SourceInfo.replicaSetNameForPartition(record.sourcePartition());
                if (replicaSetName != null) {
                    this.summaryByReplicaSet.computeIfAbsent(replicaSetName, rsName -> new ReplicaSetSummary()).add((SourceRecord)record);
                }
            });
            if (!this.summaryByReplicaSet.isEmpty()) {
                LoggingContext.PreviousContext prevContext = MongoDbConnectorTask.this.replContext.configureLoggingContext("task");
                try {
                    this.summaryByReplicaSet.forEach((rsName, summary) -> MongoDbConnectorTask.this.logger.info("{} records sent for replica set '{}', last offset: {}", new Object[]{summary.recordCount(), rsName, summary.lastOffset()}));
                }
                finally {
                    prevContext.restore();
                }
            }
        }
    }

    @Immutable
    protected static class TaskRecordQueue {
        private final int maxBatchSize;
        private final Metronome metronome;
        private final BlockingQueue<SourceRecord> records;
        private final BooleanSupplier isRunning;
        private final Consumer<List<SourceRecord>> batchConsumer;
        private final Duration pollInterval;

        protected TaskRecordQueue(Configuration config, int numThreads, BooleanSupplier isRunning, Consumer<List<SourceRecord>> batchConsumer) {
            int maxQueueSize = config.getInteger(MongoDbConnectorConfig.MAX_QUEUE_SIZE);
            this.pollInterval = Duration.ofMillis(config.getLong(MongoDbConnectorConfig.POLL_INTERVAL_MS));
            this.maxBatchSize = config.getInteger(MongoDbConnectorConfig.MAX_BATCH_SIZE);
            this.metronome = Metronome.parker((Duration)this.pollInterval, (Clock)Clock.SYSTEM);
            this.records = new LinkedBlockingDeque<SourceRecord>(maxQueueSize);
            this.isRunning = isRunning;
            this.batchConsumer = batchConsumer != null ? batchConsumer : records -> {};
        }

        public List<SourceRecord> poll() throws InterruptedException {
            ArrayList<SourceRecord> batch = new ArrayList<SourceRecord>(this.maxBatchSize);
            Threads.Timer timeout = Threads.timer((Clock)Clock.SYSTEM, (Duration)Temporals.max((Duration)this.pollInterval, (Duration)ConfigurationDefaults.RETURN_CONTROL_INTERVAL));
            while (this.isRunning.getAsBoolean() && this.records.drainTo(batch, this.maxBatchSize) == 0) {
                this.metronome.pause();
                if (!timeout.expired()) continue;
            }
            this.batchConsumer.accept(batch);
            return batch;
        }

        public void enqueue(SourceRecord record) throws InterruptedException {
            if (record != null) {
                this.records.put(record);
            }
        }
    }
}

