/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.connector.mongodb;

import com.mongodb.CursorType;
import com.mongodb.MongoClient;
import com.mongodb.ServerAddress;
import com.mongodb.client.FindIterable;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.MongoCursor;
import com.mongodb.client.MongoDatabase;
import com.mongodb.client.model.Filters;
import io.debezium.annotation.ThreadSafe;
import io.debezium.config.CommonConnectorConfig;
import io.debezium.config.ConfigurationDefaults;
import io.debezium.connector.SnapshotRecord;
import io.debezium.connector.mongodb.CollectionId;
import io.debezium.connector.mongodb.ConnectionContext;
import io.debezium.connector.mongodb.MongoDbConnector;
import io.debezium.connector.mongodb.MongoDbConnectorConfig;
import io.debezium.connector.mongodb.MongoDbTaskContext;
import io.debezium.connector.mongodb.RecordMakers;
import io.debezium.connector.mongodb.ReplicaSet;
import io.debezium.connector.mongodb.SourceInfo;
import io.debezium.function.BlockingConsumer;
import io.debezium.function.BufferedBlockingConsumer;
import io.debezium.util.Clock;
import io.debezium.util.Metronome;
import io.debezium.util.Strings;
import io.debezium.util.Threads;
import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import org.apache.kafka.connect.data.Struct;
import org.apache.kafka.connect.errors.ConnectException;
import org.apache.kafka.connect.source.SourceRecord;
import org.bson.BsonTimestamp;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@ThreadSafe
public class Replicator {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private static final String AUTHORIZATION_FAILURE_MESSAGE = "Command failed with error 13";
    private final MongoDbTaskContext context;
    private final ExecutorService copyThreads;
    private final ReplicaSet replicaSet;
    private final String rsName;
    private final AtomicBoolean running = new AtomicBoolean();
    private final SourceInfo source;
    private final RecordMakers recordMakers;
    private final BufferableRecorder bufferedRecorder;
    private final Clock clock;
    private ConnectionContext.MongoPrimary primaryClient;
    private final Consumer<Throwable> onFailure;

    public Replicator(MongoDbTaskContext context, ReplicaSet replicaSet, BlockingConsumer<SourceRecord> recorder, Consumer<Throwable> onFailure) {
        assert (context != null);
        assert (replicaSet != null);
        assert (recorder != null);
        this.context = context;
        this.source = context.source();
        this.replicaSet = replicaSet;
        this.rsName = replicaSet.replicaSetName();
        String copyThreadName = "copy-" + (replicaSet.hasReplicaSetName() ? replicaSet.replicaSetName() : "main");
        this.copyThreads = Threads.newFixedThreadPool(MongoDbConnector.class, (String)context.serverName(), (String)copyThreadName, (int)context.getConnectionContext().maxNumberOfCopyThreads());
        this.bufferedRecorder = new BufferableRecorder(recorder);
        this.recordMakers = new RecordMakers(context.filters(), this.source, context.topicSelector(), this.bufferedRecorder, context.isEmitTombstoneOnDelete());
        this.clock = this.context.getClock();
        this.onFailure = onFailure;
    }

    public void stop() {
        this.copyThreads.shutdownNow();
        this.running.set(false);
    }

    public void run() {
        if (this.running.compareAndSet(false, true)) {
            try {
                if (this.establishConnectionToPrimary()) {
                    if (this.isInitialSyncExpected()) {
                        boolean snapshotCompleted;
                        this.recordCurrentOplogPosition();
                        if (this.context.getConnectorConfig().getSnapshotMode() == MongoDbConnectorConfig.SnapshotMode.INITIAL && !(snapshotCompleted = this.performInitialSync())) {
                            return;
                        }
                    }
                    this.readOplog();
                }
            }
            catch (Throwable t) {
                this.logger.error("Replicator for replica set {} failed", (Object)this.rsName, (Object)t);
                this.onFailure.accept(t);
            }
            finally {
                if (this.primaryClient != null) {
                    this.primaryClient.stop();
                }
                this.running.set(false);
            }
        }
    }

    protected boolean establishConnectionToPrimary() {
        this.logger.info("Connecting to '{}'", (Object)this.replicaSet);
        this.primaryClient = this.context.getConnectionContext().primaryFor(this.replicaSet, this.context.filters(), (desc, error) -> {
            if (error.getMessage() != null && error.getMessage().startsWith(AUTHORIZATION_FAILURE_MESSAGE)) {
                throw new ConnectException("Error while attempting to " + desc, error);
            }
            this.logger.error("Error while attempting to {}: {}", new Object[]{desc, error.getMessage(), error});
        });
        return this.primaryClient != null;
    }

    protected void recordCurrentOplogPosition() {
        this.primaryClient.execute("get oplog position", primary -> {
            MongoCollection oplog = primary.getDatabase("local").getCollection("oplog.rs");
            Document last = (Document)oplog.find().sort((Bson)new Document("$natural", (Object)-1)).limit(1).first();
            this.source.opLogEvent(this.replicaSet.replicaSetName(), last);
        });
    }

    protected boolean isInitialSyncExpected() {
        boolean performSnapshot = true;
        if (this.source.hasOffset(this.rsName)) {
            if (this.logger.isInfoEnabled()) {
                this.logger.info("Found existing offset for replica set '{}' at {}", (Object)this.rsName, this.source.lastOffset(this.rsName));
            }
            performSnapshot = false;
            if (this.context.getConnectionContext().performSnapshotEvenIfNotNeeded()) {
                this.logger.info("Configured to performing initial sync of replica set '{}'", (Object)this.rsName);
                performSnapshot = true;
            } else if (this.source.isInitialSyncOngoing(this.rsName)) {
                this.logger.info("The previous initial sync was incomplete for '{}', so initiating another initial sync", (Object)this.rsName);
                performSnapshot = true;
            } else {
                BsonTimestamp lastRecordedTs = this.source.lastOffsetTimestamp(this.rsName);
                BsonTimestamp firstAvailableTs = this.primaryClient.execute("get oplog position", primary -> {
                    MongoCollection oplog = primary.getDatabase("local").getCollection("oplog.rs");
                    Document firstEvent = (Document)oplog.find().sort((Bson)new Document("$natural", (Object)1)).limit(1).first();
                    return SourceInfo.extractEventTimestamp(firstEvent);
                });
                if (firstAvailableTs == null) {
                    this.logger.info("The oplog contains no entries, so performing initial sync of replica set '{}'", (Object)this.rsName);
                    performSnapshot = true;
                } else if (lastRecordedTs.compareTo(firstAvailableTs) < 0) {
                    this.logger.info("Initial sync is required since the oplog for replica set '{}' starts at {}, which is later than the timestamp of the last offset {}", new Object[]{this.rsName, firstAvailableTs, lastRecordedTs});
                    performSnapshot = true;
                } else {
                    this.logger.info("The oplog contains the last entry previously read for '{}', so no initial sync will be performed", (Object)this.rsName);
                }
            }
        } else {
            this.logger.info("No existing offset found for replica set '{}', starting initial sync", (Object)this.rsName);
            performSnapshot = true;
        }
        return performSnapshot;
    }

    protected boolean performInitialSync() {
        try {
            this.delaySnapshotIfNeeded();
        }
        catch (InterruptedException e) {
            this.logger.info("Interrupted while awaiting initial snapshot delay");
            return false;
        }
        if (this.logger.isInfoEnabled()) {
            this.logger.info("Beginning initial sync of '{}' at {}", (Object)this.rsName, this.source.lastOffset(this.rsName));
        }
        this.source.startInitialSync(this.replicaSet.replicaSetName());
        try {
            this.bufferedRecorder.startBuffering();
        }
        catch (InterruptedException e) {
            this.logger.info("Interrupted while waiting to flush the buffer before starting an initial sync of '{}'", (Object)this.rsName);
            return false;
        }
        long syncStart = this.clock.currentTimeInMillis();
        List<CollectionId> collections = this.primaryClient.collections();
        ConcurrentLinkedQueue<CollectionId> collectionsToCopy = new ConcurrentLinkedQueue<CollectionId>(collections);
        int numThreads = Math.min(collections.size(), this.context.getConnectionContext().maxNumberOfCopyThreads());
        CountDownLatch latch = new CountDownLatch(numThreads);
        AtomicBoolean aborted = new AtomicBoolean(false);
        AtomicInteger replicatorThreadCounter = new AtomicInteger(0);
        AtomicInteger numCollectionsCopied = new AtomicInteger();
        AtomicLong numDocumentsCopied = new AtomicLong();
        if (this.logger.isInfoEnabled()) {
            this.logger.info("Preparing to use {} thread(s) to sync {} collection(s): {}", new Object[]{numThreads, collections.size(), Strings.join((CharSequence)", ", collections)});
        }
        for (int i = 0; i != numThreads; ++i) {
            this.copyThreads.submit(() -> {
                this.context.configureLoggingContext(this.replicaSet.replicaSetName() + "-sync" + replicatorThreadCounter.incrementAndGet());
                try {
                    CollectionId id = null;
                    while (!aborted.get() && (id = (CollectionId)collectionsToCopy.poll()) != null) {
                        long start = this.clock.currentTimeInMillis();
                        this.logger.info("Starting initial sync of '{}'", (Object)id);
                        long numDocs = this.copyCollection(id, syncStart);
                        numCollectionsCopied.incrementAndGet();
                        numDocumentsCopied.addAndGet(numDocs);
                        if (!this.logger.isInfoEnabled()) continue;
                        long duration = this.clock.currentTimeInMillis() - start;
                        this.logger.info("Completing initial sync of {} documents from '{}' in {}", new Object[]{numDocs, id, Strings.duration((long)duration)});
                    }
                }
                catch (InterruptedException e) {
                    aborted.set(true);
                }
                finally {
                    latch.countDown();
                }
            });
        }
        try {
            latch.await();
        }
        catch (InterruptedException e) {
            Thread.interrupted();
            aborted.set(true);
        }
        this.copyThreads.shutdown();
        long syncDuration = this.clock.currentTimeInMillis() - syncStart;
        if (aborted.get()) {
            if (this.logger.isInfoEnabled()) {
                int remaining = collections.size() - numCollectionsCopied.get();
                this.logger.info("Initial sync aborted after {} with {} of {} collections incomplete", new Object[]{Strings.duration((long)syncDuration), remaining, collections.size()});
            }
            return false;
        }
        this.source.stopInitialSync(this.replicaSet.replicaSetName());
        try {
            this.bufferedRecorder.stopBuffering(this.source.lastOffset(this.rsName));
        }
        catch (InterruptedException e) {
            this.logger.info("Interrupted while waiting for last initial sync record from replica set '{}' to be recorded", (Object)this.rsName);
            return false;
        }
        if (collections.isEmpty()) {
            this.logger.warn("After applying blacklist/whitelist filters there are no tables to monitor, please check your configuration");
        }
        if (this.logger.isInfoEnabled()) {
            this.logger.info("Initial sync of {} collections with a total of {} documents completed in {}", new Object[]{collections.size(), numDocumentsCopied.get(), Strings.duration((long)syncDuration)});
        }
        return true;
    }

    private void delaySnapshotIfNeeded() throws InterruptedException {
        Duration delay = Duration.ofMillis(this.context.getConnectionContext().config.getLong(CommonConnectorConfig.SNAPSHOT_DELAY_MS));
        if (delay.isZero() || delay.isNegative()) {
            return;
        }
        Threads.Timer timer = Threads.timer((Clock)Clock.SYSTEM, (Duration)delay);
        Metronome metronome = Metronome.parker((Duration)ConfigurationDefaults.RETURN_CONTROL_INTERVAL, (Clock)Clock.SYSTEM);
        while (!timer.expired()) {
            if (!this.running.get()) {
                throw new InterruptedException("Interrupted while awaiting initial snapshot delay");
            }
            this.logger.info("The connector will wait for {}s before proceeding", (Object)timer.remaining().getSeconds());
            metronome.pause();
        }
    }

    protected long copyCollection(CollectionId collectionId, long timestamp) throws InterruptedException {
        AtomicLong docCount = new AtomicLong();
        this.primaryClient.executeBlocking("sync '" + collectionId + "'", (BlockingConsumer<MongoClient>)((BlockingConsumer)primary -> docCount.set(this.copyCollection((MongoClient)primary, collectionId, timestamp))));
        return docCount.get();
    }

    protected long copyCollection(MongoClient primary, CollectionId collectionId, long timestamp) throws InterruptedException {
        RecordMakers.RecordsForCollection factory = this.recordMakers.forCollection(collectionId);
        MongoDatabase db = primary.getDatabase(collectionId.dbName());
        MongoCollection docCollection = db.getCollection(collectionId.name());
        long counter = 0L;
        int batchSize = this.context.getConnectorConfig().getSnapshotFetchSize();
        try (MongoCursor cursor = docCollection.find().batchSize(batchSize).iterator();){
            while (this.running.get() && cursor.hasNext()) {
                Document doc = (Document)cursor.next();
                this.logger.trace("Found existing doc in {}: {}", (Object)collectionId, (Object)doc);
                counter += (long)factory.recordObject(collectionId, doc, timestamp);
            }
        }
        return counter;
    }

    protected void readOplog() {
        this.primaryClient.execute("read from oplog on '" + this.replicaSet + "'", this::readOplog);
    }

    protected void readOplog(MongoClient primary) {
        BsonTimestamp oplogStart = this.source.lastOffsetTimestamp(this.replicaSet.replicaSetName());
        ServerAddress primaryAddress = primary.getAddress();
        this.logger.info("Reading oplog for '{}' primary {} starting at {}", new Object[]{this.replicaSet, primaryAddress, oplogStart});
        MongoCollection oplog = primary.getDatabase("local").getCollection("oplog.rs");
        Bson filter = Filters.and((Bson[])new Bson[]{Filters.gt((String)"ts", (Object)oplogStart), Filters.exists((String)"fromMigrate", (boolean)false)});
        FindIterable results = oplog.find(filter).sort((Bson)new Document("$natural", (Object)1)).oplogReplay(true).cursorType(CursorType.TailableAwait);
        try (MongoCursor cursor = results.iterator();){
            while (this.running.get() && cursor.hasNext()) {
                if (this.handleOplogEvent(primaryAddress, (Document)cursor.next())) continue;
                return;
            }
        }
    }

    protected boolean handleOplogEvent(ServerAddress primaryAddress, Document event) {
        this.logger.debug("Found event: {}", (Object)event);
        String ns = event.getString((Object)"ns");
        Document object = (Document)event.get((Object)"o", Document.class);
        if (object == null) {
            if (this.logger.isWarnEnabled()) {
                this.logger.warn("Missing 'o' field in event, so skipping {}", (Object)event.toJson());
            }
            return true;
        }
        if (ns == null || ns.isEmpty()) {
            String msg = object.getString((Object)"msg");
            if ("new primary".equals(msg)) {
                AtomicReference address = new AtomicReference();
                try {
                    this.primaryClient.executeBlocking("conn", (BlockingConsumer<MongoClient>)((BlockingConsumer)mongoClient -> {
                        ServerAddress currentPrimary = mongoClient.getAddress();
                        address.set(currentPrimary);
                    }));
                }
                catch (InterruptedException e) {
                    this.logger.error("Get current primary executeBlocking", (Throwable)e);
                }
                ServerAddress serverAddress = (ServerAddress)address.get();
                if (serverAddress != null && !serverAddress.equals((Object)primaryAddress)) {
                    this.logger.info("Found new primary event in oplog, so stopping use of {} to continue with new primary {}", (Object)primaryAddress, (Object)serverAddress);
                } else {
                    this.logger.info("Found new primary event in oplog, current {} is new primary. Continue to process oplog event.", (Object)primaryAddress);
                }
            }
            if (this.logger.isDebugEnabled()) {
                this.logger.debug("Skipping event with no namespace: {}", (Object)event.toJson());
            }
            return true;
        }
        int delimIndex = ns.indexOf(46);
        if (delimIndex > 0) {
            assert (delimIndex + 1 < ns.length());
            String dbName = ns.substring(0, delimIndex);
            String collectionName = ns.substring(delimIndex + 1);
            if ("$cmd".equals(collectionName)) {
                this.logger.debug("Skipping database command event: {}", (Object)event.toJson());
                return true;
            }
            if (!this.context.filters().databaseFilter().test(dbName)) {
                this.logger.debug("Skipping the event for database {} based on database.whitelist", (Object)dbName);
                return true;
            }
            CollectionId collectionId = new CollectionId(this.rsName, dbName, collectionName);
            if (this.context.filters().collectionFilter().test(collectionId)) {
                RecordMakers.RecordsForCollection factory = this.recordMakers.forCollection(collectionId);
                try {
                    factory.recordEvent(event, this.clock.currentTimeInMillis());
                }
                catch (InterruptedException e) {
                    Thread.interrupted();
                    return false;
                }
            }
        }
        return true;
    }

    protected final class BufferableRecorder
    implements BlockingConsumer<SourceRecord> {
        private final BlockingConsumer<SourceRecord> actual;
        private BufferedBlockingConsumer<SourceRecord> buffered;
        private volatile BlockingConsumer<SourceRecord> current;

        public BufferableRecorder(BlockingConsumer<SourceRecord> actual) {
            this.actual = actual;
            this.current = this.actual;
        }

        protected synchronized void startBuffering() throws InterruptedException {
            this.buffered = BufferedBlockingConsumer.bufferLast(this.actual);
            this.current = this.buffered;
        }

        protected synchronized void stopBuffering(Map<String, ?> newOffset) throws InterruptedException {
            assert (newOffset != null);
            this.buffered.close(record -> {
                if (record == null) {
                    return null;
                }
                Struct envelope = (Struct)record.value();
                Struct source = (Struct)envelope.get("source");
                if (SnapshotRecord.fromSource((Struct)source) == SnapshotRecord.TRUE) {
                    SnapshotRecord.LAST.toSource(source);
                }
                return new SourceRecord(record.sourcePartition(), newOffset, record.topic(), record.kafkaPartition(), record.keySchema(), record.key(), record.valueSchema(), record.value());
            });
            this.current = this.actual;
        }

        public void accept(SourceRecord t) throws InterruptedException {
            this.current.accept((Object)t);
        }
    }
}

