/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.connector.jdbc;

import io.debezium.connector.jdbc.JdbcSinkConnectorConfig;
import io.debezium.connector.jdbc.SinkRecordDescriptor;
import io.debezium.connector.jdbc.dialect.DatabaseDialect;
import io.debezium.connector.jdbc.dialect.DatabaseDialectResolver;
import io.debezium.connector.jdbc.relational.TableDescriptor;
import io.debezium.pipeline.sink.spi.ChangeEventSink;
import java.sql.SQLException;
import java.util.List;
import java.util.Set;
import org.apache.kafka.connect.data.Struct;
import org.apache.kafka.connect.errors.ConnectException;
import org.apache.kafka.connect.sink.SinkRecord;
import org.hibernate.SessionFactory;
import org.hibernate.StatelessSession;
import org.hibernate.Transaction;
import org.hibernate.dialect.DatabaseVersion;
import org.hibernate.query.NativeQuery;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JdbcChangeEventSink
implements ChangeEventSink {
    private static final Logger LOGGER = LoggerFactory.getLogger(JdbcChangeEventSink.class);
    private final JdbcSinkConnectorConfig config;
    private final SessionFactory sessionFactory;
    private final DatabaseDialect dialect;
    private final StatelessSession session;

    public JdbcChangeEventSink(JdbcSinkConnectorConfig config) {
        this.config = config;
        this.sessionFactory = config.getHibernateConfiguration().buildSessionFactory();
        this.dialect = DatabaseDialectResolver.resolve(config, this.sessionFactory);
        this.session = this.sessionFactory.openStatelessSession();
        DatabaseVersion version = this.dialect.getVersion();
        LOGGER.info("Database version {}.{}.{}", new Object[]{version.getMajor(), version.getMinor(), version.getMicro()});
    }

    @Override
    public void execute(SinkRecord record) {
        try {
            SinkRecordDescriptor descriptor = SinkRecordDescriptor.builder().withPrimaryKeyMode(this.config.getPrimaryKeyMode()).withPrimaryKeyFields(this.config.getPrimaryKeyFields()).withSinkRecord(record).withDialect(this.dialect).build();
            TableDescriptor table = this.checkAndApplyTableChangesIfNeeded(record, descriptor);
            this.write(table, descriptor);
        }
        catch (Exception e) {
            throw new ConnectException("Failed to process a sink record", (Throwable)e);
        }
    }

    @Override
    public void close() throws Exception {
        if (this.session != null && this.session.isOpen()) {
            LOGGER.info("Closing session.");
            this.session.close();
        } else {
            LOGGER.info("Session already closed.");
        }
        if (this.sessionFactory != null && this.sessionFactory.isOpen()) {
            LOGGER.info("Closing the session factory");
            this.sessionFactory.close();
        } else {
            LOGGER.info("Session factory already closed");
        }
    }

    private TableDescriptor checkAndApplyTableChangesIfNeeded(SinkRecord record, SinkRecordDescriptor descriptor) throws SQLException {
        String tableName = this.config.getTableNamingStrategy().resolveTableName(this.config, record);
        if (!this.hasTable(tableName)) {
            try {
                return this.createTable(tableName, descriptor);
            }
            catch (SQLException ce) {
                LOGGER.warn("Table creation failed for '{}', attempting to alter the table", (Object)tableName, (Object)ce);
                try {
                    return this.alterTableIfNeeded(tableName, descriptor);
                }
                catch (SQLException ae) {
                    LOGGER.error("Failed to alter the table '{}'.", (Object)tableName, (Object)ae);
                    throw ae;
                }
            }
        }
        try {
            return this.alterTableIfNeeded(tableName, descriptor);
        }
        catch (SQLException ae) {
            LOGGER.error("Failed to alter the table '{}'.", (Object)tableName, (Object)ae);
            throw ae;
        }
    }

    private boolean hasTable(String tableName) {
        return (Boolean)this.session.doReturningWork(connection -> this.dialect.tableExists(connection, tableName));
    }

    private TableDescriptor readTable(String tableName) {
        return (TableDescriptor)this.session.doReturningWork(connection -> this.dialect.readTable(connection, tableName));
    }

    private TableDescriptor createTable(String tableName, SinkRecordDescriptor record) throws SQLException {
        LOGGER.debug("Attempting to create table '{}'.", (Object)tableName);
        if (JdbcSinkConnectorConfig.SchemaEvolutionMode.NONE.equals((Object)this.config.getSchemaEvolutionMode())) {
            LOGGER.warn("Table '{}' cannot be created because schema evolution is disabled.", (Object)tableName);
            throw new SQLException("Cannot create table " + tableName + " because schema evolution is disabled");
        }
        Transaction transaction = this.session.beginTransaction();
        try {
            String createSql = this.dialect.getCreateTableStatement(record, tableName);
            LOGGER.trace("SQL: {}", (Object)createSql);
            this.session.createNativeQuery(createSql, Object.class).executeUpdate();
            transaction.commit();
        }
        catch (Exception e) {
            transaction.rollback();
            throw e;
        }
        return this.readTable(tableName);
    }

    private TableDescriptor alterTableIfNeeded(String tableName, SinkRecordDescriptor record) throws SQLException {
        LOGGER.debug("Attempting to alter table '{}'.", (Object)tableName);
        if (!this.hasTable(tableName)) {
            LOGGER.error("Table '{}' does not exist and cannot be altered.", (Object)tableName);
            throw new SQLException("Could not find table: " + tableName);
        }
        TableDescriptor table = this.readTable(tableName);
        Set<String> missingFields = this.dialect.resolveMissingFields(record, table);
        if (missingFields.isEmpty()) {
            return table;
        }
        LOGGER.debug("The follow fields are missing in the table: {}", missingFields);
        for (String missingFieldName : missingFields) {
            SinkRecordDescriptor.FieldDescriptor fieldDescriptor = record.getFields().get(missingFieldName);
            if (fieldDescriptor.getSchema().isOptional() || fieldDescriptor.getSchema().defaultValue() != null) continue;
            throw new SQLException(String.format("Cannot ALTER table '%s' because field '%s' is not optional but has no default value", tableName, fieldDescriptor.getName()));
        }
        if (JdbcSinkConnectorConfig.SchemaEvolutionMode.NONE.equals((Object)this.config.getSchemaEvolutionMode())) {
            LOGGER.warn("Table '{}' cannot be altered because schema evolution is disabled.", (Object)tableName);
            throw new SQLException("Cannot alter table " + tableName + " because schema evolution is disabled");
        }
        Transaction transaction = this.session.beginTransaction();
        try {
            String alterSql = this.dialect.getAlterTableStatement(table, record, missingFields);
            LOGGER.trace("SQL: {}", (Object)alterSql);
            this.session.createNativeQuery(alterSql, Object.class).executeUpdate();
            transaction.commit();
        }
        catch (Exception e) {
            transaction.rollback();
            throw e;
        }
        return this.readTable(tableName);
    }

    private void write(TableDescriptor table, SinkRecordDescriptor record) throws SQLException {
        if (!record.isDelete()) {
            switch (this.config.getInsertMode()) {
                case INSERT: {
                    this.writeInsert(this.dialect.getInsertStatement(table, record), record);
                    break;
                }
                case UPSERT: {
                    if (record.getKeyFieldNames().isEmpty()) {
                        throw new ConnectException("Cannot write to table " + table.getId().getTableName() + " with no key fields defined.");
                    }
                    this.writeUpsert(this.dialect.getUpsertStatement(table, record), record);
                    break;
                }
                case UPDATE: {
                    this.writeUpdate(this.dialect.getUpdateStatement(table, record), record);
                }
            }
        } else {
            this.writeDelete(this.dialect.getDeleteStatement(table, record), record);
        }
    }

    private void writeInsert(String sql, SinkRecordDescriptor record) throws SQLException {
        Transaction transaction = this.session.beginTransaction();
        try {
            LOGGER.trace("SQL: {}", (Object)sql);
            NativeQuery query = this.session.createNativeQuery(sql, Object.class);
            int index = this.bindKeyValuesToQuery(record, query, 1);
            this.bindNonKeyValuesToQuery(record, query, index);
            int result = query.executeUpdate();
            if (result != 1) {
                throw new SQLException("Failed to insert row from table");
            }
            transaction.commit();
        }
        catch (SQLException e) {
            transaction.rollback();
            throw e;
        }
    }

    private void writeUpsert(String sql, SinkRecordDescriptor record) throws SQLException {
        Transaction transaction = this.session.beginTransaction();
        try {
            LOGGER.trace("SQL: {}", (Object)sql);
            NativeQuery query = this.session.createNativeQuery(sql, Object.class);
            int index = this.bindKeyValuesToQuery(record, query, 1);
            this.bindNonKeyValuesToQuery(record, query, index);
            query.executeUpdate();
            transaction.commit();
        }
        catch (Exception e) {
            transaction.rollback();
            throw e;
        }
    }

    private void writeUpdate(String sql, SinkRecordDescriptor record) throws SQLException {
        Transaction transaction = this.session.beginTransaction();
        try {
            LOGGER.trace("SQL: {}", (Object)sql);
            NativeQuery query = this.session.createNativeQuery(sql, Object.class);
            int index = this.bindNonKeyValuesToQuery(record, query, 1);
            this.bindKeyValuesToQuery(record, query, index);
            query.executeUpdate();
            transaction.commit();
        }
        catch (Exception e) {
            transaction.rollback();
            throw e;
        }
    }

    private void writeDelete(String sql, SinkRecordDescriptor record) throws SQLException {
        if (!this.config.isDeleteEnabled()) {
            LOGGER.debug("Deletes are not enabled, skipping delete for topic '{}'", (Object)record.getTopicName());
            return;
        }
        Transaction transaction = this.session.beginTransaction();
        try {
            LOGGER.trace("SQL: {}", (Object)sql);
            NativeQuery query = this.session.createNativeQuery(sql, Object.class);
            this.bindKeyValuesToQuery(record, query, 1);
            query.executeUpdate();
            transaction.commit();
        }
        catch (Exception e) {
            transaction.rollback();
            throw e;
        }
    }

    private int bindKeyValuesToQuery(SinkRecordDescriptor record, NativeQuery<?> query, int index) {
        switch (this.config.getPrimaryKeyMode()) {
            case KAFKA: {
                query.setParameter(index++, (Object)record.getTopicName());
                query.setParameter(index++, (Object)record.getPartition());
                query.setParameter(index++, (Object)record.getOffset());
                break;
            }
            case RECORD_KEY: 
            case RECORD_VALUE: {
                Struct keySource = record.getKeyStruct(this.config.getPrimaryKeyMode());
                if (keySource == null) break;
                index = this.bindFieldValuesToQuery(record, query, index, keySource, record.getKeyFieldNames());
            }
        }
        return index;
    }

    private int bindNonKeyValuesToQuery(SinkRecordDescriptor record, NativeQuery<?> query, int index) {
        return this.bindFieldValuesToQuery(record, query, index, record.getAfterStruct(), record.getNonKeyFieldNames());
    }

    private int bindFieldValuesToQuery(SinkRecordDescriptor record, NativeQuery<?> query, int index, Struct source, List<String> fields) {
        for (String fieldName : fields) {
            SinkRecordDescriptor.FieldDescriptor field = record.getFields().get(fieldName);
            index += this.dialect.bindValue(field, query, index, source.get(fieldName));
        }
        return index;
    }
}

