/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.repair;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.AbstractFuture;
import com.google.common.util.concurrent.AsyncFunction;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.concurrent.Executor;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.repair.LocalSyncTask;
import org.apache.cassandra.repair.NodePair;
import org.apache.cassandra.repair.RemoteSyncTask;
import org.apache.cassandra.repair.RepairJobDesc;
import org.apache.cassandra.repair.RepairParallelism;
import org.apache.cassandra.repair.RepairResult;
import org.apache.cassandra.repair.RepairSession;
import org.apache.cassandra.repair.SnapshotTask;
import org.apache.cassandra.repair.SyncStat;
import org.apache.cassandra.repair.SyncTask;
import org.apache.cassandra.repair.SystemDistributedKeyspace;
import org.apache.cassandra.repair.TreeResponse;
import org.apache.cassandra.repair.ValidationTask;
import org.apache.cassandra.tracing.Tracing;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.MerkleTrees;
import org.apache.cassandra.utils.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RepairJob
extends AbstractFuture<RepairResult>
implements Runnable {
    private static Logger logger = LoggerFactory.getLogger(RepairJob.class);
    private final RepairSession session;
    private final RepairJobDesc desc;
    private final RepairParallelism parallelismDegree;
    private final long repairedAt;
    private final ListeningExecutorService taskExecutor;

    public RepairJob(RepairSession session, String columnFamily) {
        this.session = session;
        this.desc = new RepairJobDesc(session.parentRepairSession, session.getId(), session.keyspace, columnFamily, session.getRanges());
        this.repairedAt = session.repairedAt;
        this.taskExecutor = session.taskExecutor;
        this.parallelismDegree = session.parallelismDegree;
    }

    @Override
    public void run() {
        ListenableFuture validations;
        ArrayList<InetAddress> allEndpoints = new ArrayList<InetAddress>(this.session.endpoints);
        allEndpoints.add(FBUtilities.getBroadcastAddress());
        if (this.parallelismDegree != RepairParallelism.PARALLEL) {
            ArrayList<SnapshotTask> snapshotTasks = new ArrayList<SnapshotTask>(allEndpoints.size());
            for (InetAddress endpoint : allEndpoints) {
                SnapshotTask snapshotTask = new SnapshotTask(this.desc, endpoint);
                snapshotTasks.add(snapshotTask);
                this.taskExecutor.execute((Runnable)snapshotTask);
            }
            ListenableFuture allSnapshotTasks = Futures.allAsList(snapshotTasks);
            validations = Futures.transform((ListenableFuture)allSnapshotTasks, (AsyncFunction)new AsyncFunction<List<InetAddress>, List<TreeResponse>>(){

                public ListenableFuture<List<TreeResponse>> apply(List<InetAddress> endpoints) {
                    if (RepairJob.this.parallelismDegree == RepairParallelism.SEQUENTIAL) {
                        return RepairJob.this.sendSequentialValidationRequest(endpoints);
                    }
                    return RepairJob.this.sendDCAwareValidationRequest(endpoints);
                }
            }, (Executor)this.taskExecutor);
        } else {
            validations = this.sendValidationRequest(allEndpoints);
        }
        ListenableFuture syncResults = Futures.transform(validations, (AsyncFunction)new AsyncFunction<List<TreeResponse>, List<SyncStat>>(){

            public ListenableFuture<List<SyncStat>> apply(List<TreeResponse> trees) {
                return Futures.allAsList(RepairJob.this.createSyncTasks(trees, FBUtilities.getLocalAddress()));
            }
        }, (Executor)this.taskExecutor);
        Futures.addCallback((ListenableFuture)syncResults, (FutureCallback)new FutureCallback<List<SyncStat>>(){

            public void onSuccess(List<SyncStat> stats) {
                logger.info("[repair #{}] {} is fully synced", (Object)RepairJob.this.session.getId(), (Object)((RepairJob)RepairJob.this).desc.columnFamily);
                SystemDistributedKeyspace.successfulRepairJob(RepairJob.this.session.getId(), ((RepairJob)RepairJob.this).desc.keyspace, ((RepairJob)RepairJob.this).desc.columnFamily);
                RepairJob.this.set(new RepairResult(RepairJob.this.desc, stats));
            }

            public void onFailure(Throwable t) {
                logger.warn("[repair #{}] {} sync failed", (Object)RepairJob.this.session.getId(), (Object)((RepairJob)RepairJob.this).desc.columnFamily);
                SystemDistributedKeyspace.failedRepairJob(RepairJob.this.session.getId(), ((RepairJob)RepairJob.this).desc.keyspace, ((RepairJob)RepairJob.this).desc.columnFamily, t);
                RepairJob.this.setException(t);
            }
        }, (Executor)this.taskExecutor);
        Futures.getUnchecked(validations);
    }

    @VisibleForTesting
    List<SyncTask> createSyncTasks(List<TreeResponse> trees, InetAddress local) {
        ArrayList<SyncTask> syncTasks = new ArrayList<SyncTask>();
        for (int i = 0; i < trees.size() - 1; ++i) {
            TreeResponse r1 = trees.get(i);
            for (int j = i + 1; j < trees.size(); ++j) {
                SyncTask task;
                TreeResponse r2 = trees.get(j);
                List<Range<Token>> differences = MerkleTrees.difference(r1.trees, r2.trees);
                if (r1.endpoint.equals(local) || r2.endpoint.equals(local)) {
                    task = new LocalSyncTask(this.desc, r1.endpoint, r2.endpoint, differences, this.repairedAt, this.session.pullRepair);
                } else {
                    task = new RemoteSyncTask(this.desc, r1.endpoint, r2.endpoint, differences);
                    this.session.waitForSync(Pair.create(this.desc, new NodePair(r1.endpoint, r2.endpoint)), (RemoteSyncTask)task);
                }
                syncTasks.add(task);
                this.taskExecutor.submit((Runnable)task);
            }
        }
        return syncTasks;
    }

    private ListenableFuture<List<TreeResponse>> sendValidationRequest(Collection<InetAddress> endpoints) {
        String message = String.format("Requesting merkle trees for %s (to %s)", this.desc.columnFamily, endpoints);
        logger.info("[repair #{}] {}", (Object)this.desc.sessionId, (Object)message);
        Tracing.traceRepair(message, new Object[0]);
        int gcBefore = Keyspace.open(this.desc.keyspace).getColumnFamilyStore(this.desc.columnFamily).gcBefore(FBUtilities.nowInSeconds());
        ArrayList<ValidationTask> tasks = new ArrayList<ValidationTask>(endpoints.size());
        for (InetAddress endpoint : endpoints) {
            ValidationTask task = new ValidationTask(this.desc, endpoint, gcBefore);
            tasks.add(task);
            this.session.waitForValidation(Pair.create(this.desc, endpoint), task);
            this.taskExecutor.execute((Runnable)task);
        }
        return Futures.allAsList(tasks);
    }

    private ListenableFuture<List<TreeResponse>> sendSequentialValidationRequest(Collection<InetAddress> endpoints) {
        String message = String.format("Requesting merkle trees for %s (to %s)", this.desc.columnFamily, endpoints);
        logger.info("[repair #{}] {}", (Object)this.desc.sessionId, (Object)message);
        Tracing.traceRepair(message, new Object[0]);
        int gcBefore = Keyspace.open(this.desc.keyspace).getColumnFamilyStore(this.desc.columnFamily).gcBefore(FBUtilities.nowInSeconds());
        ArrayList<ValidationTask> tasks = new ArrayList<ValidationTask>(endpoints.size());
        LinkedList<InetAddress> requests = new LinkedList<InetAddress>(endpoints);
        InetAddress address = (InetAddress)requests.poll();
        ValidationTask firstTask = new ValidationTask(this.desc, address, gcBefore);
        logger.info("Validating {}", (Object)address);
        this.session.waitForValidation(Pair.create(this.desc, address), firstTask);
        tasks.add(firstTask);
        ValidationTask currentTask = firstTask;
        while (requests.size() > 0) {
            final InetAddress nextAddress = (InetAddress)requests.poll();
            final ValidationTask nextTask = new ValidationTask(this.desc, nextAddress, gcBefore);
            tasks.add(nextTask);
            Futures.addCallback((ListenableFuture)currentTask, (FutureCallback)new FutureCallback<TreeResponse>(){

                public void onSuccess(TreeResponse result) {
                    logger.info("Validating {}", (Object)nextAddress);
                    RepairJob.this.session.waitForValidation(Pair.create(RepairJob.this.desc, nextAddress), nextTask);
                    RepairJob.this.taskExecutor.execute((Runnable)nextTask);
                }

                public void onFailure(Throwable t) {
                }
            });
            currentTask = nextTask;
        }
        this.taskExecutor.execute((Runnable)firstTask);
        return Futures.allAsList(tasks);
    }

    private ListenableFuture<List<TreeResponse>> sendDCAwareValidationRequest(Collection<InetAddress> endpoints) {
        String message = String.format("Requesting merkle trees for %s (to %s)", this.desc.columnFamily, endpoints);
        logger.info("[repair #{}] {}", (Object)this.desc.sessionId, (Object)message);
        Tracing.traceRepair(message, new Object[0]);
        int gcBefore = Keyspace.open(this.desc.keyspace).getColumnFamilyStore(this.desc.columnFamily).gcBefore(FBUtilities.nowInSeconds());
        ArrayList<ValidationTask> tasks = new ArrayList<ValidationTask>(endpoints.size());
        HashMap<String, LinkedList<InetAddress>> requestsByDatacenter = new HashMap<String, LinkedList<InetAddress>>();
        for (InetAddress inetAddress : endpoints) {
            String dc = DatabaseDescriptor.getEndpointSnitch().getDatacenter(inetAddress);
            LinkedList<InetAddress> queue = (LinkedList<InetAddress>)requestsByDatacenter.get(dc);
            if (queue == null) {
                queue = new LinkedList<InetAddress>();
                requestsByDatacenter.put(dc, queue);
            }
            queue.add(inetAddress);
        }
        for (Map.Entry entry : requestsByDatacenter.entrySet()) {
            Queue requests = (Queue)entry.getValue();
            InetAddress address = (InetAddress)requests.poll();
            ValidationTask firstTask = new ValidationTask(this.desc, address, gcBefore);
            logger.info("Validating {}", (Object)address);
            this.session.waitForValidation(Pair.create(this.desc, address), firstTask);
            tasks.add(firstTask);
            ValidationTask currentTask = firstTask;
            while (requests.size() > 0) {
                final InetAddress nextAddress = (InetAddress)requests.poll();
                final ValidationTask nextTask = new ValidationTask(this.desc, nextAddress, gcBefore);
                tasks.add(nextTask);
                Futures.addCallback((ListenableFuture)currentTask, (FutureCallback)new FutureCallback<TreeResponse>(){

                    public void onSuccess(TreeResponse result) {
                        logger.info("Validating {}", (Object)nextAddress);
                        RepairJob.this.session.waitForValidation(Pair.create(RepairJob.this.desc, nextAddress), nextTask);
                        RepairJob.this.taskExecutor.execute((Runnable)nextTask);
                    }

                    public void onFailure(Throwable t) {
                    }
                });
                currentTask = nextTask;
            }
            this.taskExecutor.execute((Runnable)firstTask);
        }
        return Futures.allAsList(tasks);
    }
}

