/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.connector.cassandra;

import io.debezium.connector.cassandra.AbstractSourceConnector;
import io.debezium.connector.cassandra.CassandraConnectorConfig;
import io.debezium.connector.cassandra.OffsetFlushPolicy;
import io.debezium.connector.cassandra.OffsetPosition;
import io.debezium.connector.cassandra.OffsetWriter;
import io.debezium.connector.cassandra.exceptions.CassandraConnectorConfigException;
import io.debezium.connector.cassandra.exceptions.CassandraConnectorTaskException;
import io.debezium.util.Threads;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.channels.OverlappingFileLockException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.Properties;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileOffsetWriter
implements OffsetWriter {
    private static final Logger LOGGER = LoggerFactory.getLogger(FileOffsetWriter.class);
    public static final String SNAPSHOT_OFFSET_FILE = "snapshot_offset.properties";
    public static final String COMMITLOG_OFFSET_FILE = "commitlog_offset.properties";
    private final Properties snapshotProps = new Properties();
    private final Properties commitLogProps = new Properties();
    private final File snapshotOffsetFile;
    private final File commitLogOffsetFile;
    private final FileLock snapshotOffsetFileLock;
    private final FileLock commitLogOffsetFileLock;
    private final OffsetFlushPolicy offsetFlushPolicy;
    private final ExecutorService executorService;

    public FileOffsetWriter(CassandraConnectorConfig config) throws IOException {
        if (config.offsetBackingStoreDir() == null) {
            throw new CassandraConnectorConfigException("Offset file directory must be configured at the start");
        }
        this.offsetFlushPolicy = config.offsetFlushIntervalMs().isZero() ? OffsetFlushPolicy.always() : OffsetFlushPolicy.periodic(config.offsetFlushIntervalMs(), config.maxOffsetFlushSize());
        File offsetDirectory = new File(config.offsetBackingStoreDir());
        if (!offsetDirectory.exists()) {
            Files.createDirectories(offsetDirectory.toPath(), new FileAttribute[0]);
        }
        this.snapshotOffsetFile = Paths.get(offsetDirectory.getAbsolutePath(), SNAPSHOT_OFFSET_FILE).toFile();
        this.commitLogOffsetFile = Paths.get(offsetDirectory.getAbsolutePath(), COMMITLOG_OFFSET_FILE).toFile();
        this.snapshotOffsetFileLock = this.init(this.snapshotOffsetFile);
        this.commitLogOffsetFileLock = this.init(this.commitLogOffsetFile);
        this.loadOffset(this.snapshotOffsetFile, this.snapshotProps);
        this.loadOffset(this.commitLogOffsetFile, this.commitLogProps);
        this.executorService = Threads.newSingleThreadExecutor(AbstractSourceConnector.class, (String)config.getConnectorName(), (String)"offset-writer");
    }

    @Override
    public Future<?> markOffset(String sourceTable, String sourceOffset, boolean isSnapshot) {
        return this.executorService.submit(() -> this.performMarkOffset(sourceTable, sourceOffset, isSnapshot));
    }

    private void performMarkOffset(String sourceTable, String sourceOffset, boolean isSnapshot) {
        if (isSnapshot) {
            if (!this.isOffsetProcessed(sourceTable, sourceOffset, isSnapshot)) {
                this.snapshotProps.setProperty(sourceTable, sourceOffset);
            }
        } else if (!this.isOffsetProcessed(sourceTable, sourceOffset, isSnapshot)) {
            this.commitLogProps.setProperty(sourceTable, sourceOffset);
        }
        if (this.offsetFlushPolicy.shouldFlush()) {
            this.performFlush();
        }
    }

    @Override
    public boolean isOffsetProcessed(String sourceTable, String sourceOffset, boolean isSnapshot) {
        if (isSnapshot) {
            return this.snapshotProps.containsKey(sourceTable);
        }
        OffsetPosition currentOffset = OffsetPosition.parse(sourceOffset);
        OffsetPosition recordedOffset = this.commitLogProps.containsKey(sourceTable) ? OffsetPosition.parse((String)this.commitLogProps.get(sourceTable)) : null;
        return recordedOffset != null && currentOffset.compareTo(recordedOffset) <= 0;
    }

    @Override
    public Future<?> flush() {
        return this.executorService.submit(this::performFlush);
    }

    private void performFlush() {
        try {
            FileOffsetWriter.saveOffset(this.snapshotOffsetFile, this.snapshotProps);
            FileOffsetWriter.saveOffset(this.commitLogOffsetFile, this.commitLogProps);
        }
        catch (IOException e) {
            LOGGER.warn("Ignoring flush failure", (Throwable)e);
        }
    }

    @Override
    public void close() {
        try {
            if (!this.executorService.awaitTermination(1L, TimeUnit.SECONDS)) {
                this.executorService.shutdown();
            }
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        try {
            this.snapshotOffsetFileLock.release();
        }
        catch (IOException e) {
            LOGGER.warn("Failed to release snapshot offset file lock");
        }
        try {
            this.commitLogOffsetFileLock.release();
        }
        catch (IOException e) {
            LOGGER.warn("Failed to release commit log offset file lock");
        }
    }

    private static void saveOffset(File offsetFile, Properties props) throws IOException {
        try (FileOutputStream fos = new FileOutputStream(offsetFile);){
            props.store(fos, null);
        }
        catch (IOException e) {
            throw new IOException("Failed to save offset for file " + offsetFile.getAbsolutePath(), e);
        }
    }

    private void loadOffset(File offsetFile, Properties props) throws IOException {
        try (FileInputStream fis = new FileInputStream(offsetFile);){
            props.load(fis);
        }
        catch (IOException e) {
            throw new IOException("Failed to load offset for file " + offsetFile.getAbsolutePath(), e);
        }
    }

    private FileLock init(File offsetFile) throws IOException {
        Path lockPath = this.initLockPath(offsetFile);
        try {
            FileChannel channel = FileChannel.open(lockPath, StandardOpenOption.READ, StandardOpenOption.WRITE);
            FileLock lock = channel.tryLock();
            if (lock == null) {
                throw new CassandraConnectorTaskException("Failed to acquire file lock on " + String.valueOf(lockPath) + ". There might be another Cassandra Connector Task running");
            }
            return lock;
        }
        catch (OverlappingFileLockException e) {
            throw new CassandraConnectorTaskException("Failed to acquire file lock on " + String.valueOf(lockPath) + ". There might be another thread running", e);
        }
    }

    private Path initLockPath(File offsetFile) throws IOException {
        if (!offsetFile.exists()) {
            Files.createFile(offsetFile.toPath(), new FileAttribute[0]);
        }
        if (System.getProperty("os.name").toLowerCase().contains("windows") && !(offsetFile = new File(offsetFile.getAbsolutePath() + ".lock")).exists()) {
            Files.createFile(offsetFile.toPath(), new FileAttribute[0]);
        }
        return offsetFile.toPath();
    }
}

