/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.repair;

import com.codahale.metrics.Timer;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.util.concurrent.AsyncFunction;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.cassandra.concurrent.JMXEnabledThreadPoolExecutor;
import org.apache.cassandra.concurrent.NamedThreadFactory;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.cql3.QueryOptions;
import org.apache.cassandra.cql3.QueryProcessor;
import org.apache.cassandra.cql3.UntypedResultSet;
import org.apache.cassandra.cql3.statements.SelectStatement;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.ConsistencyLevel;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.gms.FailureDetector;
import org.apache.cassandra.locator.EndpointsForRange;
import org.apache.cassandra.locator.InetAddressAndPort;
import org.apache.cassandra.locator.Replica;
import org.apache.cassandra.metrics.RepairMetrics;
import org.apache.cassandra.metrics.StorageMetrics;
import org.apache.cassandra.repair.CommonRange;
import org.apache.cassandra.repair.RepairResult;
import org.apache.cassandra.repair.RepairSession;
import org.apache.cassandra.repair.RepairSessionResult;
import org.apache.cassandra.repair.SomeRepairFailedException;
import org.apache.cassandra.repair.SyncStat;
import org.apache.cassandra.repair.SystemDistributedKeyspace;
import org.apache.cassandra.repair.consistent.CoordinatorSession;
import org.apache.cassandra.repair.consistent.SyncStatSummary;
import org.apache.cassandra.repair.messages.RepairOption;
import org.apache.cassandra.service.ActiveRepairService;
import org.apache.cassandra.service.ClientState;
import org.apache.cassandra.service.QueryState;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.streaming.PreviewKind;
import org.apache.cassandra.tracing.TraceState;
import org.apache.cassandra.tracing.Tracing;
import org.apache.cassandra.transport.messages.ResultMessage;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.DiagnosticSnapshotService;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.UUIDGen;
import org.apache.cassandra.utils.WrappedRunnable;
import org.apache.cassandra.utils.progress.ProgressEvent;
import org.apache.cassandra.utils.progress.ProgressEventNotifier;
import org.apache.cassandra.utils.progress.ProgressEventType;
import org.apache.cassandra.utils.progress.ProgressListener;
import org.apache.commons.lang3.time.DurationFormatUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RepairRunnable
implements Runnable,
ProgressEventNotifier {
    private static final Logger logger = LoggerFactory.getLogger(RepairRunnable.class);
    private final StorageService storageService;
    private final int cmd;
    private final RepairOption options;
    private final String keyspace;
    private final String tag;
    private final AtomicInteger progressCounter = new AtomicInteger();
    private final int totalProgress;
    private final long creationTimeMillis = System.currentTimeMillis();
    private final UUID parentSession = UUIDGen.getTimeUUID();
    private final List<ProgressListener> listeners = new ArrayList<ProgressListener>();
    private static final AtomicInteger threadCounter = new AtomicInteger(1);
    private final AtomicReference<Throwable> firstError = new AtomicReference<Object>(null);
    private TraceState traceState;

    public RepairRunnable(StorageService storageService, int cmd, RepairOption options, String keyspace) {
        this.storageService = storageService;
        this.cmd = cmd;
        this.options = options;
        this.keyspace = keyspace;
        this.tag = "repair:" + cmd;
        this.totalProgress = 4 + options.getRanges().size();
    }

    @Override
    public void addProgressListener(ProgressListener listener) {
        this.listeners.add(listener);
    }

    @Override
    public void removeProgressListener(ProgressListener listener) {
        this.listeners.remove(listener);
    }

    protected void fireProgressEvent(ProgressEvent event) {
        for (ProgressListener listener : this.listeners) {
            listener.progress(this.tag, event);
        }
    }

    public void notification(String msg) {
        logger.info(msg);
        this.fireProgressEvent(new ProgressEvent(ProgressEventType.NOTIFICATION, this.progressCounter.get(), this.totalProgress, msg));
    }

    private void skip(String msg) {
        this.notification("Repair " + this.parentSession + " skipped: " + msg);
        this.success(msg);
    }

    private void success(String msg) {
        this.fireProgressEvent(new ProgressEvent(ProgressEventType.SUCCESS, this.progressCounter.get(), this.totalProgress, msg));
        ActiveRepairService.instance.recordRepairStatus(this.cmd, ActiveRepairService.ParentRepairStatus.COMPLETED, (List<String>)ImmutableList.of((Object)msg));
        this.complete(null);
    }

    public void notifyError(Throwable error) {
        if (error instanceof SomeRepairFailedException) {
            return;
        }
        logger.error("Repair {} failed:", (Object)this.parentSession, (Object)error);
        StorageMetrics.repairExceptions.inc();
        String errorMessage = String.format("Repair command #%d failed with error %s", this.cmd, error.getMessage());
        this.fireProgressEvent(new ProgressEvent(ProgressEventType.ERROR, this.progressCounter.get(), this.totalProgress, errorMessage));
        this.firstError.compareAndSet(null, error);
        this.maybeStoreParentRepairFailure(error);
    }

    private void fail(String reason) {
        if (reason == null) {
            Throwable error = this.firstError.get();
            reason = error != null ? error.getMessage() : "Some repair failed";
        }
        String completionMessage = String.format("Repair command #%d finished with error", this.cmd);
        ActiveRepairService.instance.recordRepairStatus(this.cmd, ActiveRepairService.ParentRepairStatus.FAILED, (List<String>)ImmutableList.of((Object)reason, (Object)completionMessage));
        this.complete(completionMessage);
    }

    private void complete(String msg) {
        long durationMillis = System.currentTimeMillis() - this.creationTimeMillis;
        if (msg == null) {
            String duration = DurationFormatUtils.formatDurationWords((long)durationMillis, (boolean)true, (boolean)true);
            msg = String.format("Repair command #%d finished in %s", this.cmd, duration);
        }
        this.fireProgressEvent(new ProgressEvent(ProgressEventType.COMPLETE, this.progressCounter.get(), this.totalProgress, msg));
        logger.info(this.options.getPreviewKind().logPrefix(this.parentSession) + msg);
        ActiveRepairService.instance.removeParentRepairSession(this.parentSession);
        TraceState localState = this.traceState;
        if (this.options.isTraced() && localState != null) {
            for (ProgressListener listener : this.listeners) {
                localState.removeProgressListener(listener);
            }
            Tracing.instance.set(localState);
            Tracing.traceRepair(msg, new Object[0]);
            Tracing.instance.stopSession();
        }
        Keyspace.open((String)this.keyspace).metric.repairTime.update(durationMillis, TimeUnit.MILLISECONDS);
    }

    @Override
    public void run() {
        try {
            this.runMayThrow();
        }
        catch (SkipRepairException e) {
            this.skip(e.getMessage());
        }
        catch (Error | Exception e) {
            this.notifyError(e);
            this.fail(e.getMessage());
        }
    }

    private void runMayThrow() throws Exception {
        ActiveRepairService.instance.recordRepairStatus(this.cmd, ActiveRepairService.ParentRepairStatus.IN_PROGRESS, (List<String>)ImmutableList.of());
        List<ColumnFamilyStore> columnFamilies = this.getColumnFamilies();
        String[] cfnames = (String[])columnFamilies.stream().map(cfs -> cfs.name).toArray(String[]::new);
        this.traceState = this.maybeCreateTraceState(columnFamilies);
        this.notifyStarting();
        NeighborsAndRanges neighborsAndRanges = this.getNeighborsAndRanges();
        this.maybeStoreParentRepairStart(cfnames);
        this.prepare(columnFamilies, neighborsAndRanges.participants, neighborsAndRanges.shouldExcludeDeadParticipants);
        this.repair(cfnames, neighborsAndRanges);
    }

    private List<ColumnFamilyStore> getColumnFamilies() throws IOException {
        String[] columnFamilies = this.options.getColumnFamilies().toArray(new String[this.options.getColumnFamilies().size()]);
        Iterable<ColumnFamilyStore> validColumnFamilies = this.storageService.getValidColumnFamilies(false, false, this.keyspace, columnFamilies);
        this.progressCounter.incrementAndGet();
        if (Iterables.isEmpty(validColumnFamilies)) {
            throw new SkipRepairException(String.format("%s Empty keyspace, skipping repair: %s", this.parentSession, this.keyspace));
        }
        return Lists.newArrayList(validColumnFamilies);
    }

    private TraceState maybeCreateTraceState(Iterable<ColumnFamilyStore> columnFamilyStores) {
        if (!this.options.isTraced()) {
            return null;
        }
        StringBuilder cfsb = new StringBuilder();
        for (ColumnFamilyStore cfs : columnFamilyStores) {
            cfsb.append(", ").append(cfs.keyspace.getName()).append(".").append(cfs.name);
        }
        UUID sessionId = Tracing.instance.newSession(Tracing.TraceType.REPAIR);
        TraceState traceState = Tracing.instance.begin("repair", (Map<String, String>)ImmutableMap.of((Object)"keyspace", (Object)this.keyspace, (Object)"columnFamilies", (Object)cfsb.substring(2)));
        traceState.enableActivityNotification(this.tag);
        for (ProgressListener listener : this.listeners) {
            traceState.addProgressListener(listener);
        }
        Thread queryThread = this.createQueryThread(this.cmd, sessionId);
        queryThread.setName("RepairTracePolling");
        queryThread.start();
        return traceState;
    }

    private void notifyStarting() {
        String message = String.format("Starting repair command #%d (%s), repairing keyspace %s with %s", this.cmd, this.parentSession, this.keyspace, this.options);
        logger.info(message);
        Tracing.traceRepair(message, new Object[0]);
        this.fireProgressEvent(new ProgressEvent(ProgressEventType.START, 0, 100, message));
    }

    private NeighborsAndRanges getNeighborsAndRanges() {
        HashSet allNeighbors = new HashSet();
        ArrayList<CommonRange> commonRanges = new ArrayList<CommonRange>();
        Set<Range<Token>> keyspaceLocalRanges = this.storageService.getLocalReplicas(this.keyspace).ranges();
        for (Range<Token> range : this.options.getRanges()) {
            EndpointsForRange neighbors = ActiveRepairService.getNeighbors(this.keyspace, keyspaceLocalRanges, range, this.options.getDataCenters(), this.options.getHosts());
            if (neighbors.isEmpty()) {
                if (this.options.ignoreUnreplicatedKeyspaces()) {
                    logger.info("{} Found no neighbors for range {} for {} - ignoring since repairing with --ignore-unreplicated-keyspaces", new Object[]{this.parentSession, range, this.keyspace});
                    continue;
                }
                throw new RuntimeException(String.format("Nothing to repair for %s in %s - aborting", range, this.keyspace));
            }
            RepairRunnable.addRangeToNeighbors(commonRanges, range, neighbors);
            allNeighbors.addAll(neighbors.endpoints());
        }
        if (this.options.ignoreUnreplicatedKeyspaces() && allNeighbors.isEmpty()) {
            throw new SkipRepairException(String.format("Nothing to repair for %s in %s - unreplicated keyspace is ignored since repair was called with --ignore-unreplicated-keyspaces", this.options.getRanges(), this.keyspace));
        }
        this.progressCounter.incrementAndGet();
        boolean shouldExcludeDeadParticipants = this.options.isForcedRepair();
        if (shouldExcludeDeadParticipants) {
            HashSet actualNeighbors = Sets.newHashSet((Iterable)Iterables.filter(allNeighbors, FailureDetector.instance::isAlive));
            shouldExcludeDeadParticipants = !allNeighbors.equals(actualNeighbors);
            allNeighbors = actualNeighbors;
        }
        return new NeighborsAndRanges(shouldExcludeDeadParticipants, allNeighbors, commonRanges);
    }

    private void maybeStoreParentRepairStart(String[] cfnames) {
        if (!this.options.isPreview()) {
            SystemDistributedKeyspace.startParentRepair(this.parentSession, this.keyspace, cfnames, this.options);
        }
    }

    private void maybeStoreParentRepairSuccess(Collection<Range<Token>> successfulRanges) {
        if (!this.options.isPreview()) {
            SystemDistributedKeyspace.successfulParentRepair(this.parentSession, successfulRanges);
        }
    }

    private void maybeStoreParentRepairFailure(Throwable error) {
        if (!this.options.isPreview()) {
            SystemDistributedKeyspace.failParentRepair(this.parentSession, error);
        }
    }

    private void prepare(List<ColumnFamilyStore> columnFamilies, Set<InetAddressAndPort> allNeighbors, boolean force) {
        try (Timer.Context ignore = Keyspace.open((String)this.keyspace).metric.repairPrepareTime.time();){
            ActiveRepairService.instance.prepareForRepair(this.parentSession, FBUtilities.getBroadcastAddressAndPort(), allNeighbors, this.options, force, columnFamilies);
            this.progressCounter.incrementAndGet();
        }
    }

    private void repair(String[] cfnames, NeighborsAndRanges neighborsAndRanges) {
        if (this.options.isPreview()) {
            this.previewRepair(this.parentSession, this.creationTimeMillis, neighborsAndRanges.filterCommonRanges(this.keyspace, cfnames), neighborsAndRanges.participants, cfnames);
        } else if (this.options.isIncremental()) {
            this.incrementalRepair(this.parentSession, this.creationTimeMillis, this.traceState, neighborsAndRanges, neighborsAndRanges.participants, cfnames);
        } else {
            this.normalRepair(this.parentSession, this.creationTimeMillis, this.traceState, neighborsAndRanges.filterCommonRanges(this.keyspace, cfnames), neighborsAndRanges.participants, cfnames);
        }
    }

    private void normalRepair(UUID parentSession, long startTime, TraceState traceState, List<CommonRange> commonRanges, Set<InetAddressAndPort> preparedEndpoints, String ... cfnames) {
        ListeningExecutorService executor = this.createExecutor();
        ListenableFuture<List<RepairSessionResult>> allSessions = this.submitRepairSessions(parentSession, false, executor, commonRanges, cfnames);
        final ArrayList<Range<Token>> successfulRanges = new ArrayList<Range<Token>>();
        final AtomicBoolean hasFailure = new AtomicBoolean();
        ListenableFuture repairResult = Futures.transformAsync(allSessions, (AsyncFunction)new AsyncFunction<List<RepairSessionResult>, Object>(){

            public ListenableFuture apply(List<RepairSessionResult> results) {
                logger.debug("Repair result: {}", results);
                for (RepairSessionResult sessionResult : results) {
                    if (sessionResult != null) {
                        if (sessionResult.skippedReplicas) continue;
                        successfulRanges.addAll(sessionResult.ranges);
                        continue;
                    }
                    hasFailure.compareAndSet(false, true);
                }
                return Futures.immediateFuture(null);
            }
        }, (Executor)MoreExecutors.directExecutor());
        Futures.addCallback((ListenableFuture)repairResult, (FutureCallback)new RepairCompleteCallback(parentSession, successfulRanges, preparedEndpoints, startTime, traceState, hasFailure, (ExecutorService)executor), (Executor)MoreExecutors.directExecutor());
    }

    private void incrementalRepair(UUID parentSession, long startTime, TraceState traceState, NeighborsAndRanges neighborsAndRanges, Set<InetAddressAndPort> preparedEndpoints, String ... cfnames) {
        ImmutableSet allParticipants = ImmutableSet.builder().addAll((Iterable)neighborsAndRanges.participants).add((Object)FBUtilities.getBroadcastAddressAndPort()).build();
        List<CommonRange> allRanges = neighborsAndRanges.filterCommonRanges(this.keyspace, cfnames);
        CoordinatorSession coordinatorSession = ActiveRepairService.instance.consistent.coordinated.registerSession(parentSession, (Set<InetAddressAndPort>)allParticipants, neighborsAndRanges.shouldExcludeDeadParticipants);
        ListeningExecutorService executor = this.createExecutor();
        AtomicBoolean hasFailure = new AtomicBoolean(false);
        ListenableFuture repairResult = coordinatorSession.execute(() -> this.submitRepairSessions(parentSession, true, executor, allRanges, cfnames), hasFailure);
        HashSet<Range<Token>> ranges = new HashSet<Range<Token>>();
        for (Collection range : Iterables.transform(allRanges, cr -> cr.ranges)) {
            ranges.addAll(range);
        }
        Futures.addCallback((ListenableFuture)repairResult, (FutureCallback)new RepairCompleteCallback(parentSession, ranges, preparedEndpoints, startTime, traceState, hasFailure, (ExecutorService)executor), (Executor)MoreExecutors.directExecutor());
    }

    private void previewRepair(final UUID parentSession, long startTime, List<CommonRange> commonRanges, final Set<InetAddressAndPort> preparedEndpoints, String ... cfnames) {
        logger.debug("Starting preview repair for {}", (Object)parentSession);
        final ListeningExecutorService executor = this.createExecutor();
        ListenableFuture<List<RepairSessionResult>> allSessions = this.submitRepairSessions(parentSession, false, executor, commonRanges, cfnames);
        Futures.addCallback(allSessions, (FutureCallback)new FutureCallback<List<RepairSessionResult>>(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            public void onSuccess(List<RepairSessionResult> results) {
                try {
                    String message;
                    if (results == null || results.stream().anyMatch(s -> s == null)) {
                        RepairRunnable.this.fail(null);
                        return;
                    }
                    PreviewKind previewKind = RepairRunnable.this.options.getPreviewKind();
                    Preconditions.checkState((previewKind != PreviewKind.NONE ? 1 : 0) != 0, (Object)"Preview is NONE");
                    SyncStatSummary summary = new SyncStatSummary(true);
                    summary.consumeSessionResults(results);
                    if (summary.isEmpty()) {
                        message = previewKind == PreviewKind.REPAIRED ? "Repaired data is in sync" : "Previewed data was in sync";
                    } else {
                        message = (previewKind == PreviewKind.REPAIRED ? "Repaired data is inconsistent\n" : "Preview complete\n") + summary.toString();
                        RepairMetrics.previewFailures.inc();
                        if (previewKind == PreviewKind.REPAIRED) {
                            RepairRunnable.this.maybeSnapshotReplicas(parentSession, RepairRunnable.this.keyspace, results);
                        }
                    }
                    RepairRunnable.this.notification(message);
                    RepairRunnable.this.success("Repair preview completed successfully");
                    ActiveRepairService.instance.cleanUp(parentSession, preparedEndpoints);
                }
                catch (Throwable t) {
                    logger.error("Error completing preview repair", t);
                    this.onFailure(t);
                }
                finally {
                    executor.shutdownNow();
                }
            }

            public void onFailure(Throwable t) {
                RepairRunnable.this.notifyError(t);
                RepairRunnable.this.fail("Error completing preview repair: " + t.getMessage());
                executor.shutdownNow();
            }
        }, (Executor)MoreExecutors.directExecutor());
    }

    private void maybeSnapshotReplicas(UUID parentSession, String keyspace, List<RepairSessionResult> results) {
        if (!DatabaseDescriptor.snapshotOnRepairedDataMismatch()) {
            return;
        }
        try {
            HashSet<String> mismatchingTables = new HashSet<String>();
            HashSet<InetAddressAndPort> nodes = new HashSet<InetAddressAndPort>();
            for (RepairSessionResult sessionResult : results) {
                for (RepairResult repairResult : RepairRunnable.emptyIfNull(sessionResult.repairJobResults)) {
                    for (SyncStat stat : RepairRunnable.emptyIfNull(repairResult.stats)) {
                        if (stat.numberOfDifferences > 0L) {
                            mismatchingTables.add(repairResult.desc.columnFamily);
                        }
                        nodes.add(stat.nodes.coordinator);
                        nodes.add(stat.nodes.peer);
                    }
                }
            }
            String snapshotName = DiagnosticSnapshotService.getSnapshotName("RepairedDataMismatch-");
            for (String table : mismatchingTables) {
                if (!Keyspace.open(keyspace).getColumnFamilyStore(table).snapshotExists(snapshotName)) {
                    logger.info("{} Snapshotting {}.{} for preview repair mismatch with tag {} on instances {}", new Object[]{this.options.getPreviewKind().logPrefix(parentSession), keyspace, table, snapshotName, nodes});
                    DiagnosticSnapshotService.repairedDataMismatch(Keyspace.open(keyspace).getColumnFamilyStore(table).metadata(), nodes);
                    continue;
                }
                logger.info("{} Not snapshotting {}.{} - snapshot {} exists", new Object[]{this.options.getPreviewKind().logPrefix(parentSession), keyspace, table, snapshotName});
            }
        }
        catch (Exception e) {
            logger.error("{} Failed snapshotting replicas", (Object)this.options.getPreviewKind().logPrefix(parentSession), (Object)e);
        }
    }

    private static <T> Iterable<T> emptyIfNull(Iterable<T> iter) {
        if (iter == null) {
            return Collections.emptyList();
        }
        return iter;
    }

    private ListenableFuture<List<RepairSessionResult>> submitRepairSessions(UUID parentSession, boolean isIncremental, ListeningExecutorService executor, List<CommonRange> commonRanges, String ... cfnames) {
        ArrayList<RepairSession> futures = new ArrayList<RepairSession>(this.options.getRanges().size());
        for (CommonRange commonRange : commonRanges) {
            logger.info("Starting RepairSession for {}", (Object)commonRange);
            RepairSession session = ActiveRepairService.instance.submitRepairSession(parentSession, commonRange, this.keyspace, this.options.getParallelism(), isIncremental, this.options.isPullRepair(), this.options.getPreviewKind(), this.options.optimiseStreams(), executor, cfnames);
            if (session == null) continue;
            Futures.addCallback((ListenableFuture)session, (FutureCallback)new RepairSessionCallback(session), (Executor)MoreExecutors.directExecutor());
            futures.add(session);
        }
        return Futures.successfulAsList(futures);
    }

    private ListeningExecutorService createExecutor() {
        return MoreExecutors.listeningDecorator((ExecutorService)new JMXEnabledThreadPoolExecutor(this.options.getJobThreads(), Integer.MAX_VALUE, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(), new NamedThreadFactory("Repair#" + this.cmd), "internal"));
    }

    private static void addRangeToNeighbors(List<CommonRange> neighborRangeList, Range<Token> range, EndpointsForRange neighbors) {
        Set<InetAddressAndPort> endpoints = neighbors.endpoints();
        Set<InetAddressAndPort> transEndpoints = ((EndpointsForRange)neighbors.filter(Replica::isTransient)).endpoints();
        for (CommonRange commonRange : neighborRangeList) {
            if (!commonRange.matchesEndpoints(endpoints, transEndpoints)) continue;
            commonRange.ranges.add(range);
            return;
        }
        ArrayList<Range<Token>> ranges = new ArrayList<Range<Token>>();
        ranges.add(range);
        neighborRangeList.add(new CommonRange(endpoints, transEndpoints, ranges));
    }

    private Thread createQueryThread(int cmd, final UUID sessionId) {
        return NamedThreadFactory.createThread(new WrappedRunnable(){

            @Override
            public void runMayThrow() throws Exception {
                TraceState.Status status;
                TraceState state = Tracing.instance.get(sessionId);
                if (state == null) {
                    throw new Exception("no tracestate");
                }
                String format = "select event_id, source, source_port, activity from %s.%s where session_id = ? and event_id > ? and event_id < ?;";
                String query = String.format(format, "system_traces", "events");
                SelectStatement statement = (SelectStatement)QueryProcessor.parseStatement(query).prepare(ClientState.forInternalCalls());
                ByteBuffer sessionIdBytes = ByteBufferUtil.bytes(sessionId);
                InetAddressAndPort source = FBUtilities.getBroadcastAddressAndPort();
                HashSet[] seen = new HashSet[]{new HashSet(), new HashSet()};
                int si = 0;
                long tlast = System.currentTimeMillis();
                long minWaitMillis = 125L;
                long maxWaitMillis = 1024000L;
                long timeout = minWaitMillis;
                boolean shouldDouble = false;
                while ((status = state.waitActivity(timeout)) != TraceState.Status.STOPPED) {
                    if (status == TraceState.Status.IDLE) {
                        timeout = shouldDouble ? Math.min(timeout * 2L, maxWaitMillis) : timeout;
                        shouldDouble = !shouldDouble;
                    } else {
                        timeout = minWaitMillis;
                        shouldDouble = false;
                    }
                    ByteBuffer tminBytes = ByteBufferUtil.bytes(UUIDGen.minTimeUUID(tlast - 1000L));
                    long tcur = System.currentTimeMillis();
                    ByteBuffer tmaxBytes = ByteBufferUtil.bytes(UUIDGen.maxTimeUUID(tcur));
                    QueryOptions options = QueryOptions.forInternalCalls(ConsistencyLevel.ONE, Lists.newArrayList((Object[])new ByteBuffer[]{sessionIdBytes, tminBytes, tmaxBytes}));
                    ResultMessage.Rows rows = statement.execute(QueryState.forInternalCalls(), options, System.nanoTime());
                    UntypedResultSet result = UntypedResultSet.create(rows.result);
                    for (UntypedResultSet.Row r : result) {
                        InetAddressAndPort eventNode;
                        int port = DatabaseDescriptor.getStoragePort();
                        if (r.has("source_port")) {
                            port = r.getInt("source_port");
                        }
                        if (source.equals(eventNode = InetAddressAndPort.getByAddressOverrideDefaults(r.getInetAddress("source"), port))) continue;
                        UUID uuid = r.getUUID("event_id");
                        if (uuid.timestamp() > (tcur - 1000L) * 10000L) {
                            seen[si].add(uuid);
                        }
                        if (seen[si == 0 ? 1 : 0].contains(uuid)) continue;
                        String message = String.format("%s: %s", r.getInetAddress("source"), r.getString("activity"));
                        RepairRunnable.this.notification(message);
                    }
                    tlast = tcur;
                    si = si == 0 ? 1 : 0;
                    seen[si].clear();
                }
            }
        }, "Repair-Runnable-" + threadCounter.incrementAndGet());
    }

    static final class NeighborsAndRanges {
        private final boolean shouldExcludeDeadParticipants;
        private final Set<InetAddressAndPort> participants;
        private final List<CommonRange> commonRanges;

        NeighborsAndRanges(boolean shouldExcludeDeadParticipants, Set<InetAddressAndPort> participants, List<CommonRange> commonRanges) {
            this.shouldExcludeDeadParticipants = shouldExcludeDeadParticipants;
            this.participants = participants;
            this.commonRanges = commonRanges;
        }

        List<CommonRange> filterCommonRanges(String keyspace, String[] tableNames) {
            if (!this.shouldExcludeDeadParticipants) {
                return this.commonRanges;
            }
            logger.debug("force flag set, removing dead endpoints if possible");
            ArrayList<CommonRange> filtered = new ArrayList<CommonRange>(this.commonRanges.size());
            for (CommonRange commonRange : this.commonRanges) {
                ImmutableSet endpoints = ImmutableSet.copyOf((Iterable)Iterables.filter(commonRange.endpoints, this.participants::contains));
                ImmutableSet transEndpoints = ImmutableSet.copyOf((Iterable)Iterables.filter(commonRange.transEndpoints, this.participants::contains));
                Preconditions.checkState((boolean)endpoints.containsAll((Collection<?>)transEndpoints), (Object)"transEndpoints must be a subset of endpoints");
                if (!endpoints.isEmpty()) {
                    Sets.SetView skippedReplicas = Sets.difference(commonRange.endpoints, (Set)endpoints);
                    skippedReplicas.forEach(endpoint -> logger.info("Removing a dead node {} from repair for ranges {} due to -force", endpoint, commonRange.ranges));
                    filtered.add(new CommonRange((Set<InetAddressAndPort>)endpoints, (Set<InetAddressAndPort>)transEndpoints, commonRange.ranges, !skippedReplicas.isEmpty()));
                    continue;
                }
                logger.warn("Skipping forced repair for ranges {} of tables {} in keyspace {}, as no neighbor nodes are live.", new Object[]{commonRange.ranges, Arrays.asList(tableNames), keyspace});
            }
            Preconditions.checkState((!filtered.isEmpty() ? 1 : 0) != 0, (Object)"Not enough live endpoints for a repair");
            return filtered;
        }
    }

    private static final class SkipRepairException
    extends RuntimeException {
        SkipRepairException(String message) {
            super(message);
        }
    }

    private class RepairCompleteCallback
    implements FutureCallback<Object> {
        final UUID parentSession;
        final Collection<Range<Token>> successfulRanges;
        final Set<InetAddressAndPort> preparedEndpoints;
        final long startTime;
        final TraceState traceState;
        final AtomicBoolean hasFailure;
        final ExecutorService executor;

        public RepairCompleteCallback(UUID parentSession, Collection<Range<Token>> successfulRanges, Set<InetAddressAndPort> preparedEndpoints, long startTime, TraceState traceState, AtomicBoolean hasFailure, ExecutorService executor) {
            this.parentSession = parentSession;
            this.successfulRanges = successfulRanges;
            this.preparedEndpoints = preparedEndpoints;
            this.startTime = startTime;
            this.traceState = traceState;
            this.hasFailure = hasFailure;
            this.executor = executor;
        }

        public void onSuccess(Object result) {
            RepairRunnable.this.maybeStoreParentRepairSuccess(this.successfulRanges);
            if (this.hasFailure.get()) {
                RepairRunnable.this.fail(null);
            } else {
                RepairRunnable.this.success("Repair completed successfully");
                ActiveRepairService.instance.cleanUp(this.parentSession, this.preparedEndpoints);
            }
            this.executor.shutdownNow();
        }

        public void onFailure(Throwable t) {
            RepairRunnable.this.notifyError(t);
            RepairRunnable.this.fail(t.getMessage());
            this.executor.shutdownNow();
        }
    }

    private class RepairSessionCallback
    implements FutureCallback<RepairSessionResult> {
        private final RepairSession session;

        public RepairSessionCallback(RepairSession session) {
            this.session = session;
        }

        public void onSuccess(RepairSessionResult result) {
            String message = String.format("Repair session %s for range %s finished", this.session.getId(), this.session.ranges().toString());
            logger.info(message);
            RepairRunnable.this.fireProgressEvent(new ProgressEvent(ProgressEventType.PROGRESS, RepairRunnable.this.progressCounter.incrementAndGet(), RepairRunnable.this.totalProgress, message));
        }

        public void onFailure(Throwable t) {
            String message = String.format("Repair session %s for range %s failed with error %s", this.session.getId(), this.session.ranges().toString(), t.getMessage());
            RepairRunnable.this.notifyError(new RuntimeException(message, t));
        }
    }
}

