/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.connector.cassandra;

import com.datastax.oss.driver.api.core.ConsistencyLevel;
import com.datastax.oss.driver.api.core.CqlIdentifier;
import com.datastax.oss.driver.api.core.DefaultConsistencyLevel;
import com.datastax.oss.driver.api.core.cql.ResultSet;
import com.datastax.oss.driver.api.core.cql.Row;
import com.datastax.oss.driver.api.core.cql.SimpleStatement;
import com.datastax.oss.driver.api.core.metadata.schema.ColumnMetadata;
import com.datastax.oss.driver.api.core.metadata.schema.TableMetadata;
import com.datastax.oss.driver.api.core.type.DataTypes;
import com.datastax.oss.driver.api.querybuilder.QueryBuilder;
import com.datastax.oss.driver.api.querybuilder.select.Select;
import com.datastax.oss.driver.api.querybuilder.select.SelectFrom;
import io.debezium.DebeziumException;
import io.debezium.connector.base.ChangeEventQueue;
import io.debezium.connector.cassandra.AbstractProcessor;
import io.debezium.connector.cassandra.CassandraClient;
import io.debezium.connector.cassandra.CassandraConnectorConfig;
import io.debezium.connector.cassandra.CassandraConnectorContext;
import io.debezium.connector.cassandra.CassandraSchemaFactory;
import io.debezium.connector.cassandra.Event;
import io.debezium.connector.cassandra.Filters;
import io.debezium.connector.cassandra.KeyValueSchema;
import io.debezium.connector.cassandra.KeyspaceTable;
import io.debezium.connector.cassandra.OffsetPosition;
import io.debezium.connector.cassandra.OffsetWriter;
import io.debezium.connector.cassandra.Record;
import io.debezium.connector.cassandra.RecordMaker;
import io.debezium.connector.cassandra.SchemaHolder;
import io.debezium.connector.cassandra.SnapshotProcessorMetrics;
import io.debezium.connector.cassandra.transforms.CassandraTypeDeserializer;
import io.debezium.function.BlockingConsumer;
import io.debezium.time.Conversions;
import io.debezium.util.Collect;
import java.io.IOException;
import java.time.Duration;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.kafka.connect.data.Schema;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SnapshotProcessor
extends AbstractProcessor {
    private static final Logger LOGGER = LoggerFactory.getLogger(SnapshotProcessor.class);
    private static final String NAME = "Snapshot Processor";
    private static final String CASSANDRA_NOW_UNIXTIMESTAMP = "TOUNIXTIMESTAMP(NOW())";
    private static final String EXECUTION_TIME_ALIAS = "execution_time";
    private static final Set<Integer> collectionTypes = Collect.unmodifiableSet((Object[])new Integer[]{32, 34, 33});
    private static final CassandraSchemaFactory schemaFactory = CassandraSchemaFactory.get();
    private final CassandraClient cassandraClient;
    private final List<ChangeEventQueue<Event>> queues;
    private final OffsetWriter offsetWriter;
    private final SchemaHolder schemaHolder;
    private final RecordMaker recordMaker;
    private final CassandraConnectorConfig.SnapshotMode snapshotMode;
    private final ConsistencyLevel consistencyLevel;
    private final Set<String> startedTableNames = new HashSet<String>();
    private final SnapshotProcessorMetrics metrics = new SnapshotProcessorMetrics();
    private boolean initial = true;
    private final String clusterName;

    public SnapshotProcessor(CassandraConnectorContext context, String clusterName) {
        super(NAME, context.getCassandraConnectorConfig().snapshotPollInterval());
        this.queues = context.getQueues();
        this.cassandraClient = context.getCassandraClient();
        this.offsetWriter = context.getOffsetWriter();
        this.schemaHolder = context.getSchemaHolder();
        this.recordMaker = new RecordMaker(context.getCassandraConnectorConfig().tombstonesOnDelete(), new Filters(context.getCassandraConnectorConfig().fieldExcludeList()), context.getCassandraConnectorConfig());
        this.snapshotMode = context.getCassandraConnectorConfig().snapshotMode();
        this.consistencyLevel = context.getCassandraConnectorConfig().snapshotConsistencyLevel();
        this.clusterName = clusterName;
    }

    @Override
    public void initialize() {
        this.metrics.registerMetrics();
    }

    @Override
    public void destroy() {
        this.metrics.unregisterMetrics();
    }

    @Override
    public void process() throws IOException {
        if (this.snapshotMode == CassandraConnectorConfig.SnapshotMode.ALWAYS) {
            this.snapshot();
        } else if (this.snapshotMode == CassandraConnectorConfig.SnapshotMode.INITIAL && this.initial) {
            this.snapshot();
            this.initial = false;
        } else {
            LOGGER.debug("Skipping snapshot [mode: {}]", (Object)this.snapshotMode);
        }
    }

    public synchronized void snapshot() throws IOException {
        Set<TableMetadata> tables = this.getTablesToSnapshot();
        if (!tables.isEmpty()) {
            String[] tableArr = (String[])tables.stream().map(SnapshotProcessor::tableName).toArray(String[]::new);
            LOGGER.debug("Found {} tables to snapshot: {}", (Object)tables.size(), (Object)tableArr);
            long startTime = System.currentTimeMillis();
            this.metrics.setTableCount(tables.size());
            this.metrics.startSnapshot();
            for (TableMetadata table : tables) {
                if (!this.isRunning()) continue;
                String tableName = SnapshotProcessor.tableName(table);
                LOGGER.info("Snapshotting table {} ...", (Object)tableName);
                this.startedTableNames.add(tableName);
                this.takeTableSnapshot(table);
                this.metrics.completeTable();
                LOGGER.info("Snapshot of table {} has been taken", (Object)tableName);
            }
            this.metrics.stopSnapshot();
            long endTime = System.currentTimeMillis();
            long durationInSeconds = Duration.ofMillis(endTime - startTime).getSeconds();
            LOGGER.debug("Snapshot completely queued in {} seconds for tables: {}", (Object)durationInSeconds, (Object)tableArr);
        } else {
            LOGGER.info("No table to snapshot");
        }
    }

    private Set<TableMetadata> getTablesToSnapshot() {
        LOGGER.info("Present tables: {}", this.schemaHolder.getCdcEnabledTableMetadataSet().stream().map(tmd -> tmd.describe(true)).collect(Collectors.toList()));
        return this.schemaHolder.getCdcEnabledTableMetadataSet().stream().filter(tm -> !this.offsetWriter.isOffsetProcessed(SnapshotProcessor.tableName(tm), OffsetPosition.defaultOffsetPosition().serialize(), true)).filter(tm -> !this.startedTableNames.contains(SnapshotProcessor.tableName(tm))).collect(Collectors.toSet());
    }

    private void takeTableSnapshot(TableMetadata tableMetadata) throws IOException {
        try {
            SimpleStatement statement = (SimpleStatement)SnapshotProcessor.generateSnapshotStatement(tableMetadata).setConsistencyLevel((ConsistencyLevel)DefaultConsistencyLevel.valueOf((String)this.consistencyLevel.name()));
            LOGGER.info("Executing snapshot query '{}' with consistency level {}", (Object)statement.getQuery(), (Object)statement.getConsistencyLevel());
            ResultSet resultSet = this.cassandraClient.execute(statement);
            LOGGER.info("Executed snapshot query for table {}", (Object)SnapshotProcessor.tableName(tableMetadata));
            this.processResultSet(tableMetadata, resultSet);
        }
        catch (IOException e) {
            throw e;
        }
        catch (Exception e) {
            throw new DebeziumException(String.format("Failed to snapshot table %s in keyspace %s", tableMetadata.getName(), tableMetadata.getKeyspace()), (Throwable)e);
        }
    }

    private static SimpleStatement generateSnapshotStatement(TableMetadata tableMetadata) {
        List allCols = tableMetadata.getColumns().values().stream().map(cmd -> cmd.getName().asInternal()).collect(Collectors.toList());
        Set primaryCols = tableMetadata.getPrimaryKey().stream().map(cmd -> cmd.getName().asInternal()).collect(Collectors.toSet());
        List collectionCols = tableMetadata.getColumns().values().stream().filter(cm -> collectionTypes.contains(cm.getType().getProtocolCode())).map(cmd -> cmd.getName().asInternal()).collect(Collectors.toList());
        SelectFrom selection = QueryBuilder.selectFrom((CqlIdentifier)tableMetadata.getKeyspace(), (CqlIdentifier)tableMetadata.getName());
        assert (!allCols.isEmpty());
        Select select = null;
        for (String col : allCols) {
            select = select == null ? selection.column(col) : select.column(col);
            if (primaryCols.contains(col) || collectionCols.contains(col)) continue;
            select = select.ttl(SnapshotProcessor.withQuotes(col)).as(SnapshotProcessor.ttlAlias(col));
        }
        return select.raw(CASSANDRA_NOW_UNIXTIMESTAMP).as(EXECUTION_TIME_ALIAS).build();
    }

    private void processResultSet(TableMetadata tableMetadata, ResultSet resultSet) throws IOException {
        String tableName = SnapshotProcessor.tableName(tableMetadata);
        KeyspaceTable keyspaceTable = new KeyspaceTable(tableMetadata);
        KeyValueSchema keyValueSchema = this.schemaHolder.getKeyValueSchema(keyspaceTable);
        Schema keySchema = keyValueSchema.keySchema();
        Schema valueSchema = keyValueSchema.valueSchema();
        Set<String> partitionKeyNames = tableMetadata.getPartitionKey().stream().map(cmd -> cmd.getName().toString()).collect(Collectors.toSet());
        Set<String> clusteringKeyNames = tableMetadata.getClusteringColumns().keySet().stream().map(cc -> cc.getName().toString()).collect(Collectors.toSet());
        Iterator rowIter = resultSet.iterator();
        long rowNum = 0L;
        if (!rowIter.hasNext()) {
            this.offsetWriter.markOffset(tableName, OffsetPosition.defaultOffsetPosition().serialize(), true);
            this.offsetWriter.flush();
        }
        while (rowIter.hasNext()) {
            if (this.isRunning()) {
                Row row = (Row)rowIter.next();
                Object executionTime = SnapshotProcessor.readExecutionTime(row);
                CassandraSchemaFactory.RowData after = SnapshotProcessor.extractRowData(row, tableMetadata.getColumns().values(), partitionKeyNames, clusteringKeyNames, executionTime);
                boolean markOffset = !rowIter.hasNext();
                this.recordMaker.insert(this.clusterName, OffsetPosition.defaultOffsetPosition(), keyspaceTable, true, Conversions.toInstantFromMicros((long)TimeUnit.MICROSECONDS.convert((Long)executionTime, TimeUnit.MILLISECONDS)), after, keySchema, valueSchema, markOffset, (BlockingConsumer<Record>)((BlockingConsumer)arg_0 -> this.queues.get(Math.abs(tableName.hashCode() % this.queues.size())).enqueue(arg_0)));
                if (++rowNum % 10000L != 0L) continue;
                LOGGER.debug("Queued {} snapshot records from table {}", (Object)rowNum, (Object)tableName);
                this.metrics.setRowsScanned(tableName, rowNum);
                continue;
            }
            LOGGER.warn("Terminated snapshot processing while table {} is in progress", (Object)tableName);
            this.metrics.setRowsScanned(tableName, rowNum);
            return;
        }
        this.metrics.setRowsScanned(tableName, rowNum);
    }

    private static CassandraSchemaFactory.RowData extractRowData(Row row, Collection<ColumnMetadata> columns, Set<String> partitionKeyNames, Set<String> clusteringKeyNames, Object executionTime) {
        CassandraSchemaFactory.RowData rowData = schemaFactory.rowData();
        for (ColumnMetadata columnMetadata : columns) {
            Object ttl;
            String name = columnMetadata.getName().asInternal();
            Object value = SnapshotProcessor.readCol(row, name, columnMetadata);
            Long deletionTs = null;
            CassandraSchemaFactory.CellData.ColumnType type = SnapshotProcessor.getType(name, partitionKeyNames, clusteringKeyNames);
            if (type == CassandraSchemaFactory.CellData.ColumnType.REGULAR && value != null && !collectionTypes.contains(columnMetadata.getType().getProtocolCode()) && (ttl = SnapshotProcessor.readColTtl(row, name)) != null && executionTime != null) {
                deletionTs = SnapshotProcessor.calculateDeletionTs(executionTime, ttl);
            }
            CassandraSchemaFactory.CellData cellData = schemaFactory.cellData(name, value, deletionTs, type);
            rowData.addCell(cellData);
        }
        return rowData;
    }

    private static CassandraSchemaFactory.CellData.ColumnType getType(String name, Set<String> partitionKeyNames, Set<String> clusteringKeyNames) {
        if (partitionKeyNames.contains(name)) {
            return CassandraSchemaFactory.CellData.ColumnType.PARTITION;
        }
        if (clusteringKeyNames.contains(name)) {
            return CassandraSchemaFactory.CellData.ColumnType.CLUSTERING;
        }
        return CassandraSchemaFactory.CellData.ColumnType.REGULAR;
    }

    private static Object readExecutionTime(Row row) {
        return CassandraTypeDeserializer.deserialize(DataTypes.BIGINT, row.getBytesUnsafe(EXECUTION_TIME_ALIAS));
    }

    private static Object readCol(Row row, String col, ColumnMetadata cm) {
        return CassandraTypeDeserializer.deserialize(cm.getType(), row.getBytesUnsafe(col));
    }

    private static Object readColTtl(Row row, String col) {
        if (row.getColumnDefinitions().contains(CqlIdentifier.fromInternal((String)SnapshotProcessor.ttlAlias(col)))) {
            return CassandraTypeDeserializer.deserialize(DataTypes.COUNTER, row.getBytesUnsafe(SnapshotProcessor.ttlAlias(col)));
        }
        return null;
    }

    private static long calculateDeletionTs(Object executionTime, Object ttl) {
        return TimeUnit.MICROSECONDS.convert((Long)executionTime, TimeUnit.MILLISECONDS) + TimeUnit.MICROSECONDS.convert(((Integer)ttl).intValue(), TimeUnit.SECONDS);
    }

    private static String ttlAlias(String colName) {
        return colName + "_ttl";
    }

    private static String withQuotes(String s) {
        return "\"" + s + "\"";
    }

    private static String tableName(TableMetadata tm) {
        return tm.getKeyspace() + "." + tm.getName();
    }
}

