/*
 * Decompiled with CFR 0.152.
 */
package io.debezium.connector.cassandra;

import io.debezium.connector.cassandra.exceptions.CassandraConnectorDataException;
import java.io.File;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class CommitLogUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(CommitLogUtil.class);
    private static final Pattern FILENAME_REGEX_PATTERN = Pattern.compile("CommitLog-\\d+-(\\d+).log");
    private static final Pattern FILENAME_INDEX_REGEX_PATTERN = Pattern.compile("CommitLog-\\d+-(\\d+)_cdc.idx");

    private CommitLogUtil() {
    }

    public static void moveCommitLog(Path file, Path toDir) {
        try {
            Matcher filenameMatcher = FILENAME_REGEX_PATTERN.matcher(file.getFileName().toString());
            if (!filenameMatcher.matches()) {
                LOGGER.warn("Cannot move file {} because it does not appear to be a CommitLog.", (Object)file.toAbsolutePath());
                return;
            }
            Files.move(file, toDir.resolve(file.getFileName()), StandardCopyOption.REPLACE_EXISTING);
            LOGGER.info("Moved CommitLog file {} to {}.", (Object)file, (Object)toDir);
        }
        catch (Exception ex) {
            LOGGER.warn("Failed to move CommitLog file {} to {}. Error:", new Object[]{file.getFileName().toString(), toDir, ex});
            throw new RuntimeException(ex);
        }
        Path indexFile = file.getParent().resolve(file.getFileName().toString().split("\\.")[0] + "_cdc.idx");
        try {
            if (Files.exists(indexFile, new LinkOption[0])) {
                Files.move(indexFile, toDir.resolve(indexFile.getFileName()), StandardCopyOption.REPLACE_EXISTING);
                LOGGER.info("Moved CommitLog index file {} to {}.", (Object)indexFile, (Object)toDir);
            }
        }
        catch (Exception ex) {
            LOGGER.warn("Failed to move CommitLog index file {} to {}. Error:", new Object[]{indexFile.toAbsolutePath(), toDir, ex});
            throw new RuntimeException(ex);
        }
    }

    public static void deleteCommitLog(File file) {
        try {
            Matcher filenameMatcher = FILENAME_REGEX_PATTERN.matcher(file.getName());
            if (!filenameMatcher.matches()) {
                LOGGER.warn("Cannot delete file {} because it does not appear to be a CommitLog", (Object)file.getName());
            }
            Files.delete(file.toPath());
            LOGGER.info("Deleted CommitLog file {} from {}.", (Object)file.getName(), (Object)file.getParent());
        }
        catch (Exception e) {
            LOGGER.warn("Failed to delete CommitLog file {} from {}. Error: ", new Object[]{file.getName(), file.getParent(), e});
            throw new RuntimeException(e);
        }
        Path indexFile = Paths.get(file.toString().split("\\.")[0] + "_cdc.idx", new String[0]);
        try {
            if (Files.exists(indexFile, new LinkOption[0])) {
                Files.delete(indexFile);
                LOGGER.info("Deleted CommitLog index file {} from {}.", (Object)indexFile.getFileName(), (Object)indexFile.getParent());
            }
        }
        catch (Exception ex) {
            LOGGER.warn("Failed to delete CommitLog index file {}. Error:", (Object)indexFile.toAbsolutePath(), (Object)ex);
        }
    }

    public static File[] getCommitLogs(File directory) {
        if (!directory.isDirectory()) {
            throw new IllegalArgumentException("Given directory does not exist: " + directory);
        }
        return directory.listFiles(f -> f.isFile() && FILENAME_REGEX_PATTERN.matcher(f.getName()).matches());
    }

    public static File[] getIndexes(File directory) {
        if (!directory.isDirectory()) {
            throw new IllegalArgumentException("Given directory does not exist: " + directory);
        }
        return directory.listFiles(f -> f.isFile() && FILENAME_INDEX_REGEX_PATTERN.matcher(f.getName()).matches());
    }

    public static int compareCommitLogs(File file1, File file2) {
        if (file1.equals(file2)) {
            return 0;
        }
        long ts1 = CommitLogUtil.extractTimestamp(file1.getName(), FILENAME_REGEX_PATTERN);
        long ts2 = CommitLogUtil.extractTimestamp(file2.getName(), FILENAME_REGEX_PATTERN);
        return Long.compare(ts1, ts2);
    }

    public static int compareCommitLogs(String file1, String file2) {
        if (file1.equals(file2)) {
            return 0;
        }
        long ts1 = CommitLogUtil.extractTimestamp(file1, FILENAME_REGEX_PATTERN);
        long ts2 = CommitLogUtil.extractTimestamp(file2, FILENAME_REGEX_PATTERN);
        return Long.compare(ts1, ts2);
    }

    public static int compareCommitLogsIndexes(File file1, File file2) {
        if (file1.equals(file2)) {
            return 0;
        }
        long ts1 = CommitLogUtil.extractTimestamp(file1.getName(), FILENAME_INDEX_REGEX_PATTERN);
        long ts2 = CommitLogUtil.extractTimestamp(file2.getName(), FILENAME_INDEX_REGEX_PATTERN);
        return Long.compare(ts1, ts2);
    }

    private static long extractTimestamp(String commitLogFileName, Pattern pattern) {
        Matcher filenameMatcher = pattern.matcher(commitLogFileName);
        if (!filenameMatcher.matches()) {
            throw new CassandraConnectorDataException("Cannot extract timestamp because " + commitLogFileName + " does not appear to be a CommitLog");
        }
        return Long.parseLong(filenameMatcher.group(1));
    }
}

