package io.datalbry.jira.cloud.client.feign.board

import feign.FeignException
import io.datalbry.jira.cloud.client.BoardClient
import io.datalbry.jira.cloud.client.exception.UnsupportedBoardTypeException
import io.datalbry.jira.cloud.client.feign.iterator.PaginatedIterator
import io.datalbry.jira.cloud.client.model.*

class FeignBoardClient(private val feignBoardResource: FeignBoardResource) : BoardClient {

    override fun getBoardKeys(pageSize: Int): Iterator<BoardKey> {
        val page = feignBoardResource.getBoardKeys(start = 0, limit = pageSize)
        return PaginatedIterator(page) { feignBoardResource.getBoardKeys(pageSize, it.getStart() + it.getMaxResults()) }
    }

    override fun getBoard(boardKey: BoardKey): Board {
        return feignBoardResource.getBoard(boardKey.id)
    }

    override fun getConfiguration(boardKey: BoardKey): Configuration {
        return feignBoardResource.getConfiguration(boardKey.id)
    }

    override fun getIssueKeys(boardKey: BoardKey, pageSize: Int): Iterator<IssueKey> {
        val page = feignBoardResource.getIssueKeys(boardKey.id, start = 0, limit = pageSize)
        return PaginatedIterator(page) {
            feignBoardResource.getIssueKeys(
                boardKey.id, pageSize, it.getStart() + it.getMaxResults()
            )
        }
    }

    override fun getIssue(issueKey: IssueKey): Issue {
        return feignBoardResource.getIssue(issueKey.id)
    }

    override fun getFeatures(boardKey: BoardKey): Features {
        return try {
            feignBoardResource.getFeatures(boardKey.id)
        } catch (e: FeignException.BadRequest) {
            throw UnsupportedBoardTypeException(boardKey)
        }
    }

    override fun getQuickfilterKeys(boardKey: BoardKey, pageSize: Int): Iterator<QuickfilterKey> {
        val page = feignBoardResource.getQuickfilterKeys(boardKey.id, start = 0, limit = pageSize)
        return PaginatedIterator(page) {
            feignBoardResource.getQuickfilterKeys(
                boardKey.id, pageSize, it.getStart() + it.getMaxResults()
            )
        }
    }

    override fun getQuickfilter(quickfilterKey: QuickfilterKey): Quickfilter {
        return feignBoardResource.getQuickfilter(quickfilterKey.boardId, quickfilterKey.id)
    }

    override fun getVersions(boardKey: BoardKey, pageSize: Int): Iterator<Version> {
        val page = feignBoardResource.getVersions(boardKey.id, start = 0, limit = pageSize)
        return PaginatedIterator(page) {
            feignBoardResource.getVersions(
                boardKey.id, pageSize, it.getStart() + it.getMaxResults()
            )
        }
    }

    override fun getReports(boardKey: BoardKey): Reports {
        try {
            return feignBoardResource.getReports(boardKey.id)
        } catch (e: FeignException.BadRequest) {
            throw UnsupportedBoardTypeException(boardKey)
        }
    }

    override fun getSprintKeys(boardKey: BoardKey, pageSize: Int): Iterator<SprintKey> {
        try {
            val page = feignBoardResource.getSprintsKeys(boardKey.id, start = 0, limit = pageSize)
            return PaginatedIterator(page) {
                feignBoardResource.getSprintsKeys(
                    boardKey.id, pageSize, it.getStart() + it.getMaxResults()
                )
            }
        } catch (e: FeignException.BadRequest) {
            throw UnsupportedBoardTypeException(boardKey)
        }
    }

    override fun getSprint(sprintKey: SprintKey): Sprint {
        return feignBoardResource.getSprint(sprintKey.id)
    }

    override fun getProjects(boardKey: BoardKey, pageSize: Int): Iterator<Project> {
        val page = feignBoardResource.getProjects(boardKey.id, start = 0, limit = pageSize)
        return PaginatedIterator(page) {
            feignBoardResource.getProjects(
                boardKey.id, pageSize, it.getStart() + it.getMaxResults()
            )
        }
    }


}
