package io.datalbry.alxndria.client.api

import io.datalbry.alxndria.client.api.pagination.Paginated
import io.datalbry.precise.api.schema.document.Document

/**
 * Client implementation for the Index REST endpoints of the knowledge integration platform
 *
 * @author timo gruen - 2020-10-10
 */
interface IndexClient: AutoCloseable {

    /**
     * Put a document to the index
     *
     * @param datasourceId of the datasource which should be accessed
     * @param document to put
     *
     */
    fun putDocument(datasourceId: String, document: Document)

    /**
     * Delete a document from the index
     *
     * @param datasourceId of the datasource which should be accessed
     * @param documentId of the document which should be deleted
     */
    fun deleteDocument(datasourceId: String, documentId: String)

    /**
     * Get a [Document] by its id
     *
     * @param datasourceId to get the [Document] from
     * @param documentId of the [Document]
     *
     * @return the [Document]
     *
     * @throws NoSuchElementException if the [Document] isn't present in the Datasource
     */
    fun getDocument(datasourceId: String, documentId: String): Document

    /**
     * Get all document ids
     *
     * The implementation isn't enforced to iterator lazy using the [getDocumentIdPage] functions.
     *
     * @param datasourceId of the datasource to get the documentIds from
     *
     * @return iterator of all documentIds present in the DatasourceId
     */
    fun getDocumentIds(datasourceId: String, pageSize: Int = DEFAULT_PAGE_SIZE): Iterator<String>

    /**
     * Get a [Paginated] of document ids
     *
     * Returns the first page of document ids. If there are any further Pages, the [Paginated]
     * result contains a [Paginated]#nextToken which can be used to fetch the next page.
     *
     * NOTE: The nextToken is deterministic but the ids are not sorted by insertion.
     * Adding documents while retrieving the pages might result into skipping some documents
     *
     * If one wants to check if a document has been deleted or if it just was skipped use [getDocument]
     *
     * @param datasourceId of the datasource to get the documentIds from
     * @param pageSize of the [Paginated]#payload size
     *
     * @return [Paginated] collection of document ids
     */
    fun getDocumentIdPage(datasourceId: String, pageSize: Int = DEFAULT_PAGE_SIZE): Paginated<String>

    /**
     * Get a [Paginated] of document ids
     *
     * Returns the a page of document ids.  If there are any further Pages, the [Paginated]
     * result contains a [Paginated]#nextToken which can be used to fetch the next page.
     *
     * @param datasourceId of the datasource to get the documentIds from
     * @param pageSize of the [Paginated]#payload size
     * @param token of the [Paginated]#nextToken
     *
     * @return [Paginated] collection of document ids
     */
    fun getDocumentIdPage(datasourceId: String, token: String, pageSize: Int = DEFAULT_PAGE_SIZE): Paginated<String>


    companion object {
        const val DEFAULT_PAGE_SIZE = 50
    }
}
