package io.continual.onap.services.publisher;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.zip.GZIPOutputStream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import okhttp3.Credentials;
import okhttp3.MediaType;
import okhttp3.OkHttpClient;
import okhttp3.Request;
import okhttp3.RequestBody;
import okhttp3.Response;

/**
 * A simple message router publisher. This class issues HTTP transactions that execute in the foreground
 * to push messages to the ONAP Message Router service. 
 */
public class OnapMsgRouterPublisher
{
	/**
	 * A builder for the publisher.
	 */
	public static class Builder
	{
		public Builder () {}

		/**
		 * Add a host to set the publisher can use. If you do not provide a protocol,
		 * "http://" is assumed. You may specify "https://" or "http://". If you do not 
		 * provide a port (e.g. "host:3904"), then 3904 is assumed for http, and 3905
		 * for https.  Thus "localhost" is treated as "http://localhost:3904".
		 * 
		 * @param host the host to add to the host set
		 * @return this builder
		 */
		public Builder withHost ( String host )
		{
			fHosts.add ( host );
			return this;
		}

		/**
		 * Add each host to the host list. See withHost ( String host ) for details. 
		 * @param hosts a collection of hosts to add to the host set
		 * @return this builder
		 */
		public Builder withHosts ( Collection<String> hosts )
		{
			for ( String host : hosts )
			{
				withHost ( host );
			}
			return this;
		}
		
		/**
		 * Add each host to the host list. See withHost ( String host ) for details. 
		 * @param hosts a collection of hosts to add to the host set
		 * @return this builder
		 */
		public Builder withHosts ( String[] hosts )
		{
			for ( String host : hosts )
			{
				withHost ( host );
			}
			return this;
		}

		/**
		 * Clear any hosts the builder already knows about.
		 * @return this builder
		 */
		public Builder forgetHosts ()
		{
			fHosts.clear ();
			return this;
		}

		/**
		 * Specify the topic to publish to.
		 * @param topic the topic on which to post messages
		 * @return this builder
		 */
		public Builder onTopic ( String topic )
		{
			fTopic = topic;
			return this;
		}

		/**
		 * Specify the amount of time to wait on a socket connection, read, or write.
		 * @param ms the number of milliseconds to wait for a socket operation (connect/read/write)
		 * @return this builder
		 */
		public Builder waitingAtMost ( long ms )
		{
			fWaitTimeoutMs = ms;
			return this;
		}

		/**
		 * Specify the log to use. If never called, the default logger, named for this class, is used.
		 * @param log the slf4j logger to use for this library. Do not pass null.
		 * @return this builder
		 */
		public Builder logTo ( Logger log )
		{
			fLog = log;
			return this;
		}

		/**
		 * Set HTTP basic auth credentials. If user is null, the auth info is removed from the builder.
		 * @param user the username for basic auth credentials
		 * @param pwd  the password for basic auth credentials
		 * @return this builder
		 */
		public Builder asUser ( String user, String pwd )
		{
			fUser = user;
			fPwd = user == null ? null : pwd;
			return this;
		}

		/**
		 * If no protocol is provided on a host string, default to http://
		 * @return this builder
		 */
		public Builder defaultHttp ()
		{
			return defaultHttps ( false );
		}

		/**
		 * If no protocol is provided on a host string, default to https://
		 * @return this builder
		 */
		public Builder defaultHttps ()
		{
			return defaultHttps ( true );
		}

		/**
		 * If no protocol is provided on a host string, default to https:// if true,
		 * http:// if false.
		 * @param https if true, use https. if false, use http
		 * @return this builder
		 */
		public Builder defaultHttps ( boolean https )
		{
			fDefaultHttps = https;
			return this;
		}

		public Builder withClock ( Clock clock )
		{
			fClock = clock;
			return this;
		}

		/**
		 * Build the publisher given this specification.
		 * @return a new publisher
		 */
		public OnapMsgRouterPublisher build ()
		{
			return new OnapMsgRouterPublisher ( this );
		}
		
		private final LinkedList<String> fHosts = new LinkedList<> ();
		private String fTopic = null;
		private long fWaitTimeoutMs = 30000L;
		private String fUser = null;
		private String fPwd = null;
		private Logger fLog = defaultLog;
		private boolean fDefaultHttps = false;
		private Clock fClock = null;
	}

	/**
	 * A clock for timing
	 */
	public interface Clock
	{
		/**
		 * Get the current time in epoch millis.
		 * @return the current time
		 */
		long nowMs ();
	}

	/**
	 * Get a local test publisher builder to optionally customize further. By default, the publisher
	 * will run against http://localhost:3904, publishing to TEST-TOPIC.
	 * 
	 * @return a builder
	 */
	public static Builder localTest ()
	{
		return new Builder ()
			.withHost ( "localhost" )
			.onTopic ( "TEST-TOPIC" )
			.waitingAtMost ( 30000L )
		;
	}

	/**
	 * Get a new builder
	 * 
	 * @return a builder
	 */
	public static Builder builder ()
	{
		return new Builder ();
	}

	/**
	 * A message includes an event stream name and a payload
	 */
	public static class Message
	{
		public Message ( String eventStreamName, String payload )
		{
			fStreamName = eventStreamName;
			fData = payload;
		}

		public final String fStreamName;
		public final String fData;

		public byte[] getBytesForSend ()
		{
			return fData.toString ().getBytes ( kUtf8 );
		}
	}

	public static class OnapMrResponse
	{
		public OnapMrResponse ( int statusCode, String msg )
		{
			fStatusCode = statusCode;
			fMsg = msg;
		}

		public int getStatusCode () { return fStatusCode; }
		public String getStatusText () { return fMsg; }

		private final int fStatusCode;
		private final String fMsg;
	}

	@Override
	public String toString ()
	{
		return fLabel;
	}

	/**
	 * Send a single message to the MR cluster.
	 * @param msg the message to post
	 * @return the HTTP status code from MR
	 */
	public OnapMrResponse send ( Message msg )
	{
		final LinkedList<Message> msgs = new LinkedList<> ();
		msgs.add ( msg );
		return send ( msgs );
	}

	/**
	 * Send a set of messages to the MR cluster in an all or nothing attempt. Each host in
	 * the host list will be attempted at most once.
	 * 
	 * @param msgList a list of messages
	 * @return the HTTP status code from MR
	 */
	public OnapMrResponse send ( List<Message> msgList )
	{
		// if we have nothing to send, reply ok
		if ( msgList.size () < 1 ) return skAccepted;

		// generate the transaction payload for content-type "application/cambria-zip"
		ByteArrayOutputStream baos;
		try
		{
			baos = new ByteArrayOutputStream ();
			final OutputStream wrapperOs = new GZIPOutputStream ( baos );
			for ( Message m : msgList )
			{
				final byte[] streamBytes = m.fStreamName.getBytes ( kUtf8 );
				final byte[] payloadBytes = m.getBytesForSend ();

				wrapperOs.write ( ( "" + streamBytes.length ).getBytes ( kUtf8 ) );
				wrapperOs.write ( '.' );
				wrapperOs.write ( ( "" + payloadBytes.length ).getBytes ( kUtf8 ) );
				wrapperOs.write ( '.' );
				wrapperOs.write ( streamBytes );
				wrapperOs.write ( payloadBytes );
				wrapperOs.write ( '\n' );
			}
			wrapperOs.close ();
			baos.close ();
		}
		catch ( IOException e )
		{
			// an I/O exception while building the request body isn't likely
			fLog.error ( "Error while building payload for MR publish. Returning 400 Bad Request. " + e.getMessage (), e );
			return new OnapMrResponse ( k400_badRequest, "Unable to build payload." );
		}
		final byte[] msgBody = baos.toByteArray ();

		// send the data to MR, trying each host in order until we have a conclusion...

		final ArrayList<String> hostsLeft = new ArrayList<> ();
		hostsLeft.addAll ( fHosts );

		final long noResponseTimeoutMs = fClock.nowMs () + fWaitTimeoutMs;
		while ( fClock.nowMs () < noResponseTimeoutMs && hostsLeft.size () > 0 )
		{
			final String host = hostsLeft.remove ( 0 );
			final String path = buildPath ( host );

			final RequestBody body = RequestBody.create ( kCambriaZip, msgBody );
			final Request.Builder reqBuilder = new Request.Builder ()
				.url ( path )
				.post ( body )
			;
			if ( fUser != null )
			{
				reqBuilder.addHeader ( "Authorization",
					Credentials.basic ( fUser, fPwd )
				);
			}
			final Request req = reqBuilder.build ();

			fLog.info ( "POST {} ({})", path, fUser == null ? "anonymous" : fUser );

			final long trxStartMs = fClock.nowMs ();
			try ( Response response = fHttpClient.newCall ( req ).execute () )
			{
				final long trxEndMs = fClock.nowMs ();
				final long trxDurationMs = trxEndMs - trxStartMs;

				final int statusCode = response.code ();
				final String statusText = response.message ();
				final String responseBody = response.body ().string ();

				fLog.info ( "    MR reply {} {} ({} ms): {}", statusCode, statusText, trxDurationMs, responseBody );

				if ( isSuccess ( statusCode ) || isClientFailure ( statusCode ) )
				{
					// just relay MR's reply
					return new OnapMrResponse ( statusCode, statusText );
				}
				else if ( isServerFailure ( statusCode ) )
				{
					// that host has a problem, move on
					demote ( host );
				}
			}
			catch ( IOException x )
			{
				final long trxEndMs = fClock.nowMs ();
				final long trxDurationMs = trxEndMs - trxStartMs;

				fLog.warn ( "    MR failure for host [{}]: {} ({} ms)", host, x.getMessage (), trxDurationMs );
				demote ( host );
			}
		}

		// if we're here, we've timed out on all MR hosts and we have to fail the transaction.
		return skSvcUnavailable;
	}

	/**
	 * Build a URL path for Message Router, provided protocol, port, and path as needed
	 * @param host
	 * @return a complete URL path
	 */
	private String buildPath ( String host )
	{
		final StringBuilder sb = new StringBuilder ();

		// add a protocol if one is not provided
		if ( !host.contains ( "://" ) )
		{
			sb.append ( fDefaultHttps ? "https://" : "http://" );
		}

		// add the host
		sb.append ( host );

		// add a port if necessary
		if ( !host.contains ( ":" ) )
		{
			sb.append ( host.startsWith ( "https://" ) ? ":3905" : ":3904" );
		}

		// finally the path parts
		sb.append ( "/events/" );
		sb.append ( urlEncode ( fTopic ) );

		return sb.toString ();
	}

	/**
	 * Move the given host to the back of the list for next time
	 * @param host
	 */
	private void demote ( String host )
	{
		fHosts.remove ( host );
		fHosts.addLast ( host );
	}

	private final LinkedList<String> fHosts;
	private final String fTopic;
	private final long fWaitTimeoutMs;
	private final String fUser;
	private final String fPwd;
	private final boolean fDefaultHttps;
	private final String fLabel;
	private final Clock fClock;

	private final OkHttpClient fHttpClient;

	private final Logger fLog;

	private static final MediaType kCambriaZip = MediaType.get ( "application/cambria-zip" );
	private static final Charset kUtf8 = Charset.forName ( "UTF-8" );

	private static final Logger defaultLog = LoggerFactory.getLogger ( OnapMsgRouterPublisher.class );

	private OnapMsgRouterPublisher ( Builder builder )
	{
		if ( builder.fHosts.size () < 1 ) throw new IllegalArgumentException ( "No hosts provided." );
		if ( builder.fTopic == null || builder.fTopic.length () < 1 ) throw new IllegalArgumentException ( "No topic provided." );

		fHosts = new LinkedList<> ();
		fHosts.addAll ( builder.fHosts );

		fTopic = builder.fTopic;

		fWaitTimeoutMs = builder.fWaitTimeoutMs;
		fDefaultHttps = builder.fDefaultHttps ;

		fUser = builder.fUser;
		fPwd = builder.fPwd;
		if ( fUser != null && fPwd == null ) throw new IllegalArgumentException ( "When a username is provided, a password is required." );
		
		if ( builder.fLog == null ) throw new IllegalArgumentException ( "You must provide a logger." );
		fLog = builder.fLog;

		fClock = builder.fClock == null ? new StdClock () : builder.fClock;

		// setup our HTTP client
		fHttpClient = new OkHttpClient.Builder ()
			.connectTimeout ( 15, TimeUnit.SECONDS )
			.writeTimeout ( 15, TimeUnit.SECONDS )
			.readTimeout ( 30, TimeUnit.SECONDS )
			.build ()
		;

		fLabel = new StringBuilder ()
			.append ( fTopic )
			.append ( " on " )
			.append ( fHosts.toString () )
			.append ( " as " )
			.append ( fUser == null ? "anonymous" : fUser )
			.toString ()
		;
	}

	private static final int k200_ok = 200;
	private static final int k202_accepted = 202;
	private static final int k300_multipleChoices = 300;
	private static final int k400_badRequest = 400;
	private static final int k500_internalServerError = 500;
	private static final int k503_serviceUnavailable = 503;

	private static final OnapMrResponse skAccepted = new OnapMrResponse ( k202_accepted, "Accepted." );
	private static final OnapMrResponse skSvcUnavailable = new OnapMrResponse ( k503_serviceUnavailable, "No Message Router server could acknowledge the request." );

	private static boolean isSuccess ( int code )
	{
		return code >= k200_ok && code < k300_multipleChoices;
	}

	private static boolean isClientFailure ( int code )
	{
		return code >= k400_badRequest && code < k500_internalServerError;
	}

	private static boolean isServerFailure ( int code )
	{
		return code >= k500_internalServerError;
	}

	private static String urlEncode ( String s )
	{
		if ( s == null ) return null;
		try
		{
			return URLEncoder.encode ( s, "UTF-8" );
		}
		catch ( UnsupportedEncodingException e )
		{
			throw new RuntimeException ( e );
		}
	}

	private static class StdClock implements Clock
	{
		@Override
		public long nowMs () { return System.currentTimeMillis (); }
	}
}
