/*
 * Copyright © 2021 Cask Data, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package io.cdap.wrangler.parser;

import io.cdap.wrangler.api.CompileException;
import io.cdap.wrangler.api.CompileStatus;
import io.cdap.wrangler.api.Compiler;
import io.cdap.wrangler.api.DirectiveContext;
import io.cdap.wrangler.api.DirectiveParseException;
import io.cdap.wrangler.api.TokenGroup;
import io.cdap.wrangler.api.parser.DirectiveName;
import io.cdap.wrangler.api.parser.SyntaxError;

import java.util.Iterator;

/**
 * This class provides logic to walk a grammar tree by parsing a wrangler recipe.
 */
public class GrammarWalker {

  private final Compiler compiler;
  private final DirectiveContext context;

  /**
   * A visitor for the the recipe grammar.
   * @param <E> type of the exception thrown by the {@link #visit(String, TokenGroup)} method
   */
  public interface Visitor<E extends Exception> {
    void visit(String command, TokenGroup tokenGroup) throws E;
  }

  public GrammarWalker(Compiler compiler, DirectiveContext context) {
    this.compiler = compiler;
    this.context = context;
  }

  /**
   * Walks the grammar tree generated by the recipe.
   *
   * @param recipe the recipe to parse
   * @param visitor the visitor for walking the grammar tree
   * @param <E> type of exception that can be thrown by the visitor
   * @throws CompileException if failed to compile the recipe
   * @throws DirectiveParseException if a directive in the recipe is invalid
   * @throws E if the visitor throws an exception
   */
  public <E extends Exception> void walk(String recipe,
                                         Visitor<E> visitor) throws CompileException, DirectiveParseException, E {
    CompileStatus status = compiler.compile(recipe);
    if (!status.isSuccess()) {
      Iterator<SyntaxError> errors = status.getErrors();
      String prefix = "Encountered syntax error, please ensure the directive is valid:\n";
      throw new DirectiveParseException(prefix + errors.next().getMessage(), errors);
    }

    Iterator<TokenGroup> tokenGroups = status.getSymbols().iterator();
    while (tokenGroups.hasNext()) {
      TokenGroup tokenGroup = tokenGroups.next();
      if (tokenGroup == null) {
        continue;
      }
      String command = ((DirectiveName) tokenGroup.get(0)).value();
      String root = command;
      if (context.hasAlias(root)) {
        root = context.getAlias(command);
      }

      // Checks if the directive has been excluded from being used.
      if (!root.equals(command) && context.isExcluded(command)) {
        throw new DirectiveParseException(
          command, String.format("Aliased directive '%s' has been configured as restricted directive and "
                                   + "is hence unavailable. Please contact your administrator", command)
        );
      }

      if (context.isExcluded(root)) {
        throw new DirectiveParseException(
          command, String.format("Directive '%s' has been configured as restricted directive and is hence " +
                                   "unavailable. Please contact your administrator", command));
      }

      visitor.visit(root, tokenGroup);;
    }
  }
}
